/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import java.util.Enumeration;
import java.util.Vector;

import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.ActorAddress;
import org.coos.javaframe.ActorFrameException;
import org.coos.javaframe.ActorSpec;
import org.coos.javaframe.ApplicationSpec;
import org.coos.javaframe.PartSpec;
import org.coos.javaframe.RoleCS;
import org.coos.javaframe.StateMachine;
import org.coos.javaframe.TraceConstants;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;

/**
 * Specifies the actor specific persistent data.
 * 
 * @author Geir Melby, Tellu AS
 * @author Knut Eilif Husa, Tellu AS
 */
public class ActorSM extends StateMachine implements AFConstants {

	protected static String ROUTER_UPDATE_INTERVAL_ID = "ROUTER_UPDATE_INTERVAL";

	public static final String ACTOR_REPORT_TIMER = "ACTOR_REPORT_TIMER";
	public static long ACTOR_REPORT_TIMEOUT = 10 * 1000; // each 10 seconds
	// protected String ActorDesc;

	protected static final String li = "<li>"; // todo constants; uppercase
	protected static final String lie = "</li>";
	protected static final String br = "<br>";
	protected static final String h3 = "<h3>";
	protected static final String h3e = "</h3>";
	protected static final String bold = "<b>";
	protected static final String boldend = "</b>";

	public Vector resumedActors;
	public Vector suspendedActors;
	public ActorAddress suspendActor;
	public ActorAddress resumeActor;
	public ActorSpec newActorSpec;
	public Vector removedParts;
	public Vector changedParts;
	public ActorAddress roleUpdateSender;

	public int startLevel = AFConstants.LEVEL_0;

	/**
	 * A constructor
	 */
	public ActorSM() {
		setBehaviorClass(new ActorCS("ActorCS"));
	}

	public Vector clonePorts(Vector ports) {
		Vector res = new Vector();

		if (ports != null) {
			Enumeration it = ports.elements();

			while (it.hasMoreElements()) {
				ActorPortSpec ps = (ActorPortSpec) it.nextElement();
				res.addElement(ps.clone());
			}
		}

		return res;
	}

	/**
	 * A method that is called by the State Machine class when the State Machine
	 * is created. It can be used to initilize varaibles, read xml files etc
	 * that are common for instances for this state machines.
	 */
	protected void initStateMachine() {
		if (myCompositeState == null) {
			myCompositeState = new ActorCS("ActorCS");
		}
		insertActorRoles(actorSpec.getRoleDesc());
	}

	/**
	 * todo check if this shoud be deleted
	 */
	protected void updateCompositeStructure() {
		readActorDescription();

		Vector partSpecs = actorSpec.getPartDesc();

		// todo traveser childrenroles sjekk om de skal fjernes / gi melding om
		// det som er galt
		for (int i = 0; i < partSpecs.size(); i++) {
			PartSpec o = (PartSpec) partSpecs.elementAt(i);

			if (!(context.sizeOfChildrenRoles(o.getRoleType()) <= o.getHigh())) {
				if (scheduler.isTraceOn()) {
					trace.traceOut(TraceConstants.tlWarn, trace.getTraceHeader() + "INNER PART SET " + o.getRoleType()
							+ " exceeds redefined cardinality!!");
				}
			}
		}

		Vector childrenRoles = context.getChildrenRoles();
		boolean defined = false;

		for (int i = 0; i < childrenRoles.size(); i++) {
			String actualRoleType = ((ActorAddress) childrenRoles.elementAt(i)).getActorType();
			ActorAddress aa = (ActorAddress) childrenRoles.elementAt(i);
			defined = false;

			for (int j = 0; j < partSpecs.size(); j++) {
				PartSpec o = (PartSpec) partSpecs.elementAt(j);
				String specRoleType = o.getRoleType();

				if (actualRoleType.equals(specRoleType)) {
					defined = true;

					// RoleUpdateMsg rum = new RoleUpdateMsg(); //update
					// innerparts with new port definition
					// Vector v = ((ActorPartSpec)
					// partSpecs.elementAt(j)).getPortNames();
					Vector names = getApplicationSpec().getActorSpec(specRoleType).getPortNames();
					AFPropertyMsg msg = new AFPropertyMsg(ROLE_UPDATE_MSG, true);
					msg.setProperty("ports", names);
					sendMessage(msg, aa);

					break;
				}
			}

			if (!defined) {
				if (scheduler.isTraceOn()) {
					trace.traceOut(TraceConstants.tlError, trace.getTraceHeader() + "PART: " + actualRoleType
							+ " is not defined!!");
				}

				// todo delete parts??
			}
		}

		if (createParts()) {
			if (scheduler.isTraceOn()) {
				trace.traceTask("New parts defined");
			}
		}
	}

	public boolean createParts() {
		return createParts(actorSpec, AFConstants.LEVEL_4);
	}

	public boolean createParts(ActorSpec as) {
		return createParts(as, AFConstants.LEVEL_4);
	}

	/**
	 * Creates the inner parts (actors) that shall exists at creation time of
	 * the enclosing actor. Not used by the application developers. Create the
	 * part only if it
	 */
	public boolean createParts(ActorSpec as, int level) {
		boolean childrenCreated = false;
		String s = "INNER PARTS: ";
		Vector partSpecs = as.getPartDesc();
		if (partSpecs == null) {
			throw new ActorFrameException("No part description for <" + as.getActorType() + ">");
		}
		while (!childrenCreated && level <= AFConstants.LEVEL_4) {

			for (int i = 0; i < partSpecs.size(); i++) {
				PartSpec ps = (PartSpec) partSpecs.elementAt(i);

				if (ps.getLevel() != level) {
					continue;
				}

				// if the bind property is set (Instance id for the actor that
				// implements this part),
				// an instance of the actor shall not be created
				if (ps.getBind() != null) {
					if (scheduler.isTraceOn())
						trace.traceOut(TraceConstants.tlInfo, " Scheduler.createParts:" + " Actor spec: "
								+ ps.getRoleType() + " is bind to: " + ps.getBind());
					continue;
				}

				s = s + ", " + ps.toString();

				ActorAddress aa;
				int noOfExistingRoles = context.getChildrenRoles(ps.getRoleType()).size();

				if ((ps.getLow() - noOfExistingRoles) > 0) {
					for (int j = noOfExistingRoles; j < ps.getLow(); j++) {
						Vector names = getApplicationSpec().getActorSpec(ps.getRoleType()).getPortNames();
						AFPropertyMsg rcm = createRoleCreateMsg(ps, names);

						// check also if it exists a role name for that
						// particular role instance
						if ((ps.getRoleNames() == null) || (ps.getRoleNames().length <= j)) {
							String instanceName = getContextString() // +
									// o.getType()
									+ ps.getSetId() // + o.getSetNo()
									+ String.valueOf(j + noOfExistingRoles);

							aa = new ActorAddress(instanceName, ps.getRoleType());
						} else {
							aa = new ActorAddress(getContextString() + ps.getRoleNames()[j], ps.getRoleType());
						}

						// set remote actor, null if it not exists, read from
						// the actor descriptors
						// Send only the creation msg to actors that
						if (ps.getActorDomain() != null) {
							rcm.setProperty(ROLE_CREATE_MSG_TARGET_ACTOR, aa);
							// add the actor only if it is there again
							context.getCreationOfChildren().addElement(aa);
							this.sendMessage(rcm, ps.getActorDomain()); // send
							// the
							// message
						} else {
							// add the actor only if it is not there
							context.getCreationOfChildren().addElement(aa);
							this.sendMessage(rcm, aa); // send the message
						}

						childrenCreated = true;
					}
				}

			}
			if (!childrenCreated) {
				level++; // try out next start level
			}

		}
		if (scheduler.isTraceOn()) {
			trace.traceOut(TraceConstants.tlInfo, trace.getTraceHeader() + s);
		}
		startLevel = level > AFConstants.LEVEL_4 ? LEVEL_4 : level;
		return childrenCreated;
	}

	/**
	 * Create parts listed as strings of format <ericsson/arts@CASDemo> in
	 * vector v. The actor id and actor type has to be read from this string.
	 * 
	 * @param v
	 *            contains strings of format <ericsson/arts@CASDemo> which is
	 *            the key of ActorAddresses of actors that has not responed on
	 *            RoleCreate message
	 * @return true if parts has been created
	 */
	public boolean createParts(Vector v) {
		boolean childrenCreated = false;
		Vector partSpecs = actorSpec.getPartDesc();

		for (int i = 0; i < partSpecs.size(); i++) {
			PartSpec ps = (PartSpec) partSpecs.elementAt(i);
			// if the bind property is set (Instance id for the actor that
			// implements this part),
			// an instance of the actor shall not be created
			if (ps.getBind() != null) {
				if (scheduler.isTraceOn())
					trace.traceOut(TraceConstants.tlInfo, " Scheduler.createParts:" + " Actor spec: "
							+ ps.getRoleType() + " is bind to: " + ps.getBind());
				continue;
			}
			// check if v contains this actor type
			for (int j = 0; j < v.size(); j++) {
				// Read the key string, which has to be converted to
				// ActorAddress
				Vector ports = getApplicationSpec().getActorSpec(ps.getRoleType()).getPortNames();
				AFPropertyMsg rcm = createRoleCreateMsg(ps, ports);
				ActorAddress aa = (ActorAddress) v.elementAt(j);
				String s = aa.key();
				String actorId;

				if (s.endsWith(ps.getRoleType())) {
					// the key ends with actortype, calculate the id string
					actorId = s.substring(0, s.indexOf("@"));
					aa = new ActorAddress(actorId, ps.getRoleType());

					if (ps.getActorDomain() != null) {
						String id = getContextString() + ps.getRoleNames()[j];
						rcm.setProperty(ROLE_CREATE_MSG_TARGET_ACTOR, new ActorAddress(id, ps.getRoleType()));
						this.sendMessage(rcm, ps.getActorDomain()); // send the
						// message
					} else {
						this.sendMessage(rcm, aa); // send the message
					}

					childrenCreated = true;
				}
			}
		}

		return childrenCreated;
	}

	/**
	 * Create the Port instances belonging to an Actor and add it to the
	 * Hashtable this.port. If the port already exists the port is not added to
	 * the port list
	 * 
	 * @param connectors
	 */
	public boolean createPorts(Vector connectors) {
		this.ports.clear();
		boolean createdPorts = false;
		// create the default in port for this Actor
		ActorPortSpec dPS = new ActorPortSpec();
		dPS.setPortName("defaultInPort");
		dPS.setIsBehavior(true);
		this.ports.put("defaultInPort", new Port(dPS, getMyActorAddress()));
		// Create all port instances with their configuration
		if (actorSpec.getPortDesc() == null) {
			return false;
		}

		Enumeration enumer = actorSpec.getPortDesc().elements();

		while (enumer.hasMoreElements()) {
			ActorPortSpec actorPortSpec = (ActorPortSpec) enumer.nextElement();
			if (!ports.containsKey(actorPortSpec.getPortName())) {
				Port port = new Port(actorPortSpec, getMyActorAddress());
				// Add it to the port list [from StateMachine]
				this.ports.put(port.name, port);
				createdPorts = true;
			}

		}
		return createdPorts;
	}

	/**
	 * Create connectors associated with the ports of childrens actors.
	 */
	public void createConnectors() {
		if (actorSpec.getConnectorDesc() == null) {
			return;
		}

		Vector parts = actorSpec.getPartDesc();

		for (int i = 0; i < parts.size(); i++) {
			PartSpec spec = (PartSpec) parts.elementAt(i);
			createConnectors(spec);
		}
		createConnectorsToItself();
	}

	/**
	 * Create connector based on an ActorAddress.
	 * 
	 * @param aa
	 *            is the actor address of the connector endpoint
	 */
	public void createConnectors(ActorAddress aa) {
		PartSpec ps = findRoleSpec(aa.getActorType(), actorSpec.getPartDesc());

		if (ps == null) {
			return;
		}

		Enumeration names = getApplicationSpec().getActorSpec(aa.getActorType()).getPortNames().elements();

		if (aa.getActorPort() != null) {
			Vector connectors = rewriteContextualConnectors(actorSpec.getConnectorDesc(aa.getActorType(), aa
					.getActorPort()));
			ActorAddress receiverPort = (ActorAddress) aa.clone();
			receiverPort.setActorPort(aa.getActorPort());

			AFPropertyMsg msg = new AFPropertyMsg(PORT_CREATE_MSG, true);
			msg.setFrameworkMsg(true);
			msg.setProperty(PORT_CREATE_MSG_CONNECTORS, connectors);
			msg.setSenderRole(getMyActorAddress());
			sendMessage(msg, receiverPort);

			return;
		}

		while (names.hasMoreElements()) {
			String key = (String) names.nextElement();
			Vector connectors = rewriteContextualConnectors(actorSpec.getConnectorDesc(aa.getActorType(), key));
			ActorAddress receiverPort = (ActorAddress) aa.clone();
			receiverPort.setActorPort(key);

			AFPropertyMsg msg = new AFPropertyMsg(PORT_CREATE_MSG, true);
			msg.setFrameworkMsg(true);
			msg.setProperty(PORT_CREATE_MSG_CONNECTORS, connectors);
			msg.setSenderRole(getMyActorAddress());
			sendMessage(msg, receiverPort);
		}
	}

	/**
	 * Create Port connectors for all instances with a given PartSpec.
	 * 
	 * @param ps
	 *            PartSpec to create for.
	 * @return boolean if successful
	 */
	public boolean createConnectors(PartSpec ps) {
		boolean created = false;
		Vector children = context.getChildrenRoles(ps.getRoleType());
		Enumeration it = getApplicationSpec().getActorSpec(ps.getRoleType()).getPortNames().elements();

		while (it.hasMoreElements()) {
			String portName = (String) it.nextElement();
			Vector connectors = actorSpec.getConnectorDesc(ps.getRoleType(), portName);
			Vector connectorsAddresses = rewriteContextualConnectors(connectors);

			// create the receiver ActorAddress
			Enumeration cit = children.elements();
			Vector portAA = new Vector();

			while (cit.hasMoreElements()) {
				ActorAddress recv = (ActorAddress) cit.nextElement();
				String[] roleNames = ps.getRoleNames();

				if (roleNames != null) {
					for (int i = 0; i < roleNames.length; i++) {
						String roleName = roleNames[i];

						if (recv.getActorID().endsWith(roleName)) {
							recv = (ActorAddress) recv.clone();
							recv.setActorPort(portName);
							portAA.addElement(recv);
							created = true;

							break;
						}
					}
				} else {
					if (recv.getActorID().endsWith("Set" + ps.getSetNo())) {
						recv = (ActorAddress) recv.clone();
						recv.setActorPort(portName);
						portAA.addElement(recv);
						created = true;
					}
				}
			}

			AFPropertyMsg pcm = new AFPropertyMsg(PORT_CREATE_MSG, true);
			pcm.setProperty(PORT_CREATE_MSG_CONNECTORS, connectorsAddresses);
			pcm.setSenderRole(getMyActorAddress());
			// pcm.setFrameworkMsg(true);
			sendMessage(pcm, portAA);
		}

		return created;
	}

	/**
	 * Sends the connectors to port of this actor instance. A PortCreateMsg is
	 * send to all ports defined for this actor even if there are no connectorts
	 * to that port
	 */
	public void createConnectorsToItself() {
		Enumeration names = getActorSpec().getPortNames().elements();
		// the connectors that belongs the

		while (names.hasMoreElements()) {
			String portName = (String) names.nextElement();
			Vector connectors = actorSpec.getConnectorDesc(getMyActorAddress().getActorType(), portName);
			Vector v = rewriteContextualConnectors(connectors);

			AFPropertyMsg pcm = new AFPropertyMsg(PORT_CREATE_MSG, true);
			pcm.setProperty(PORT_CREATE_MSG_CONNECTORS, v);
			pcm.setSenderRole(getMyActorAddress());
			pcm.setFrameworkMsg(true);
			sendMessage(pcm, portName);
		}
	}

	/**
	 * is the method called by the framework each time a new instance of this
	 * actor type is created. Normaly will this method be redefined in sub type,
	 * but it has to be called from the sub type
	 */
	protected void initInstance() {
	}

	/**
	 * Set a reference to a Port instance.
	 * 
	 * @param port
	 *            Port instance to add to the hash.
	 */
	public void addPort(Port port) {
		this.ports.put(port.name, port);
	}

	/**
	 * Remove the port instance.
	 * 
	 * @param port
	 *            Port instance to add to the hash.
	 */
	public void removePort(Port port) {
		if (ports.containsKey(port.name)) {
			this.ports.remove(port);
		}
	}

	/**
	 * Get a Port instance by name.
	 * 
	 * @param portName
	 *            Name of the port to find.
	 * @return Port instance, null if not found.
	 */
	public Port getPort(String portName) {
		if (this.ports.containsKey(portName)) {
			return (Port) this.ports.get(portName);
		}
		return null;
	}

	/**
	 * The ActorMsg is destined for a Port instance. This method ensures only
	 * the nec. StateData is resurrected upon reception of a message.
	 * 
	 * @param msg
	 *            ActorMsg to process
	 */
	protected void processPortMessage(ActorMsg msg) {
		// Get the state data for this port.
		myActorId = msg.getReceiverRole().getActorID();
		myActorType = msg.getReceiverRole().getActorType();

		Port port = getPort(msg.getReceiverRole().getActorPort());

		if (port != null) {
			trace.traceInit(this);
			trace.setActor(msg.getReceiverRole());
			trace.setInputSignal(msg);
			port.exec(msg, this);
		} else {
			if (scheduler.isTraceOn()) {
				trace.traceError("Port: " + msg.getReceiverRole().getActorPort()
						+ " set in ActorMsg, but the Port was not found for the ActorAddress " + msg.getReceiverRole());
			}
		}

		if (scheduler.isTraceOn() && getTraceLevel() <= TraceConstants.tlDebug) {
			trace.traceOut(TraceConstants.tlDebug, TraceConstants.tcFramework, trace.toString()); // portString());
		}

		// Store StateMachine specific data only
	}

	/**
	 * Process the incoming message. If the actor message RoleCreateMsg is
	 * received, an new actor instance is created before the exec method is
	 * called which will treat the message.
	 * 
	 * @param msg
	 *            is the actor message.
	 * @see ActorMsg
	 */
	public void processMessage(ActorMsg msg) {

		if (msg.getReceiverRole().hasActorPort()) {
			// Message destined for a Port defined here, try
			// to find it and execute the Port mechanism.
			processPortMessage(msg);
			return;
		}
		super.processMessage(msg);
	}

	/**
	 * Sends a message to another actor via a port name. The port is set up at
	 * initiation of actor, normaly using actor deployment descriptors.
	 * 
	 * @param am
	 *            ActorMsg to send.
	 * @param portName
	 *            the port id. This has to be the same name as defined in the
	 *            descriptor file.
	 */
	public void sendMessage(ActorMsg am, String portName) {
		Port port = getPort(portName);

		if (port == null) {
			if (scheduler.isTraceOn()) {
				trace.traceError("Illegal port name: " + portName + " Message: " + am.toString());
			}

			return;
		}

		am.setSenderRole(getMyActorAddress());
		if (isTesting()) {
			if (scheduler.isTraceOn()) {
				trace.traceOutput(am);
			}
		}
		port.exec(am, this);
	}

	/**
	 * Search for actors(parts) that belongs to this statemachine
	 * 
	 * @param actorId
	 *            is the actor id, null = any
	 * @param actorType
	 *            is the actor type, null = any
	 * @return a vector of actor addresses of part instances
	 */

	// todo add a search based on pattern GM 24.1.2007
	protected Vector searchForActors(String actorId, String actorType) {
		if (actorId == null && actorType == null) {
			return context.getChildrenRoles();
		}

		Vector res = new Vector();
		Vector roles = context.getChildrenRoles();

		for (int i = 0; i < roles.size(); i++) {
			ActorAddress actorAddress = (ActorAddress) roles.elementAt(i);
			boolean added = false;

			if (actorId != null) {
				if (actorAddress.getActorID().equals(actorId)) {
					res.addElement(actorAddress);
					added = true;
				}
			}
			if (actorType != null && !added) { // Don't add twice
				if (actorAddress.getActorType().equals(actorType))
					res.addElement(actorAddress);
			}
		}

		return res;
	}

	protected boolean isCreateMsg(ActorMsg msg) {
		return msg.equals("RoleCreateMsg") || msg.equals(AFConstants.ROLE_PLAY_MSG);
	}

	public ApplicationSpec getApplicationSpec() {
		return getScheduler().getSchedulerData().getApplicationSpec();
	}

	/**
	 * Creates a full id for an actor role
	 * 
	 * @param roleId
	 *            is the role id
	 * @return the id
	 */
	protected String makeRoleName(String roleId) {
		String s = getContextString();

		return s.substring(0, s.length() - 1) + "." + roleId;
	}

	/**
	 * Creates a role based on a class name
	 * 
	 * @param id
	 *            is the new instance id
	 * @param className
	 *            is the name of class name to be created
	 * @return an actor address of the role if success, otherwise null
	 */
	public ActorAddress createRole(String id, String className) {
		RoleCS rcs = null;
		ActorAddress actorAddress = null;
		try {
			rcs = (RoleCS) scheduler.getClassLoader().loadClass(className).newInstance();
			String actorId = myActorId + "." + id;
			actorAddress = new ActorAddress(actorId, myActorType);
			addStateMachine(id, rcs);
			if (scheduler.isTraceOn()) {
				trace.traceTask("Role: " + actorId + " added");
			}
		} catch (IllegalAccessException e) {
			if (scheduler.isTraceOn())
				trace.traceError("IllegalAccessException in method insertActorRoles()");
		} catch (InstantiationException e) {
			if (scheduler.isTraceOn())
				trace.traceError("InstantiationException in method insertActorRoles()");
		} catch (ClassNotFoundException e) {
			if (scheduler.isTraceOn())
				trace.traceError("ClassNotFoundException in method insertActorRoles()");
			return null;
		}
		if (actorAddress != null) {
			sendMessage(START_PLAYING_MSG, actorAddress);
		}
		return actorAddress;
	}

	/**
	 * Creates roles for an Actor. The roles are added to the state machine
	 * 
	 * @param roleSpecs
	 *            is the specification for the roles to be created
	 * @return true if the role inserted
	 */
	protected boolean insertActorRoles(Vector roleSpecs) {
		boolean res = true;
		for (int i = 0; i < roleSpecs.size(); i++) {
			RoleSpec rs = (RoleSpec) roleSpecs.elementAt(i);
			if (rs.isPersistent() && rs.getInstance() != null) {
				res = res && createRole(rs.getInstance(), rs.getRoleClass()) != null;
			}
		}
		return res;
	}

	/**
	 * If the role spec is changed, the roles is updated. The old composite
	 * state machines are removed and new ones are create
	 * 
	 * @param
	 */

	/*
	 * protected void updateActorRoles(Vector roleSpecs) { Hashtable tmpRoles =
	 * (Hashtable) getStateMachines().clone(); //keep the original vaules, for
	 * later find out which shall be deleted
	 * 
	 * try { for (int i = 0; i < roleSpecs.size(); i++) { RoleSpec rs =
	 * (RoleSpec) roleSpecs.elementAt(i); String roleIdWithContext =
	 * makeRoleName(rs.getInstance()); // role id with context string
	 * "hia/geir.ping" if (getStateMachines().containsKey(roleIdWithContext)) {
	 * // id exist, check if of same type RoleCS myRole = (RoleCS)
	 * getStateMachines().get(roleIdWithContext); if
	 * (rs.getType().equalsIgnoreCase(myRole.getMyRoleType())) { // of same type
	 * and id, not checking the class name, could be validateLicense
	 * tmpRoles.remove(roleIdWithContext); // remove the object from the tmp
	 * variable } else { // id is the same, but not the type,, if
	 * (scheduler.isTraceOn()) trace.traceOut(TraceConstants.tcFramework,
	 * TraceConstants.tlError, "RoleId exist already for role spec: " +
	 * rs.toString()); tmpRoles.remove(roleIdWithContext); // remove the object
	 * from the tmp variable } } else { // add the new state machine Class role
	 * = Class.forName(rs.getRoleClass()); CompositeState rcs = (CompositeState)
	 * role.newInstance(); addStateMachine(rs.getInstance(), rcs); } }
	 * Enumeration en = tmpRoles.keys(); while (en.hasMoreElements()) { String s
	 * = (String) en.nextElement(); //s = makeRoleName(rs.getInstance())
	 * getStateMachines().remove(s); }
	 * 
	 * } catch (IllegalAccessException e) { if (scheduler.isTraceOn())
	 * trace.traceOut(TraceConstants.tcFramework, TraceConstants.tlError,
	 * "IllegalAccessException in method updateActorRoles()"); } catch
	 * (InstantiationException e) { if (scheduler.isTraceOn())
	 * trace.traceOut(TraceConstants.tcFramework, TraceConstants.tlError,
	 * "InstantiationException in method updateActorRoles()"); } catch
	 * (ClassNotFoundException e) { if (scheduler.isTraceOn())
	 * trace.traceOut(TraceConstants.tcFramework, TraceConstants.tlError,
	 * "ClassNotFoundException in method updateActorRoles()"); } }
	 */

	/*
	 * public ActorAddress addPart(String actorType, String actorId) { //int
	 * status = 0; ActorAddress childAddress = null; Vector v =
	 * context.getChildrenRoles(actorType); // v > 0 -> children roles exists of
	 * right type PartSpec p = findRoleSpec(actorType, getPartSpecs());
	 * 
	 * if (p != null) { // Part specification exists
	 * 
	 * AFPropertyMsg msg = new AFPropertyMsg(ROLE_CREATE_MSG, true);
	 * msg.setProperty(ROLE_CREATE_MSG_PORTS,
	 * getApplicationSpec().getActorSpec(p.getRoleType()).getPortNames());
	 * msg.setProperty(ROLE_CREATE_MSG_TARGET_ACTOR, p.getActorDomain());
	 * 
	 * if ((actorId != null) && !actorId.equals("")) { // ActorId is specified
	 * and children roles may exists ActorAddress aa =
	 * context.getChildrenRole(actorId);
	 * 
	 * if (aa != null) { // The requested role actor exists //sendMessage(msg,
	 * aa); // Send RolePlay message to the existing actor childAddress = aa; }
	 * else if (checkMaxLimit(p)) { // the requested actor does not exists and
	 * new actor roles are allowed to be created ActorAddress newaa = new
	 * ActorAddress(getContextString() + actorId, actorType);
	 * newaa.setActorDomain
	 * (getScheduler().getSchedulerData().getActorDomainName());
	 * sendMessage(msg, newaa); context.addChildrenRole(newaa); childAddress =
	 * newaa; } } else if ((actorId == null) && v.isEmpty()) { if
	 * (checkMaxLimit(p)) { // new actor roles are allowed to be created, assign
	 * a new unique actorid ActorAddress aa = new
	 * ActorAddress(getContextString() + actorType + UNIQ_ID++ + "Set" +
	 * p.getSetNo(), actorType); sendMessage(msg, aa);
	 * aa.setActorDomain(getScheduler
	 * ().getSchedulerData().getActorDomainName()); context.addChildrenRole(aa);
	 * childAddress = aa; } } } return childAddress; }
	 */
}
