/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.javaframe.ActorAddress;
import org.coos.javaframe.TraceConstants;
import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.AFSerializer;
import org.coos.util.serialize.HashtableHelper;
import org.coos.util.serialize.StringHelper;
import org.coos.util.serialize.VectorHelper;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * A report that contains information about an actors and all its inner actors.
 * 
 * @author Geir Melby, Tellu AS
 */
public class ActorReport implements AFSerializer {
	public String RequestType;
	public ActorAddress myId;
	public ActorAddress myParrentId;
	private Hashtable myProperties;
	public String myState;
	public String myStateData;
	public Vector myChildren = new Vector(); // elements are ActorReport
	public Vector myRequestedRoles; // elements are ActorAddress
	public Vector yourRequestedRoles; // elements are ActorAddress
	public boolean active;
	public int traceLevel; // trace levels that shall be traced for this actor
	public Vector myRoles; // elements of String
	public String status;
	private String result;
	private String repstatus = "Status of the report request: ";
	private int tabLength;
	public Hashtable ports;

	public ActorReport() {
	}

	public Vector getChildrenOfType(String type) { // <ActorReport>
		if (myChildren == null)
			return new Vector();

		Vector res = new Vector();

		for (int x = 0; x < myChildren.size(); x++) {
			ActorReport ar = (ActorReport) myChildren.elementAt(x);
			if (ar.myId.getActorType().equals(type))
				res.addElement(ar);
		}
		return res;
	}

	public String printActorReport() {
		result = "Status Report of type: " + this.RequestType + "\n\n";
		tabLength = -1; // Intial value
		getActorReport(this);

		if (repstatus != null) {
			result = result + status;
		}

		return result;
	}

	public String getActorReport(ActorReport actorReport) {
		ActorAddress aa;
		Enumeration i;

		// summarize the status of each error
		if (actorReport.status != null) {
			repstatus = repstatus + actorReport.status;
		}

		result = result + "ACTOR: " + actorReport.myId + " STATE: " + actorReport.myState + "  STATE DATA: "
				+ actorReport.myStateData;
		result += (" TRACE LEVEL: " + TraceConstants.getTraceLevel(actorReport.traceLevel));
		// todo try to avoid this dependency to agentframe package
		result = result + " ROLES: ";

		if (actorReport.myRoles != null) {
			i = actorReport.myRoles.elements();

			while (i.hasMoreElements()) {
				result = result + i.nextElement();
			}
		}

		result = result + "  REQUESTED ROLES: ";

		if (actorReport.myRequestedRoles != null) {
			i = actorReport.myRequestedRoles.elements();

			while (i.hasMoreElements()) {
				aa = (ActorAddress) i.nextElement();
				result = result + aa.toString() + ", ";
			}
		}

		result = result + "  REQUESTOR ROLES: ";

		if (actorReport.yourRequestedRoles != null) {
			i = actorReport.yourRequestedRoles.elements();

			while (i.hasMoreElements()) {
				aa = (ActorAddress) i.nextElement();
				result = result + aa.toString() + ", ";
			}
		}

		result = result + "\n";

		if (actorReport.myChildren != null) {
			i = actorReport.myChildren.elements();
			tabLength++;

			while (i.hasMoreElements()) {
				for (int j = tabLength; j >= 0; j--) {
					result = result + "-->";
				}

				getActorReport((ActorReport) i.nextElement());
			}
		} else {
			result = result + " NO OF CHILDREN: ";
		}

		tabLength--;

		return "not supported";
	}

	public String printXMLActorReport() {
		return "<xml>" + getXMLActorReport(this) + "</xml>";
	}

	public String getXMLActorReport(ActorReport actorReport) {
		ActorAddress aa;
		Enumeration i;

		// summarize the status of each error
		if (actorReport.status != null) {
			repstatus = repstatus + actorReport.status;
		}

		result += "<actor>" + actorReport.myId.keyWithOutPort() + "<content><state>" + actorReport.myState
				+ "</state><statedata>" + actorReport.myStateData + "</statedata>";
		result += ("<tracelevel>" + TraceConstants.getTraceLevel(actorReport.traceLevel) + "</tracelevel>");
		// todo try to avoid this dependency to agentframe package
		result += "<actoraddress>";
		result += "<actorport>" + actorReport.myId.getActorPort() + "</actorport>";
		result += "<actorip>" + actorReport.myId.getActorIP() + "</actorip>";

		result += "</actoraddress>";

		result = result + "<ports>";

		if (actorReport.ports != null) {
			i = actorReport.ports.keys();
			while (i.hasMoreElements()) {
				String portname = (String) i.nextElement();
				// Port port = (Port)actorReport.ports.get(portname);
				result = result + portname + "\n";
			}
		}

		result = result + "</ports>";

		result = result + "<roles>";

		if (actorReport.myRoles != null) {
			i = actorReport.myRoles.elements();

			while (i.hasMoreElements()) {
				result = result + i.nextElement();
			}
		}

		result = result + "</roles><requestedRoles>";

		if (actorReport.myRequestedRoles != null) {
			i = actorReport.myRequestedRoles.elements();

			while (i.hasMoreElements()) {
				aa = (ActorAddress) i.nextElement();
				result = result + aa.toString() + ", ";
			}
		}

		result = result + "</requestedRoles><requestorRoles>";

		if (actorReport.yourRequestedRoles != null) {
			i = actorReport.yourRequestedRoles.elements();

			while (i.hasMoreElements()) {
				aa = (ActorAddress) i.nextElement();
				result = result + aa.toString() + ", ";
			}
		}

		result = result + "</requestorRoles></content>\n";

		if (actorReport.myChildren != null) {
			i = actorReport.myChildren.elements();
			tabLength++;

			while (i.hasMoreElements()) {
				getXMLActorReport((ActorReport) i.nextElement());
			}
		} else {
			result = result + " NO OF CHILDREN: ";
		}

		result = result + "</actor>\n";

		return result;
	}

	public ActorReport(DataInputStream din, AFClassLoader cl) throws IOException {
		RequestType = StringHelper.resurrect(din);
		myId = new ActorAddress(din);
		myParrentId = new ActorAddress(din);
		myState = StringHelper.resurrect(din);
		myStateData = StringHelper.resurrect(din);
		myChildren = VectorHelper.resurrect(din, cl);
		myRequestedRoles = VectorHelper.resurrect(din, cl);
		yourRequestedRoles = VectorHelper.resurrect(din, cl);
		active = din.readBoolean();
		traceLevel = din.readInt();
		myRoles = VectorHelper.resurrect(din, cl);
		status = StringHelper.resurrect(din);
		myProperties = HashtableHelper.resurrect(din, cl);
	}

	/**
	 * This function implements the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.write(StringHelper.persist(RequestType));
		dout.write(myId.serialize());
		dout.write(myParrentId.serialize());
		dout.write(StringHelper.persist(myState));
		dout.write(StringHelper.persist(myStateData));
		dout.write(VectorHelper.persist(myChildren));
		dout.write(VectorHelper.persist(myRequestedRoles));
		dout.write(VectorHelper.persist(yourRequestedRoles));
		dout.writeBoolean(active);
		dout.writeInt(traceLevel);
		dout.write(VectorHelper.persist(myRoles));
		dout.write(StringHelper.persist(status));
		dout.write(HashtableHelper.persist(myProperties));
		dout.flush();

		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(data);
		DataInputStream din = new DataInputStream(bin);
		RequestType = StringHelper.resurrect(din);
		myId = new ActorAddress(din);
		myParrentId = new ActorAddress(din);
		myState = StringHelper.resurrect(din);
		myStateData = StringHelper.resurrect(din);
		myChildren = VectorHelper.resurrect(din, cl);
		myRequestedRoles = VectorHelper.resurrect(din, cl);
		yourRequestedRoles = VectorHelper.resurrect(din, cl);
		active = din.readBoolean();
		traceLevel = din.readInt();
		myRoles = VectorHelper.resurrect(din, cl);
		status = StringHelper.resurrect(din);
		myProperties = HashtableHelper.resurrect(din, cl);

		return bin;
	}

	public Hashtable getMyProperties() {
		return myProperties;
	}

	public void setMyProperties(Hashtable myProperties) {
		this.myProperties = myProperties;
	}

}
