/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.*;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;
import java.util.Enumeration;
import java.util.Vector;

/**
 * The behavior for for an actor. It mainly consists of the ActorFrame protocol.
 * 
 * @author Geir Melby, Tellu AS
 * @author Knut Eilif Husa, Tellu AS
 */
public class ActorCS extends StateMachineCS implements AFConstants {
	/**
	 * The constructor
	 * 
	 * @param sn
	 *            The name of the state
	 * @param cs
	 *            The enclosing state
	 */
	public ActorCS(String sn, CompositeState cs) {
		super(sn, cs);
	}

	public ActorCS() {
		super("ActorCS");
	}

	public ActorCS(String sn) {
		super(sn);
	}

	protected State waitConfirmPorts = new State("waitConfirmPorts", this);
	protected State actorSuspended = new SuspendedCS("actorSuspended", this);

	public void outofInnerCompositeState(CompositeState cs, int exNo, StateMachine curfsm) {
		switch (exNo) {
		case 0:
			performExit(curfsm);
			nextState(init, curfsm);
			break;
		case 1:
			performExit(curfsm);
			nextState(findCurrentState(curfsm.context.getHistoryStateId()), curfsm);
			break;

		case 2:
			performExit(curfsm);
			if (curfsm.scheduler.isTraceOn()) {
				curfsm.trace.traceError("Suspended state not supported");
			}
			nextState(init, curfsm);
			break;
		}
	}

	/**
	 * Execute transition (if defined) for ActorMsg sig and State st. For each
	 * CompositeState class execTrans has to be defined. It defines the
	 * transitions (and saves) handled by its inner States (not included those
	 * inside inner CompositeStates). NB: execTrans is not called from user
	 * code.
	 * 
	 * @param sig
	 *            The consumed ActorMsg.
	 * @param st
	 *            The State to search for defined transition.
	 * @param curfsm
	 *            The performing StateMachine.
	 */
	public void execTrans(ActorMsg sig, State st, StateMachine curfsm) {
		ActorSM asm = (ActorSM) curfsm;
		// this.curfsm = curfsm;
		if (sig.equals(REPORT_REQUEST_MSG)) {
			super.execTrans(sig, st, curfsm);
			return;
		} else if (sig.equals(REPORT_RESPONS_MSG)) {
			super.execTrans(sig, st, curfsm);
			return;
		} else if (sig.equals(ACTOR_REPORT_TIMER_MSG)) {
			super.execTrans(sig, st, curfsm);
			return;
		}

		if (st == init) {
			if (sig.equals(ROLE_PLAY_MSG)) {
				AFPropertyMsg rrm = (AFPropertyMsg) sig.getProperty("rrm");
				curfsm.context.setRootPlayMsg(sig); // Actor that starts
				// creation hiearchy
				curfsm.context.setMyParentAddress(sig.getSenderRole());
				curfsm.context.setMyAddress(sig.getReceiverRole());

				if (rrm.getSenderRole() != null && rrm.getSenderRole().getActorID() != null)
					curfsm.context.addRequestorRole(rrm.getSenderRole().getActorID(), rrm.getSenderRole());

				Vector connectors = (Vector) sig.getProperty(ROLE_PLAY_MSG_CONNECTORS);

				Vector ports = asm.getActorSpec().getPortNames();
				curfsm.context.setTmpPorts(ports, connectors);

				curfsm.createPorts(connectors); // create the ports as defined.
				// Send ack to parent.
				curfsm.sendMessage(new AFPropertyMsg(AFConstants.ROLE_PLAY_ACK_MSG, true), sig.getSenderRole());

				asm.startLevel = AFConstants.LEVEL_0;

				// check if ports and connectors exists
				if ((ports == null) || ports.isEmpty() || connectors == null || connectors.isEmpty()) {
					// no ports shall be configured.

					if (asm.createParts(asm.actorSpec, asm.startLevel)) {
						// create inner parts
						performExit(curfsm);
						asm.startTimer(CREATION_TIMER_SPEC, CREATION_TIMER_ID); // create
						// timer
						// for
						// inner
						// parts
						asm.setNoOfTrialsLeft(NO_OF_RETRIALS - 1);
						asm.context.setHistoryStateId(idle.stateName());
						nextState(waitCreateAck, curfsm);

						return;
					} else {
						// no parts or inner ports to confirm.
						finishedPartCreation(sig, asm);
						// transStartPlaying(asm, idle);
						return;
					}
				} else {
					// there are ports which shall be configured.
					performExit(curfsm);
					asm.startTimer(CREATION_TIMER_SPEC, CREATION_TIMER_ID); // create
					// timer
					// for
					// port
					// creation
					asm.setNoOfTrialsLeft(NO_OF_RETRIALS - 1);
					asm.context.setHistoryStateId(idle.stateName());
					nextState(waitConfirmPorts, curfsm);

					return;
				}

			} else if (sig.equals(ROLE_CREATE_MSG)) {
				// handle role create message
				curfsm.context.setRootPlayMsg(null); // no confirm to be send
				curfsm.setPersistent(true);
				curfsm.context.setMyParentAddress(sig.getSenderRole());
				curfsm.context.setMyAddress(sig.getReceiverRole());

				Vector connectors = (Vector) sig.getProperty(ROLE_CREATE_MSG_CONNECTORS);
				Vector ports = asm.getActorSpec().getPortNames();
				curfsm.context.setTmpPorts(ports, connectors);

				// Send ack to parent.
				ActorAddress aa = sig.getSenderRole();
				if (aa.getActorDomain() == null && curfsm.getContainer() != null) {
					aa.setActorDomain(curfsm.getScheduler().getSchedulerData().getActorDomainName());
				}
				curfsm.sendMessage(new AFPropertyMsg(ROLE_CREATE_ACK_MSG, true), aa);

				// create my own ports
				curfsm.createPorts(connectors); // create the ports as defined.
				// Start with the highest level
				asm.startLevel = AFConstants.LEVEL_0;
				if ((ports == null) || ports.isEmpty() || connectors == null || connectors.isEmpty()) {
					// no ports shall be configured.
					if (asm.createParts(asm.actorSpec, asm.startLevel)) {
						asm.getScheduler().upDateVisibleActors(asm);
						// create inner parts
						performExit(curfsm);
						nextState(waitCreateAck, curfsm);
						return;
					} else {
						// no parts or inner ports to confirm.
						finishedPartCreation(sig, asm);
					}
				} else {
					// there are ports which shall be configured.

					performExit(curfsm);
					asm.startTimer(CREATION_TIMER_SPEC, CREATION_TIMER_ID); // create
					// timer
					// for
					// port
					// creation
					asm.setNoOfTrialsLeft(NO_OF_RETRIALS - 1);
					nextState(waitConfirmPorts, curfsm);
					return;
				}

			} else if (sig.equals(TIMER_MSG) && sig.getString(TIMER_ID).equals(START_LEVEL_TIMER)) {
				traceTask("Start parts on level " + asm.startLevel);
				if (asm.createParts(asm.actorSpec, asm.startLevel)) {
					asm.getScheduler().upDateVisibleActors(asm);
					// create inner parts
					performExit(curfsm);
					nextState(waitCreateAck, curfsm);
					return;
				} else {
					// no parts or inner ports to confirm on this start level.

					if (asm.startLevel < LEVEL_4) {
						asm.startLevel++;
						AFPropertyMsg timer = new AFPropertyMsg(TIMER_MSG);
						timer.setProperty(TIMER_ID, START_LEVEL_TIMER);
						sendMessage(timer, getMyActorAddress());
						sameState();
						return;
					}

					// Send ack to parent.
					finishedPartCreation(sig, asm);
					return;
				}

			}

		} else if (st == waitCreateAck) {
			if (sig.equals(ROLE_CREATE_ACK_MSG)) {
				asm.context.removeCreationChild(sig.getSenderRole());

				Vector v = asm.context.getCreationOfChildren();

				asm.context.addPersistentChildrenRole(sig.getSenderRole());

				if (v.isEmpty()) {
					// No more parts to be created on this level
					if (asm.startLevel < LEVEL_4) {
						performExit();
						asm.startLevel++;
						startTimer(START_LEVEL_TIMER_VALUE, START_LEVEL_TIMER);
						nextState(init);
						return;
					}
					// Send ack to parent.
					if (asm.context.getRootPlayMsg() != null) {
						asm.stopTimer(CREATION_TIMER_ID);
					}

                    // all inner parts are created on this level, check if we need to
                    // create any port-connectors for these inner parts.

					finishedPartCreation(sig, asm);
					return;
				} else {
					sameState(curfsm);
					return;
				}
			} else if (sig.equals(ROLE_CREATE_NACK_MSG)) {
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Inner part rejected request -- creation of inner parts aborted");
				}

				if (asm.context.getParentAddress().getActorID() != null) {
					curfsm.sendMessage(new AFPropertyMsg(ROLE_PLAY_NACK_MSG, true), curfsm.context.getActorAddress());
				}

				// if root node, the requestor of the RolePlayMsg should be
				// informed
				sendRolePlayNackMsg(curfsm);
				asm.sendMessage(new AFPropertyMsg(ROLE_REMOVE_MSG, true), asm.context.getActorAddress());
				sameState(curfsm);
				return;
			} else {
				super.execTrans(sig, st, curfsm);
			}

		} else if (st == waitConfirmPorts) {
			if (sig.equals(PORT_CREATE_ACK_MSG)) {
				// A port has been created.
				if (asm.context.allPortsAcked(sig.getSenderRole().getActorPort())) { // all
					// ports
					// confirmed
					asm.stopTimer(CREATION_TIMER_ID); // stop the port creation
					// timer

					if (asm.createParts(asm.actorSpec, asm.startLevel)) {

						Object rpm = asm.context.getRootPlayMsg();

						if (rpm != null) {
							// start creation timer if this is a RolePlay
							asm.startTimer(CREATION_TIMER_SPEC, CREATION_TIMER_ID); // create
							// timer
							// for
							// port
							// creation
							asm.setNoOfTrialsLeft(NO_OF_RETRIALS - 1);
						}

						// create inner parts
						performExit(curfsm);
						nextState(waitCreateAck, curfsm);
						return;
					} else {
						// no parts or inner ports to confirm.
						performExit();
						if (asm.startLevel < LEVEL_4) {
                            asm.startLevel++;
                            AFPropertyMsg timer = new AFPropertyMsg(TIMER_MSG);
                            timer.setProperty(TIMER_ID, AFConstants.START_LEVEL_TIMER);
                            sendMessage(timer, getMyActorAddress());
                            nextState(init);
                            return;
                        }

                        // Send ack to parent.
                        finishedPartCreation(sig, asm);
                        return;
					}

				} else {
					// more ports to wait for.
					sameState(curfsm);
					return;
				}

				// } else if (sig instanceof PortCreateNackMsg) {
			} else if (sig.equals(PORT_CREATE_NACK_MSG)) {
				// creation of a port failed.
				performExit(curfsm);
				asm.stopTimer(CREATION_TIMER_ID); // stop port the creation
				// timer.
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Port creation failed, creation of ports aborted");
				}

				if (asm.context.getParentAddress().getActorID() != null) {
					// asm.sendMessage(new RoleCreateNackMsg(),
					// asm.context.getParentAddress());
					asm.sendMessage(new AFPropertyMsg(ROLE_CREATE_NACK_MSG, true), curfsm.context.getParentAddress());
				}

				// if root node, the requestor of the RolePlayMsg should be
				// informed
				sendRolePlayNackMsg(curfsm);
				// remove its children and itself
				// asm.sendMessage(new RoleRemoveMsg(),
				// asm.context.getActorAddress());
				asm.sendMessage(new AFPropertyMsg(ROLE_REMOVE_MSG, true), asm.context.getActorAddress());

				if (asm.isVisible()) {
					asm.stopTimer(ActorSM.ROUTER_UPDATE_INTERVAL_ID);
				}

				nextState(idle, curfsm);
				return;
			} else if (sig.equals(TIMER_MSG) && sig.getProperty(TIMER_ID).equals(CREATION_TIMER_ID)) {
				// Resend the port connector requests to inner children
				if (asm.getNoOfTrialsLeft() > 0) {
					// decrement no of trials and resend connector requests
					asm.setNoOfTrialsLeft(asm.getNoOfTrialsLeft() - 1);

					Enumeration enumer = asm.context.getTmpPorts().elements();

					while (enumer.hasMoreElements()) {
						// notify the port of the timeout event.
						// asm.sendMessage(new PortTimeOutMsg(), (String)
						// enumer.nextElement());
						asm.sendMessage(new AFPropertyMsg(PORT_TIME_OUT_MSG, true), (String) enumer.nextElement());
					}

					asm.startTimer(CREATION_TIMER_SPEC, CREATION_TIMER_ID);
					sameState(curfsm);
					return;
				} else {
					performExit(curfsm);
					if (asm.scheduler.isTraceOn()) {
						asm.trace.traceError("ActorCS: Port creation timed out, creation of ports aborted");
					}

					if (asm.context.getParentAddress().getActorID() != null) {
						// asm.sendMessage(new RoleCreateNackMsg(),
						// asm.context.getParentAddress());
						asm.sendMessage(new AFPropertyMsg(ROLE_CREATE_NACK_MSG, true), asm.context.getParentAddress());
					}

					sendRolePlayNackMsg(curfsm);
					// remove its children and itself
					asm.sendMessage(new AFPropertyMsg(ROLE_REMOVE_MSG, true), asm.context.getActorAddress());

					if (asm.isVisible()) {
						asm.stopTimer(ActorSM.ROUTER_UPDATE_INTERVAL_ID);
					}

					nextState(idle, curfsm);
					return;
				}
			}
		}

		// Behavior that is common for some states, multiple state names in UML
		if (st == waitCreateAck || st == waitConfirmPorts || st == actorSuspended) {

			if (!(sig.equals(ROLE_REMOVE_MSG) || sig.equals(ROLE_RELEASE_MSG) || sig.equals(ROLE_PLAY_ENDED_MSG)
					|| sig.equals(SET_ACTOR_TRACE_MSG) || sig.equals(REPORT_REQUEST_MSG)
					|| sig.equals(REPORT_RESPONS_MSG) || sig.equals(ACTOR_REPORT_TIMER_MSG))) {
				// save the message until other state is reached.
				save(sig, curfsm);
				sameState(curfsm);

				return;
			}
		}

		// Common for all states except specified, that equal to "all, but"
		// concept for this state machine
		if (st != init) {
			if (sig.equals(REQUEST_STATUS_MSG)) {
				AFPropertyMsg rsm = (AFPropertyMsg) sig;

				if (rsm.getString(REQUEST_TYPE_PROP).equals("partspec")) {
					String result = "<?xml version=\"1.0\"?>\n" + "<partspec>" + asm.myActorId + "@" + asm.myActorType
							+ "</partspec>\n";
					AFPropertyMsg resp = new AFPropertyMsg(RESPONS_STATUS_MSG, true);
					resp.setProperty(RESULT_TYPE_PROP, sig.getString(REQUEST_TYPE_PROP));
					resp.setProperty(RESULT_PROP, result);
					resp.setProperty(JAVA_OBJECT_PROP, asm.getPartSpecs());
					asm.sendMessage(resp, rsm.getSenderRole());
					sameState(curfsm);

					return;
				}
			} else if (sig.equals(PART_SPEC_REQUEST_MSG)) {
				AFPropertyMsg rsm = (AFPropertyMsg) sig;
				String result;
				String visible;

				if (asm.isVisible()) {
					visible = "true";
				} else {
					visible = "false";
				}

				result = "<?xml version=\"1.0\"?>\n" + "<partspec>" + asm.myActorId + "@" + asm.myActorType
						+ "</partspec>\n";

				AFPropertyMsg msg = new AFPropertyMsg(PART_SPEC_RESPONSE_MSG, true);
				msg.setProperty(ACTOR_TYPE_PROP, result);
				msg.setProperty(VISIBLE_PROP, visible);
				msg.setProperty(PART_SPECS_PROP, asm.getApplicationSpec());

				asm.sendMessage(msg, rsm.getSenderRole());
				sameState(curfsm);

				return;
			} else if (sig.equals(ROLE_REQUEST_MSG)) {
				// check first if an role exists for that role type
				// All persistent ROLES HAvE BEEN CREATED ALREADY, but not 
				AFPropertyMsg rrm = (AFPropertyMsg) sig;
				String roleType = (String) rrm.getProperty(ActorAddress.ROLE_TYPE);

				RoleSpec roleSpec = asm.getActorSpec().getRoleSpec(roleType);
				if (roleSpec != null) {
					// the role spec exists.
					AFPropertyMsg msg = new AFPropertyMsg(ROLE_PLAY_MSG, true);
					msg.setProperty(ROLE_PLAY_MSG_RRM, rrm);
					msg.setProperty(ROLE_PLAY_MSG_PORTS, null);
					msg.setProperty(ROLE_PLAY_MSG_CONNECTORS, null);
					String roleId = rrm.getString(ActorAddress.ROLE_ID);
					String id = asm.makeRoleName(roleId);
					asm.sendMessage(msg, new ActorAddress(id, roleType));
					return;
				}

				super.execTrans(sig, st, curfsm);

				return;

			} else if (sig.equals(ROLE_CONFIRM_MSG)) {
				super.execTrans(sig, st, curfsm);
				return;
				
			} else if (sig.equals(ROLE_PLAY_ACK_MSG)) {
				asm.createConnectors(sig.getSenderRole());
				sameState(curfsm);
				return;
				
			} else if (sig.equals(ROLE_PLAY_NACK_MSG)) {
				super.execTrans(sig, st, curfsm);
				return;
				
			} else if (sig.equals(ROLE_RELEASE_MSG)) {
				super.execTrans(sig, st, curfsm);
				return;

			} else if (sig.equals(ROLE_PLAY_ENDED_MSG)) {
				super.execTrans(sig, st, curfsm);
				return;

			} else if (sig.equals(ROLE_DENIED_MSG)) {
				super.execTrans(sig, st, curfsm);
				return;

			} else if (sig.equals(ROLE_PLAY_MSG)) {
				// sends RoleConfirm to the requestor, and RolePlayAck to the
				// senderAddress of RolePlay
				AFPropertyMsg rrm = (AFPropertyMsg) sig.getProperty("rrm");
				ActorAddress requestor = rrm.getSenderRole();
				asm.context.addRequestorRole(requestor.getActorID(), requestor);

				AFPropertyMsg msg = new AFPropertyMsg(ROLE_CONFIRM_MSG, true);
				msg.setProperty("rrm", rrm);
				asm.sendMessage(msg, rrm.getSenderRole());

				curfsm.sendMessage(new AFPropertyMsg(ROLE_PLAY_ACK_MSG, true), sig.getSenderRole());
				sameState(curfsm);

				return;

			} else if (sig.equals(ROLE_RESET_MSG)) {
				super.execTrans(sig, st, curfsm);

				return;
			} else if (sig.equals(ROLE_REMOVE_MSG)) {
				// Force the port to release all connectors.
				Enumeration ports = asm.ports.keys();

				while (ports.hasMoreElements()) {
					String pname = (String) ports.nextElement();
					// asm.sendMessage(new PortRemoveMsg(), pname);
					asm.sendMessage(new AFPropertyMsg(PORT_REMOVE_MSG, true), pname);
				}

				super.execTrans(sig, st, curfsm);
				return;

			} else if (sig.equals(LOOK_UP_MSG)) {
				Vector res = asm.searchForActors(sig.getString(ACTOR_ID_PROP), sig.getString(ACTOR_TYPE_PROP));
				AFPropertyMsg msg = new AFPropertyMsg(LOOK_UP_RESULT_MSG, true);
				msg.setProperty(RES_PROP, res);
				asm.sendMessage(msg, sig.getSenderRole());
				sameState(curfsm);
				return;
			} else if (sig.equals(ROLE_CREATE_MSG)) {
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceError("ActorCS: Instance already created");
				}
				asm.sendMessage(new AFPropertyMsg(ROLE_CREATE_NACK_MSG, true), sig.getSenderRole());
				sameState(asm);

				return;
			} else if (sig.equals(SET_ACTOR_TRACE_MSG)) {
				AFPropertyMsg satm = (AFPropertyMsg) sig;
				asm.setTraceLevel(satm.getInt(TRACE_LEVEL_PROP));
				asm.sendMessage(sig, asm.context.getChildrenRoles());
				if (asm.scheduler.isTraceOn()) {
					asm.trace.traceTask("Trace level: " + TraceConstants.getTraceLevel(satm.getInt(TRACE_LEVEL_PROP)));
				}
				sameState(curfsm);

				return;
			} else if (sig.equals(START_PLAYING_MSG)) {
				asm.routerUpdate();

				return;
			} else if (sig.equals(DELETE_ACTOR_MSG)) {
				ActorAddress aa = asm.context.getChildrenRole(sig.getString("actorId"));

				if (aa != null) {
					// the requested actorid exists
					asm.sendMessage(new AFPropertyMsg(ROLE_REMOVE_MSG, true), aa); // Send
					// RolePlay
					// message
					// to
					// the
					// existing
					// actor
					asm.sendMessage(new AFPropertyMsg(MANAGEMENT_REQUEST_CONFIRM_MSG, true).setProperty(INFO_PROP, aa
							+ " deleted"), sig.getSenderRole());
				}

				sameState(curfsm);

				return;
			} else if (sig.equals(REQUEST_STATUS_MSG)) {
				// List out children
				AFPropertyMsg rsm = (AFPropertyMsg) sig;
				String res = "Children: ";
				Vector v = asm.context.getChildrenRoles();

				for (int i = 0; i < v.size(); i++) {
					ActorAddress actorAddress = (ActorAddress) v.elementAt(i);
					res = res + actorAddress + ", ";
				}

				AFPropertyMsg resp = new AFPropertyMsg(RESPONS_STATUS_MSG, true);
				resp.setProperty(RESULT_TYPE_PROP, "text");
				resp.setProperty(RESULT_PROP, res);
				asm.sendMessage(resp, rsm.getSenderRole());
				sameState(curfsm);

				return;
			}
		}

		if (st != init && st != actorSuspended) {
			if (sig.equals(ROLE_CREATE_ACK_MSG)) {
				super.execTrans(sig, st, curfsm);

				return;
			} else if (sig.equals(ROLE_CREATE_NACK_MSG)) {
				super.execTrans(sig, st, curfsm);

				return;
			}
		}

		// code that implements management functionality for all actors. This
		// includes adding and deleting children.
		// The actor type has to be specified in the PartSpec specified with
		// Actor descriptor files or set in the
		// initInstance() method. These functions are only valid in user-defined
		// states, which means that they
		// are saved in all actor specific states.
		if ((st != init) && (st != waitConfirmPorts) && (st != actorSuspended) && (st != waitCreateAck)) {
			if (sig.equals(REQUEST_STATUS_MSG)) {
				// List out children
				String res = "Children: ";
				Vector v = asm.context.getChildrenRoles();

				for (int i = 0; i < v.size(); i++) {
					ActorAddress actorAddress = (ActorAddress) v.elementAt(i);
					res = res + actorAddress + ", ";
				}

				AFPropertyMsg resp = new AFPropertyMsg(RESPONS_STATUS_MSG, true);
				resp.setProperty(RESULT_TYPE_PROP, "text");
				resp.setProperty(RESULT_PROP, res);
				asm.sendMessage(resp, sig.getSenderRole());
				sameState(curfsm);

				return;
				// Receeive suspending message, send
			} else if (sig.equals(SUSPEND_MSG)) {
				performExit(curfsm);
				asm.suspendActor = sig.getSenderRole();
				asm.setHistoryState(st);
				nextState(actorSuspended, curfsm);
				return;
			} /*
			 * else if (sig.equals(RESUME_MSG)) { sameState(curfsm);
			 * asm.sendMessage(RESUMED_MSG, sig.getSenderRole()); return; }
			 */
		} else

		{
			// in actor specific states, save all management messages
			if (sig.equals(REQUEST_STATUS_MSG)) {
				save(sig, curfsm);
			}

			if (sig.equals(DELETE_ACTOR_MSG)) {
				save(sig, curfsm);
			}

			if (sig.equals(ADD_ACTOR_MSG)) {
				save(sig, curfsm);
			}
		}
	}

	private void finishedPartCreation(ActorMsg sig, ActorSM asm) {
		asm.createConnectors();

		asm.getScheduler().upDateVisibleActors(asm);

		ActorAddress aa = sig.getSenderRole();
		if (aa.getActorDomain() == null && asm.getContainer() != null) {
			aa.setActorDomain(asm.getScheduler().getSchedulerData().getActorDomainName());
		}
		// asm.sendMessage(new AFPropertyMsg(ROLE_CREATE_ACK_MSG, true), aa);

		transStartPlaying(asm, idle);
	}

	/**
	 * Check if a state is in the States as defined by generic actor behavior
	 * sm.
	 * 
	 * @param st
	 *            State to check.
	 * @return true iff the state is part of the generic actor behavior sm.
	 */
	protected boolean isStateIsInSuper(State st) {
		return ((st == init) || (st == waitConfirmPorts) || (st == waitCreateAck) || (st == actorSuspended));
	}
}