/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.actorframe.application.Application;
import org.coos.javaframe.TraceConstants;

import java.io.File;
import java.util.Timer;
import java.util.TimerTask;

/**
 * This class will monitor the ApplicationSpec xml-file for modifications and
 * perform an application updates.
 * 
 * @author Robert Bjarum, Tellu AS
 */
public class ActorSpecFileMonitor {
	private Application application;
	private File applicationSpecFile;
	private long timerDelay = 1000;
	private Timer timer;
	private long lastModyfied;

	public ActorSpecFileMonitor(Application application, File applicationSpecFile) {
		this.application = application;
		this.applicationSpecFile = applicationSpecFile;
		lastModyfied = applicationSpecFile.lastModified();
	}

	public void setTimerDelay(long timerDelay) {
		this.timerDelay = timerDelay;
	}

	public void startMonitoring() {
		application.applictionActive();
		startTimer();
	}

	private void waitState(int state) {
		try {
			while (application.getState() != state) {
				Thread.sleep(10);
			}
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}

	private void startTimer() {
		if (timer == null) {
			timer = new Timer();
		}

		timer.schedule(new J2SETimerTask(), timerDelay);
	}

	private class J2SETimerTask extends TimerTask {
		public void run() {
			// check the applicationSpecFile her
			if (application.getState() != Application.STATE_ACTIVE) {
				startTimer();

				return;
			}

			long modified = applicationSpecFile.lastModified();

			if (modified > lastModyfied) {
				lastModyfied = modified;
				application.getLogger().log(TraceConstants.tlInfo, "File changed: " + applicationSpecFile.getName());
				application.suspend();
				waitState(Application.STATE_SUSPENDED);
				application.getLogger().log(TraceConstants.tlInfo,
						"Application suspended. State: " + application.getState());
				application.updateApplication();
				waitState(Application.STATE_UPDATED);
				application.resume();
				waitState(Application.STATE_ACTIVE);
			}

			startTimer();
		}
	}
}
