/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import java.util.Hashtable;

/**
 * Factor class for Logger. This factory will search for predefined loggers. You
 * may insert your own implementation of Logger class using the method
 * setLoggerImplClass().
 * 
 * @author Knut Eilif Husa, Tellu AS
 * @author Robert Bjarum, Tellu AS
 * @see Logger
 */
public class LoggerFactory {
    static boolean traceOn = true, traceError = true;
    private static Class LOGGER_IMPL_CLASS = null;
    private static Hashtable LOGGERS = new Hashtable();

    public static boolean isTraceOn() {
        return traceOn;
    }

    public static void setTrace(boolean enabled) {
        traceOn = enabled;
    }

    public static boolean isTraceError() {
        return traceError;
    }

    public static void setTraceError(boolean on) {
        traceError = on;
    }

    public static void setLoggerImplClass(Class loggerImplClass) {
        LOGGER_IMPL_CLASS = loggerImplClass;
        Logger logger;
        if (loggerImplClass != null) {
            try {
                logger = (Logger) LOGGER_IMPL_CLASS.newInstance();
            } catch (Exception e) {
                System.err.println("Failed create instance of " + LOGGER_IMPL_CLASS.getClass().getName());
                System.err.println();
                e.printStackTrace();
            } finally {
                logger = getLogger(LoggerFactory.class.getName());
                logger.log(TraceConstants.tlInfo, "Using " + logger.getClass().getName());
            }
        }
    }
    
    private static Logger createLogger(Class clazz, String loggerName) throws Exception {
        Logger logger = (Logger) clazz.newInstance();
        logger.setLoggerName(loggerName);
        LOGGERS.put(loggerName, logger);
        return logger;
    }

    public static Logger getLogger(String loggerName) {
        Logger logger = (Logger) LOGGERS.get(loggerName);
        if (logger != null) {
            return logger;
        }
        
        /*
         * Your own Logger implementation
         */
        try {
            if (LOGGER_IMPL_CLASS != null) {
                return createLogger(LOGGER_IMPL_CLASS, loggerName);
            }
        } catch (Exception e) {
        }

        /*
         * Logger for J2SE
         */
        try {
            return createLogger(Class.forName("org.coos.javaframe.J2SELoggerImpl"), loggerName);
        } catch (Exception e) {
        }

        try {
            return createLogger(Class.forName("no.tellu.cdc.javaframe.LoggerImpl"), loggerName);
        } catch (Exception e) {
        }

        try {
            return createLogger(Class.forName("org.coos.javaframe.CLDCLoggerImpl"), loggerName);
        } catch (Exception e) {
        }

        logger = new BasicLoggerImpl(loggerName);
        LOGGERS.put(loggerName, logger);
        return logger;
    }
}
