/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe.messages;

import org.coos.javaframe.ActorAddress;
import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.HashtableHelper;
import org.coos.util.serialize.StringHelper;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * The Actoframe property message.
 */
public class AFPropertyMsg extends ActorMsg {
	public String msgId;
	public Hashtable property;
	private Object o;

	public AFPropertyMsg() {
	}

	public String getMsgId() {
		return msgId;
	}

	public AFPropertyMsg(ActorMsg am, String msgId) {
		this.setSenderRole(am.getSenderRole());
		this.setReceiverRole(am.getReceiverRole());
		this.msgId = msgId;
	}

	public void setMsgId(String msgId) {
		this.msgId = msgId;
		this.frameworkMsg = false;
	}

	public void setProperties(Hashtable properties) {
		this.property = properties;
	}

	public AFPropertyMsg(String msgId) {
		this.msgId = msgId;
	}

	public AFPropertyMsg(String msgId, boolean frameWorkMsg) {
		this.frameworkMsg = frameWorkMsg;
		this.msgId = msgId;
	}

	public AFPropertyMsg(String msgId, ActorAddress receiver, ActorAddress sender) {
		this.msgId = msgId;
		this.setReceiverRole(receiver);
		this.setSenderRole(sender);
	}

	public AFPropertyMsg(String msgId, String receiver, String sender) {
		this.msgId = msgId;
		this.setReceiverRole(new ActorAddress(receiver));
		this.setSenderRole(new ActorAddress(sender));
	}
	
	public Object getBody() {
		if (property == null) {
			return null;
		}

		o = property.get(BODY_PROPERTY);

		return o;
	}

	public Hashtable getProperty() {
		return property;
	}

	public Object getProperty(String name) {
		if (property == null) {
			return null;
		}

		o = property.get(name);

		return o;
	}

	public String getString(String name, String defaultValue) {
		if (property == null) {
			return defaultValue;
		}

		o = property.get(name);
		if (o == null)
			return defaultValue;
		else
			return (String) o;
	}

	public String getString(String name) {
		if (property == null) {
			return null;
		}

		o = property.get(name);

		return (String) o;
	}

	public ActorAddress getActorAddress(String name) {
		if (property == null) {
			return null;
		}

		o = property.get(name);

		if (o == null) {
			return null;
		}

		return (ActorAddress) o;
	}

	public Vector getVector(String name) {
		if (property == null) {
			return null;
		}

		o = property.get(name);

		if (o == null) {
			return null;
		}

		return ((Vector) o);
	}

	/**
	 * Get int value.
	 * <br/>
	 * Property can be of type Long and Integer.
	 * @throws PropertyException if casting from Long to Integer changes the value
	 */
	public int getInt(String name) {
		if (property == null) {
			return 0;
		}

		o = property.get(name);
		
		 if (o == null) {
		     return 0;
		 }

        if (o instanceof Integer) {
            // Normal condition checked first
            return ((Integer) o).intValue();
        } else if (o instanceof Long) {
            long value = ((Long) o).longValue();
            int result = (int) value;
            if (value == (long) result) {
                return result;
            } else {
                throw new PropertyException("Cannot cast Long value " + value + " to (int) without truncation. Use getLong() instead.");
            }
        } else {
            //This will intentionally throw an exception
            return ((Integer) o).intValue();
        }
	}

	public int getInt(String name, int defaultValue) {
		if (property == null) {
			return defaultValue;
		}

		o = property.get(name);

		if (o == null) {
			return defaultValue;
		}
		
		return getInt(name);
	}

	public boolean getBoolean(String name) {
		if (property == null) {
			return false;
		}

		o = property.get(name);
		if (o == null) {
			return false;
		}

		return ((Boolean) o).booleanValue();
	}

	public AFPropertyMsg setByteArray(String name, byte[] array) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, array);

		return this;
	}

	public byte[] getByteArray(String name) {
		if (property == null) {
			return null;
		}

		o = property.get(name);

		if (o == null) {
			return null;
		}

		return ((byte[]) o);
	}

	public AFPropertyMsg setDouble(String name, double value) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, new Double(value));

		return this;
	}

	public double getDouble(String name) {
		if (property == null) {
			return 0.0;
		}

		o = property.get(name);

		if (o == null) {
			return 0.0;
		}
		
		if (o instanceof Double) {
		    return ((Double) o).doubleValue();
		} else if (o instanceof Integer) {
		    return ((Integer)o).doubleValue();
		} else if (o instanceof Long) {
		    return ((Long)o).doubleValue();
		}

		return ((Double) o).doubleValue();
	}

	public AFPropertyMsg setFloat(String name, float value) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, new Float(value));

		return this;
	}

	public float getFloat(String name) {
		if (property == null) {
			return 0;
		}

		o = property.get(name);

		if (o == null) {
			return 0;
		}
		
        if (o instanceof Float) {
            // Normal condition checked first
            return ((Float) o).floatValue();
        } else if (o instanceof Double) {
            double value = ((Double) o).doubleValue();
            if (value >= -Float.MAX_VALUE && value <= Float.MAX_VALUE) {
                return (float) value;
            } else {
                throw new PropertyException("Cannot cast Double value " + value + " to (float) without truncation. Use getDouble() instead.");
            }
        } else if (o instanceof Integer) {
            return ((Integer)o).floatValue();
        } else if (o instanceof Long) {
            return ((Long)o).floatValue();
        } else {
            //This will intentionally throw an exception
            return ((Float) o).floatValue();
        }
	}

	public AFPropertyMsg setLong(String name, long value) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, new Long(value));

		return this;
	}

	/**
	 * Return long value.<br/>
	 * Property with can be an instance of Long or Integer.
	 */
	public long getLong(String name) {
		if (property == null) {
			return 0;
		}

		o = property.get(name);

		if (o == null) {
			return 0;
		}
		
		if (o instanceof Long) {
		    return ((Long) o).longValue();
		} else {
		    return ((Integer) o).longValue();
		}

	}

	public long getLong(String name, long defaultValue) {
		if (property == null) {
			return defaultValue;
		}

		o = property.get(name);

		if (o == null) {
			return defaultValue;
		}

		return getLong(name);
	}

	public AFPropertyMsg setProperty(String name, Object value) {
		if (value == null) {
			removeProperty(name);
			return this;
		}

		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, value);

		return this;
	}

	public AFPropertyMsg setBoolean(String name, boolean value) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, new Boolean(value));

		return this;
	}

	public AFPropertyMsg setInt(String name, int value) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(name, new Integer(value));

		return this;
	}

	public String getSignalName() {
		return msgId;
	}

	public boolean hasProperty(String propertyName) {
		return property.containsKey(propertyName);
	}
	
	

	public AFPropertyMsg setBody(Object body) {
		if (property == null) {
			property = new Hashtable();
		}

		property.put(BODY_PROPERTY, body);

		return this;
	}
	
	

	/**
	 * Chkec if two messages are identical, of same type. If the object is
	 * another AFPropertyMsg, they msgId's is compared
	 * 
	 * @param o
	 *            may be AFpropertyMsg or a String
	 * @return
	 */
	public boolean equals(Object o) {
		if (o instanceof AFPropertyMsg) {
			return ((AFPropertyMsg) o).msgId.equals(this.msgId);
		} else if (o instanceof String) {
			return o.equals(msgId);
		} else {
			return false;
		}
	}

	/**
	 * Create a deep copy of this message.
	 * 
	 * @param cl
	 *            container dependent classloader
	 * @return the cloned message
	 */
	public AFPropertyMsg cloneAFPropertyMsg(AFClassLoader cl) {
		return (AFPropertyMsg) super.cloneMsg(cl);
	}

	public String messageContent() {
		StringBuffer sb = new StringBuffer();
		sb.append("MsgId: " + msgId);

		if (property != null) {
			Enumeration en = property.keys();

			while (en.hasMoreElements()) {
				String s = (String) en.nextElement();
				sb.append(" " + s + ": " + property.get(s) + " : ");
			}
		}

		return super.messageContent() + sb;
	}

	public String toString() {
		return super.toString() + ", MSG_ID: " + msgId;
	}

	public AFPropertyMsg(InputStream is, AFClassLoader cl) throws IOException {
		DataInputStream din = new DataInputStream(is);
		super.readInput(din, null);
		if (getMsgType() != MSG_TYPE_TEST) {
			msgId = StringHelper.resurrect(din);
			property = HashtableHelper.resurrect(din, cl);
		}
	}

	/**
	 * This function implements the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.write(super.serialize());
		if (getMsgType() != MSG_TYPE_TEST) {
			dout.write(StringHelper.persist(msgId));
			dout.write(HashtableHelper.persist(property));
		}
		dout.flush();

		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = super.deSerialize(data, cl);
		DataInputStream din = new DataInputStream(bin);
		if (getMsgType() != MSG_TYPE_TEST) {
			msgId = StringHelper.resurrect(din);
			property = HashtableHelper.resurrect(din, cl);
		}
		return bin;
	}

	/*
	 * private void readInput(DataInputStream din, AFClassLoader cl) throws
	 * IOException { msgId = StringHelper.resurrect(din); property =
	 * HashtableHelper.resurrect(din, cl); }
	 */

	public Object removeProperty(String name) {
		if (property != null) {
			return property.remove(name);
		} else {
			return null;
		}
	}
}
