/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.javaframe.CompositeState;
import org.coos.javaframe.State;
import org.coos.javaframe.StateMachine;
import org.coos.util.serialize.AFClassLoader;
import org.coos.javaframe.messages.ActorMsg;
import org.coos.util.serialize.StringHelper;

import java.io.*;

/**
 * This class implements a simple port state machine. The method signatures in
 * this class is almost the equivalent to the CompositeState/ActorSM usage.
 * However, the Port state machine is more lightweight and as of yet does not
 * implement all the functionality.
 * <p/>
 * The main usage of this class is intended for validation of messages received
 * and sent through the Port class.
 * 
 * @author Geir Melby, Tellu AS
 */
public class PortSM extends CompositeState {
	protected State currentState;
	protected State nextState;
	protected boolean performExitIsDone = false;
	public State init = new State("init", this);

	public PortSM() {
		super("");
	}

	/**
	 * Overriders of this method should not call super, as the default behavior
	 * is to allow the message.
	 * 
	 * @param sig
	 * @param st
	 * @param curfsm
	 */
	protected void execTrans(ActorMsg sig, State st, Port curfsm) {
		if (st == init) {
			// default is to send the message
			curfsm.sendMessage(sig);
			sameState(curfsm);

			return;
		}
	}

	// get the current state
	protected State getCurrentState() {
		return currentState;
	}

	// set the current state
	protected void setCurrentState(State currentState) {
		this.currentState = currentState;
	}

	public void enterState(Port psm) {
		currentState = init;
	}

	/**
	 * Set the next state. Ensure performExit() is called before you call this.
	 * 
	 * @param nextState
	 *            next state to be in.
	 * @param psm
	 *            Port instance.
	 */
	public void nextState(State nextState, Port psm) {
		this.currentState = nextState;
	}

	/**
	 * Keep the same state.
	 * 
	 * @param psm
	 *            Port instance
	 */
	public void sameState(Port psm) {
		// do nothing
		if (performExitIsDone) {
		} else {
			nextState = null;
		}
	}

	/**
	 * Execute exit() from currentState up to (but not including) this
	 * CompositeState. The method is to be used within the method execTrans(). A
	 * performExit flag in StateMachine is set to enable check of inconsistent
	 * use of nextstate and performExit in the transistion.
	 */
	public void performExit(Port psm) {
		// TODO: Fix performExit
		// State st = currentState;
		performExitIsDone = true;
	}

	/**
	 * Override this method to set default instance values for the instance.
	 */
	protected void initInstance() {
	}

	/**
	 * This function must implement the serialization of the object. This is
	 * used to save instance variables contained by the protocol state machine.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	protected byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.write(StringHelper.persist(currentState.stateName()));

		// dout.write(StringHelper.persist(nextState.stateName()));
		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object state data.
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	protected ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(data);
		DataInputStream din = new DataInputStream(bin);
		currentState = findCurrentState(StringHelper.resurrect(din));
		return bin;
	}

	public void execTrans(ActorMsg sig, State st, StateMachine curfsm) {
		System.exit(1); // should not be used
	}
}
