/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import java.util.Vector;

import org.coos.javaframe.ActorAddress;
import org.coos.javaframe.TraceObject;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;

public class XmlBuilder {
    private static String QUOTE = "\"";

    /**
     * Create XML representation of TraceObject.
     * <br/>
     * If the trace objects logger has trace enabled, i.e. logger.isTraceEnabled(),
     * then message contents will also be included.
     * 
     * @param to the TraceObject instance
     * @return XML as string
     */
    public static String toXML(TraceObject to) {
        Vector v;
        boolean includeContent = to.getLogger() == null ? false : to.getLogger().isTraceEnabled();
        StringBuffer sb = new StringBuffer();
        sb.append("<transition id=");
        sb.append(quoted(to.getTransitionId()));
        sb.append(" actor=");
        sb.append(quoted(to.getCurrentStateMachine()));
        sb.append(" currentState=");
        sb.append(quoted(to.filterStateName(to.getCurrentState())));
        sb.append(" nextState=");
        sb.append(quoted(to.filterStateName(to.getNewState())));
        sb.append(" timestamp=");
        sb.append(quoted(new Long(System.currentTimeMillis())));
        sb.append(" >");
        if (to.getInputSignal() != null) {
            sb.append("<input>");
            sb.append(toXMLInputSignal(to, includeContent));
            sb.append("</input>");
        }
        if (to.getOutputSignals() != null && to.getOutputSignals().size() > 0) {
            v = to.getOutputSignals();
            sb.append("<output>");
            for (int i = 0; i < v.size(); i++) {
                sb.append(toXML((ActorMsg) v.elementAt(i), includeContent));
            }
            sb.append("</output>");
        }
        if (to.getTraceTask() != null && to.getTraceTask().length() > 0) {
            sb.append("<task>");
            sb.append(toXmlString(to.getTraceTask()));
            sb.append("</task>");
        }
        if (to.getErrors() != null && to.getErrors().size() > 0) {
            v = to.getErrors();
            sb.append("<errors>");
            for (int i = 0; i < v.size(); i++) {
                sb.append("<error>");
                sb.append(toXmlString((String) v.elementAt(i)));
                sb.append("</error>");
            }
            sb.append("</errors>");
        }
        if (to.getWarnings() != null && to.getWarnings().size() > 0) {
            v = to.getWarnings();
            sb.append("<warnings>");
            for (int i = 0; i < v.size(); i++) {
                sb.append("<warning>");
                sb.append(toXmlString((String) v.elementAt(i)));
                sb.append("</warning>");
            }
            sb.append("</warnings>");
        }
        sb.append("</transition>");
        return sb.toString();
    }
    
    /**
     * Create XML representation of InputSignal from TraceObject.
     * 
     * @param msg the message
     * @param includeContent if true, message content will also be included
     * @return XML as string
     */
    private static String toXMLInputSignal(TraceObject to, boolean includeContent) {
        ActorMsg msg = to.getInputSignal();
        StringBuffer sb = new StringBuffer();
        sb.append("<message");
        sb.append(" id=");
        sb.append(quoted(new Long(to.getInputSignalMsgRef())));
        sb.append(" name=");
        sb.append(quoted(msg.getSignalName()));
        
        sb.append(" receiver=");
        sb.append(quoted(to.getInputSignalReceiver()));
        
        sb.append(" sender=");
        sb.append(quoted(to.getInputSignalSender()));

        sb.append(" >");
        if (msg instanceof AFPropertyMsg) {
            AFPropertyMsg afMsg = (AFPropertyMsg) msg;
            if (afMsg.getProperty() != null && afMsg.getProperty().size() > 0) {
                sb.append("<properties>");
                sb.append(toXmlString(afMsg.getProperty().toString()));
                sb.append("</properties>");
            }
        }
        sb.append("</message>");
        return sb.toString();
    }
    
    /**
     * Create XML representation of ActorMsg.
     * 
     * @param msg the message
     * @param includeContent if true, message content will also be included
     * @return XML as string
     */
    public static String toXML(ActorMsg msg, boolean includeContent) {
        StringBuffer sb = new StringBuffer();
        sb.append("<message");
        sb.append(" id=");
        sb.append(quoted(msg.getMsgRef()));
        sb.append(" name=");
        sb.append(quoted(msg.getSignalName()));
        
        sb.append(" receiver=");
        ActorAddress a = msg.getReceiverRole();
        sb.append(quoted(a));
        
        sb.append(" sender=");
        a = msg.getSenderRole();
        sb.append(quoted(a));

        sb.append(" >");
        if (msg instanceof AFPropertyMsg) {
            AFPropertyMsg afMsg = (AFPropertyMsg) msg;
            if (afMsg.getProperty() != null && afMsg.getProperty().size() > 0) {
                sb.append("<properties>");
                sb.append(toXmlString(afMsg.getProperty().toString()));
                sb.append("</properties>");
            }
        }
        sb.append("</message>");
        return sb.toString();
    }

    static String quoted(int integer) {
        return QUOTE + integer + QUOTE;
    }

    static String quoted(Object o) {
        if (o == null) {
            return QUOTE + QUOTE;
        }
        return QUOTE + toXmlString(o.toString()) + QUOTE;
    }
    /*
     * CLDC does not support s.replace(String, String).
     * Therefore, instead of s.replace("&", "&amp;"),
     * doing s.replace('&', '-'). Only to ensure that
     * the resulting XML can be processed later.
     */
    static String toXmlString(String s) {
        String t =  s.replace('&', '-')
        .replace('<', '[')
        .replace('>', ']');
        return t;
    }

}
