/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 *
 *
 * Changed by Magne Rasmussen for Telespor AS
 * Copyright applies to changes made to original file
 * 
 */

// Original copyright. Still applies to original work:
/*
 * Copyright (c) OSGi Alliance (2007, 2008). All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.coos.extender;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleEvent;

/**
 * The <code>BundleTrackerCustomizer</code> interface allows a
 * <code>BundleTracker</code> to customize the <code>Bundle</code>s that are
 * tracked. A <code>BundleTrackerCustomizer</code> is called when a bundle is
 * being added to a <code>BundleTracker</code>. The
 * <code>BundleTrackerCustomizer</code> can then return an object for the
 * tracked bundle. A <code>BundleTrackerCustomizer</code> is also called when a
 * tracked bundle is modified or has been removed from a
 * <code>BundleTracker</code>.
 * <p>
 * The methods in this interface may be called as the result of a
 * <code>BundleEvent</code> being received by a <code>BundleTracker</code>.
 * Since <code>BundleEvent</code>s are received synchronously by the
 * <code>BundleTracker</code>, it is highly recommended that implementations of
 * these methods do not alter bundle states while being synchronized on any
 * object.
 * <p>
 * The <code>BundleTracker</code> class is thread-safe. It does not call a
 * <code>BundleTrackerCustomizer</code> while holding any locks.
 * <code>BundleTrackerCustomizer</code> implementations must also be
 * thread-safe.
 * 
 * @ThreadSafe
 */
public interface BundleTrackerCustomizer<O> {
	/**
	 * A bundle is being added to the <code>BundleTracker</code>.
	 * <p>
	 * This method is called before a bundle which matched the search parameters
	 * of the <code>BundleTracker</code> is added to the
	 * <code>BundleTracker</code>. This method should return the object to be
	 * tracked for the specified <code>Bundle</code>. The returned object is
	 * stored in the <code>BundleTracker</code> and is available from the
	 * {@link BundleTracker#getObject(Bundle) getObject} method.
	 * 
	 * @param bundle
	 *            The <code>Bundle</code> being added to the
	 *            <code>BundleTracker</code>.
	 * @param event
	 *            The bundle event which caused this customizer method to be
	 *            called or <code>null</code> if there is no bundle event
	 *            associated with the call to this method.
	 * @return The object to be tracked for the specified <code>Bundle</code>
	 *         object or <code>null</code> if the specified <code>Bundle</code>
	 *         object should not be tracked.
	 */
	O addingBundle(Bundle bundle, BundleEvent event);

	/**
	 * A bundle tracked by the <code>BundleTracker</code> has been modified.
	 * <p>
	 * This method is called when a bundle being tracked by the
	 * <code>BundleTracker</code> has had its state modified.
	 * 
	 * @param bundle
	 *            The <code>Bundle</code> whose state has been modified.
	 * @param event
	 *            The bundle event which caused this customizer method to be
	 *            called or <code>null</code> if there is no bundle event
	 *            associated with the call to this method.
	 * @param object
	 *            The tracked object for the specified bundle.
	 */
	void modifiedBundle(Bundle bundle, BundleEvent event, O object);

	/**
	 * A bundle tracked by the <code>BundleTracker</code> has been removed.
	 * <p>
	 * This method is called after a bundle is no longer being tracked by the
	 * <code>BundleTracker</code>.
	 * 
	 * @param bundle
	 *            The <code>Bundle</code> that has been removed.
	 * @param event
	 *            The bundle event which caused this customizer method to be
	 *            called or <code>null</code> if there is no bundle event
	 *            associated with the call to this method.
	 * @param object
	 *            The tracked object for the specified bundle.
	 */
	void removedBundle(Bundle bundle, BundleEvent event, O object);
}
