/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */

package org.coos.extender;

import org.coos.messaging.util.Log;
import org.coos.messaging.util.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;

/**
 * @author Magne Rasmussen for Telespor AS
 */
public final class CoosExtender extends BundleTracker<CoosContainer> {
	private static final Log LOG = LogFactory.getLog(CoosExtender.class);

	private static final String CONFIG_DIR = "configDir";

	private static final String DEFAULT_CONFIG_DIR = "./coosConfig";

	private static final String COOS_CONTEXT = "Coos-Context";

	private static final String DEFAULT_COOS_CONTEXT = "/META-INF/coos";

	private final String configDir;

	public CoosExtender(final BundleContext context) {
		super(context, Bundle.ACTIVE, null);
		this.configDir = System.getProperty(CONFIG_DIR, DEFAULT_CONFIG_DIR);
		LOG.info("Coos configuration directory: " + this.configDir);
	}

	@Override
	public CoosContainer addingBundle(final Bundle bundle, final BundleEvent event) {
		if (isCoosBundle(bundle)) {
			final String coosContext = getCoosContext(bundle);
			final CoosContainer container = new CoosContainer(bundle.getBundleContext(), bundle, coosContext,
					this.configDir);
			container.start();
			return container;
		}
		// LOG.debug("Ignoring bundle " + bundle.getSymbolicName() +
		// " - not a Coos bundle");
		return null;
	}

	@Override
	public void modifiedBundle(final Bundle bundle, final BundleEvent event, final CoosContainer container) {
		// should not be called because we are only interested in the ACTIVE
		// state
	}

	@Override
	public void removedBundle(final Bundle bundle, final BundleEvent event, final CoosContainer container) {
		container.stop();
	}

	private boolean isCoosBundle(final Bundle bundle) {
		if (bundle.getHeaders().get(COOS_CONTEXT) != null) {
			LOG.debug("Bundle is a Coos bundle with context:" + bundle.getHeaders().get(COOS_CONTEXT));
			return true;
		}
		return bundle.getEntryPaths(DEFAULT_COOS_CONTEXT) != null;
	}

	private String getCoosContext(final Bundle bundle) {
		final String coosContext = (String) bundle.getHeaders().get(COOS_CONTEXT);
		if (coosContext == null) {
			return DEFAULT_COOS_CONTEXT;
		}
		return coosContext;
	}
}
