/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */

package org.coos.extender;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.coos.messaging.BaseCOContainer;
import org.coos.messaging.util.Log;
import org.coos.messaging.util.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;

/**
 * @author Magne Rasmussen for Telespor AS
 */
public abstract class OSGICOContainer extends BaseCOContainer {

    private static final Log LOG = LogFactory.getLog(OSGICOContainer.class);
    protected final BundleContext context;
    protected final Bundle bundle;
    protected final String contextDir;
    protected final File configDir;
    private CoosExtender extender;

    public OSGICOContainer(final BundleContext context, final Bundle bundle, final String contextDir,
            final String configDir, CoosExtender extender) {
        this.context = context;
        this.bundle = bundle;
        this.contextDir = contextDir;
        this.configDir = new File(configDir);
        this.extender=extender;
    }

    @Override
    public Class<?> loadClass(final String className) throws ClassNotFoundException {
        try {
            try {
                return this.bundle.loadClass(className);
            } catch (ClassNotFoundException e) {
                String timeout=(String) context.getBundle().getHeaders().get(CoosExtender.COOS_TIMEOUT);
                long startTime=System.currentTimeMillis();
                long duration=CoosExtender.DEFAULT_COOS_TIMEOUT;
                int n=0;
                if(timeout!=null)
                    duration=Long.parseLong(timeout);
                long stopTime=startTime+(duration*1000);
                while(System.currentTimeMillis()<stopTime){
                    try {
                        return extender.loadProcessorClass(className);
                    }
                    catch(ClassNotFoundException e2) {
                        LOG.info("Did not find class "+className+", Waited "+(n++)+" sec up to "+duration);
                        try {
                            Thread.sleep(1000);
                        } catch (InterruptedException e1) {
                            e1.printStackTrace();
                        }
                    }
                }
                throw new ClassNotFoundException("Extender didn't find class within timeout time.");
            }
        } catch (final IllegalStateException ex) {
            throw new ClassNotFoundException("BundleContext no longer valid", ex);
        }
    }

    @Override
    public InputStream getResource(final String resourceName) throws IOException {
        try {
            final URL resourceUrl = getResourceUrl(resourceName);
            if (resourceUrl == null) {
                return null;
            }
            final InputStream stream = resourceUrl.openStream();
            if (resourceName.endsWith(".xml")) {
                return substitute(stream);
            }
            return stream;
        } catch (final IllegalStateException ex) {
            throw new IOException("BundleContext no longer valid", ex);
        }
    }

    @Override
    public Object getObject(final String name) {
        if (BUNDLE_CONTEXT.equals(name)) {
            return this.context;
        }
        if(name != null){
            final ServiceReference reference = this.context.getServiceReference(name);
            if(reference != null){
                return this.context.getService(reference);
            }
        }

        return null;
    }

    private URL getBundleResourceUrl(final String resourceName) {
        return this.bundle.getResource(getBundleResourceName(resourceName));
    }

    private String getBundleResourceName(final String resourceName) {
        if (resourceName != null && resourceName.length() > 0 && resourceName.charAt(0) == '/') {
            return this.contextDir + resourceName;
        }
        return this.contextDir + "/" + resourceName;
    }

    private URL getResourceUrl(final String resourceName) {
        URL resourceUrl = getConfigResourceUrl(resourceName);
        if (resourceUrl == null) {
            resourceUrl = getBundleResourceUrl(resourceName);
        }
        return resourceUrl;
    }

    private URL getConfigResourceUrl(final String resourceName) {
        final File resource = new File(this.configDir + "/" + this.contextDir, resourceName);
        if (resource.exists() && resource.isFile()) {
            try {
                LOG.info("Container resource [" + getBundleResourceName(resourceName) + "] overridden by [" + resource
                        + "]");
                return resource.toURI().toURL();
            } catch (final MalformedURLException ex) {
                // do nothing
            }
        }
        return null;
    }
}
