/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.extender;

import java.util.Hashtable;

import org.coos.messaging.EndpointException;
import org.coos.messaging.util.Log;
import org.coos.messaging.util.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleException;


/**
 * @author Magne Rasmussen for Telespor AS
 */
public final class CoosExtender extends BundleTracker<OSGICOContainer> {
    private static final Log LOG = LogFactory.getLog(CoosExtender.class);

    private static final String CONFIG_DIR = "configDir";
    private static final String DEFAULT_CONFIG_DIR = "./coosConfig";
    private static final String COOS_PLUGIN_CONTEXT = "CoosPlugin-Context";
    private static final String COOS_CONTEXT = "Coos-Context";
    private static final String DEFAULT_COOS_CONTEXT = "/META-INF/coos";
    private static final String COOS_PROCESSOR = "Coos-Processor";
    public static final String COOS_TIMEOUT = "Coos-Timeout";
    public static final long DEFAULT_COOS_TIMEOUT=10;
    private final String configDir;
    private Hashtable<String,ClassLoader> procClassLoaders;

    public CoosExtender(final BundleContext context) {
        super(context, Bundle.ACTIVE, null);
        this.configDir = System.getProperty(CONFIG_DIR, DEFAULT_CONFIG_DIR);
        LOG.info("Coos configuration directory: " + this.configDir);
        procClassLoaders=new Hashtable<String,ClassLoader>();
        //TODO: Look for and add already added bundles?!
    }

    @Override
    public OSGICOContainer addingBundle(final Bundle bundle, final BundleEvent event) {
        LOG.debug("Got bundle starting: "+bundle.getSymbolicName());
        if(isProcessorProvider(bundle)) {
            LOG.debug("Bundle is ProcessorProvider.");
            final String [] providedProcessors = getCoosProcessors(bundle);
            for(int i=0;i<providedProcessors.length;i++) {
                LOG.debug("Adding classloader to class: '"+providedProcessors[i]+"'");
                procClassLoaders.put(providedProcessors[i], new BundleClassLoader(bundle));
            }
        }

        if (isCoosPluginBundle(bundle)) {
            final String coosContext = getCoosPluginContext(bundle);
            LOG.debug("Bundle is a COOS Plugin bundle with context:" + coosContext);
            final OSGICoosPluginContainer container = new OSGICoosPluginContainer(bundle.getBundleContext(), bundle, coosContext,
                    this.configDir,this);
            startContainer(container);
            return container;
        }
        else if(isCoosBundle(bundle)) {
            final String coosContext = getCoosContext(bundle);
            LOG.debug("Bundle is a COOS bundle with context:" + coosContext);
            final OSGICoosContainer container = new OSGICoosContainer(bundle.getBundleContext(), bundle, coosContext,
                    this.configDir,this);
            startContainer(container);
            return container;
        }
        LOG.debug("Ignoring bundle " + bundle.getSymbolicName() +
         " - not a Coos bundle");
        return null;
    }
    
    private void startContainer(final OSGICOContainer container) {
        Thread t=new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    container.start();
                } catch (Exception e) {
                    LOG.error("Failed to start container.", e);
                    try {
                        container.bundle.stop();
                    } catch (BundleException e1) {
                        LOG.warn("BundleException ignored while stopping.", e1);
                    }
                }
            }
        });
        t.start();
    }

    @Override
    public Bundle[] parseInitialBundleArray(Bundle[] bundles) {
        //TODO: Might need this
        return bundles;
    }

    private boolean isProcessorProvider(final Bundle bundle) {
        final String coosContext = (String) bundle.getHeaders().get(COOS_PROCESSOR);
        if(coosContext != null) {
            return true;
        }
        return false;
    }

    private ClassLoader getProcClassLoader(String processorClassName) {
        return (ClassLoader) procClassLoaders.get(processorClassName);
    }

    public Class<?> loadProcessorClass(String className) throws ClassNotFoundException {
        ClassLoader cl=getProcClassLoader(className);
        if(cl != null) {
            return cl.loadClass(className);
        }
        throw new ClassNotFoundException("ClassLoader from extender is null");
    }

    @Override
    public void modifiedBundle(final Bundle bundle, final BundleEvent event, final OSGICOContainer container) {
        // should not be called because we are only interested in the ACTIVE
        // state
    }

    @Override
    public void removedBundle(final Bundle bundle, final BundleEvent event, final OSGICOContainer container) {
        LOG.debug("bundle stopping: "+bundle.getSymbolicName());
        if(isProcessorProvider(bundle)) {
            LOG.debug("Bundle provides Processors");
            final String [] providedProcessors = getCoosProcessors(bundle);
            for(int i=0;i<providedProcessors.length;i++) {
                LOG.debug("Removing classloader to class: '"+providedProcessors[i]+"'");
                procClassLoaders.remove(providedProcessors[i]);
            }
        }
        try {
            container.stop();
            LOG.info("COOSs stopped");
        } catch (Exception e) {
            try {
                container.bundle.stop();
            } catch (BundleException e1) {
                LOG.warn("BundleException ignored while stopping.", e1);
            }
        }
    }

    private boolean isCoosPluginBundle(final Bundle bundle) {
        return isBundle(bundle, COOS_PLUGIN_CONTEXT);
    }

    private boolean isCoosBundle(Bundle bundle) {
        return isBundle(bundle,COOS_CONTEXT);
    }

    private boolean isBundle(Bundle bundle, String context) {
        if (bundle.getHeaders().get(context) != null) {
            return true;
        }
        return bundle.getEntryPaths(DEFAULT_COOS_CONTEXT) != null;
    }

    private String[] getCoosProcessors(final Bundle bundle) {
        String[] providedProcessors = ((String)bundle.getHeaders().get(COOS_PROCESSOR)).split("\\,");
        for(int i=0;i<providedProcessors.length;i++) {
            providedProcessors[i] = providedProcessors[i].trim();
        }
        return providedProcessors;
    }

    private String getCoosPluginContext(final Bundle bundle) {
        return getContext(bundle, COOS_PLUGIN_CONTEXT);
    }

    private String getCoosContext(final Bundle bundle) {
        return getContext(bundle, COOS_CONTEXT);
    }

    private String getContext(Bundle bundle, String context) {
        final String coosContext = (String) bundle.getHeaders().get(context);
        if (coosContext == null) {
            return DEFAULT_COOS_CONTEXT;
        }
        return coosContext;
    }
}
