/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.util.cldc.rms;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Vector;

import javax.microedition.rms.RecordEnumeration;
import javax.microedition.rms.RecordStore;
import javax.microedition.rms.RecordStoreException;
import javax.microedition.rms.RecordStoreNotOpenException;

import org.coos.util.serialize.AFSerializer;
import org.coos.util.serialize.StringHelper;

/**
 * Utility for doing IO to Mobile storage.
 * 
 * @author Geir Melby, Tellu AS
 * 
 */
public class RMSHelper {
	public synchronized static boolean checkIfTheStoreExist(String storeName) {
		if (storeName == null) {
			return false;
		}

		String[] stores = RecordStore.listRecordStores();

		if (stores != null) {
			// stores exist
			for (int i = 0; i < stores.length; i++) {
				String store = stores[i];

				if (store.equals(storeName)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Gets the record id of the last record stored
	 * 
	 * @param storeName
	 *            the name of the store
	 * @return -1 if store does not exist
	 */
	public synchronized static int getLastRecordId(String storeName) {
		if (!checkIfTheStoreExist(storeName)) {
			return -1;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);

			return store.getNextRecordID() - 1;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return -1;
	}

	/**
	 * Gets the record id of the last record stored
	 * 
	 * @param storeName
	 *            the name of the store
	 * @return -1 if store does not exist
	 */
	public synchronized static int getNextRecordId(String storeName) {
		if (!checkIfTheStoreExist(storeName)) {
			return -1;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);

			return store.getNextRecordID();
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return -1;
	}

	public synchronized static void deleteStore(String storeName) {
		if (!checkIfTheStoreExist(storeName)) {
			return;
		}

		try {
			RecordStore.deleteRecordStore(storeName);

			return;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return;
	}

	public synchronized static boolean loadFromStore(String storeName, int recordNumber, AFSerializer element) {
		if (element == null) {
			return false;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			byte[] data = store.getRecord(recordNumber);
			element.deSerialize(data, null);
			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	public synchronized static byte[] loadFromStore(String storeName, int recordNumber) {
		byte[] data = null;

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			data = store.getRecord(recordNumber);
			store.closeRecordStore();

			return data;
		} catch (RecordStoreException rse) {
			throw new NullPointerException("Ivalid record number <" + recordNumber + ">");
		}
	}

	public synchronized static int appendToStore(String storeName, byte[] data) {
		int no = 0;

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			// no = store.getNextRecordID();
			no = store.addRecord(data, 0, data.length);
			store.closeRecordStore();

			return no;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return no;
	}

	public synchronized static int saveToStore(String storeName, byte[] data, int recordNumber) {
		int no = 0;

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			no = store.getNextRecordID();

			if (recordNumber >= no) {
				no = store.addRecord(data, 0, data.length);
			} else {
				store.setRecord(recordNumber, data, 0, data.length);
				no = recordNumber;
			}

			store.closeRecordStore();

			return no;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return no;
	}

	public synchronized static Vector loadFromStore(String storeName) {
		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			RecordEnumeration re = store.enumerateRecords(null, null, false);
			int no = store.getNextRecordID();
			Vector v = new Vector(no);

			while (re.hasNextElement()) {
				byte[] data = re.nextRecord();
				v.addElement(data);
			}

			store.closeRecordStore();

			return v;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return null;
	}

	public synchronized static String[] loadStringsFromStore(String storeName) {
		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			RecordEnumeration re = store.enumerateRecords(null, null, false);
			int no = store.getNextRecordID();
			int i = 0;
			String[] sa = new String[no - 1];

			while (re.hasNextElement()) {
				byte[] data = re.nextRecord();
				ByteArrayInputStream bin = new ByteArrayInputStream(data);
				DataInputStream din = new DataInputStream(bin);
				String s = StringHelper.resurrect(din);
				sa[i++] = s;
			}

			store.closeRecordStore();

			return sa;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return null;
	}

	public synchronized static boolean saveStringsToStore(String storeName, String[] sa) {
		if (!checkIfTheStoreExist(storeName)) {
			return false;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			int j = 0;

			for (int i = sa.length; i > 0; i--) {
				String s = sa[i - 1];
				ByteArrayOutputStream bout = new ByteArrayOutputStream();
				DataOutputStream dout = new DataOutputStream(bout);
				dout.write(StringHelper.persist(s));
				dout.flush();

				byte[] data = bout.toByteArray();

				if (store.getNextRecordID() <= (j + 1)) {
					store.addRecord(data, 0, data.length);
				} else {
					store.setRecord(j + 1, data, 0, data.length);
				}

				j++;
			}

			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	public synchronized static boolean appendStringsToStore(String storeName, String[] sa) {
		if (!checkIfTheStoreExist(storeName)) {
			return false;
		}

		try {
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			DataOutputStream dout = new DataOutputStream(bout);
			RecordStore store = RecordStore.openRecordStore(storeName, false);

			for (int i = 0; i < sa.length; i++) {
				String s = sa[i];
				dout.write(StringHelper.persist(s));
				dout.flush();

				byte[] data = bout.toByteArray();
				store.addRecord(data, 0, data.length);
			}

			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	public synchronized static boolean saveToStore(String storeName, Vector v) {
		if (v == null) {
			return false;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			cleanStore(storeName);

			for (int i = 0; i < v.size(); i++) {
				AFSerializer el = (AFSerializer) v.elementAt(i);
				byte[] data = el.serialize();

				if (store.getNextRecordID() <= (i + 1)) {
					store.setRecord(i + 1, data, 0, data.length);
				} else {
					store.addRecord(data, 0, data.length);
				}
			}

			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	public synchronized static boolean appendToStore(String storeName, Vector v) {
		if (v == null) {
			return false;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);

			for (int i = 0; i < v.size(); i++) {
				AFSerializer el = (AFSerializer) v.elementAt(i);
				byte[] data = el.serialize();
				store.addRecord(data, 0, data.length);
			}

			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return false;
	}

	public synchronized static int appendToStore(String storeName, AFSerializer element) {
		if (element == null) {
			return 0;
		}

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, false);
			byte[] data = element.serialize();
			int recordNo = store.getNextRecordID();
			store.addRecord(data, 0, data.length);
			store.closeRecordStore();

			return recordNo;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return 0;
	}

	public synchronized static int getNumberOfRecords(String storeName) {
		int reti = 0;

		try {
			RecordStore store = RecordStore.openRecordStore(storeName, true, RecordStore.AUTHMODE_ANY, false);
			reti = store.getNumRecords();
			store.closeRecordStore();
		} catch (RecordStoreNotOpenException err) {
			err.printStackTrace();
		} catch (RecordStoreException err) {
			err.printStackTrace();
		}

		return reti;
	}

	public synchronized static boolean createStore(String storeName) {
		try {
			RecordStore store = RecordStore.openRecordStore(storeName, true, RecordStore.AUTHMODE_ANY, false);
			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return false;
	}

	public synchronized static long getLastUpdated(String storeName) {
		try {
			if (!checkIfTheStoreExist(storeName)) {
				return 0;
			}

			RecordStore store = RecordStore.openRecordStore(storeName, false, RecordStore.AUTHMODE_ANY, false);
			long lastModified = store.getLastModified();
			store.closeRecordStore();

			return lastModified;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return 0;
	}

	public synchronized static boolean cleanStore(String storeName) {
		try {
			RecordStore.deleteRecordStore(storeName);
			createStore(storeName);

			return true;
		} catch (RecordStoreException rse) {
		}

		return false;
	}

	/**
	 * Removes record index from the rms store starting with 0
	 * 
	 * @param storeName
	 * @param no
	 * @return
	 */

	public synchronized static boolean removeFirstElementFromStore(String storeName, int no) {
		try {
			if (!checkIfTheStoreExist(storeName)) {
				return false;
			}

			RecordStore store = RecordStore.openRecordStore(storeName, false);
			int lnr = 0;
			for (RecordEnumeration e = store.enumerateRecords(null, null, false); e.hasNextElement();) {
				if (lnr++ == no) {
					store.deleteRecord(e.nextRecordId());
					break;
				} else
					e.nextRecordId();

			}

			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return false;
	}

	public synchronized static boolean removeElementFromStore(String storeName, int no) {
		try {
			if (!checkIfTheStoreExist(storeName)) {
				return false;
			}

			RecordStore store = RecordStore.openRecordStore(storeName, false);

			store.deleteRecord(no);
			store.closeRecordStore();

			return true;
		} catch (RecordStoreException rse) {
			rse.printStackTrace();
		}

		return false;
	}

	/**
	 * Checks if the store has been updated during the priod specified in
	 * paramter period
	 * 
	 * @param storeName
	 *            the rms store name
	 * @param period
	 *            in seconds it is checked agains
	 * @return true if the store is up to date
	 */
	public synchronized static boolean checkIfStoreIsUptoDate(String storeName, String period) {
		long p = 0;

		if (period != null) {
			p = Integer.parseInt(period);
		}

		return checkIfStoreIsUptoDate(storeName, p);
	}

	/**
	 * Checks if the store has been updated during the priod specified in
	 * paramter period
	 * 
	 * @param storeName
	 *            the rms store name
	 * @param period
	 *            in seconds it is checked agains
	 * @return true if the store is up to date
	 */
	public synchronized static boolean checkIfStoreIsUptoDate(String storeName, long period) {
		long now = new Date().getTime();
		long lastUpdated = RMSHelper.getLastUpdated(storeName);
		System.out.println("Difference:" + ((now - lastUpdated) / 1000));

		if (lastUpdated > (now - (period * 1000))) {
			// RMS nees to be updated
			return true;
		} else {
			return false;
		}
	}
}
