/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.util.serialize;

import java.io.*;

/**
 * Use this class to serialize Vector's
 * 
 * 
 * 
 */
public class ArrayHelper {

	public static byte[] persist(Object object) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		persist(object, dout);

		return bout.toByteArray();

	}

	public static void persist(Object object, DataOutputStream dout) throws IOException {

		if (object == null) {
			dout.writeInt(-1);
		} else {
			if (object instanceof String[]) {
				// String Array helper
				String[] stringArray = (String[]) object;
				int n = stringArray.length;
				dout.writeInt(n);
				StringHelper.persist("StringArray", dout);

				for (int i = 0; i < n; i++) {
					String s = stringArray[i];
					dout.write(StringHelper.persist(s));
				}
			} else if (object instanceof byte[]) {
				// byte array helper
				byte[] byteArray = (byte[]) object;
				int n = byteArray.length;
				dout.writeInt(n);
				StringHelper.persist("ByteArray", dout);
				dout.write(byteArray, 0, n);
			} else if (object instanceof int[]) {
				// byte array helper
				int[] intArray = (int[]) object;
				int n = intArray.length;
				dout.writeInt(n);
				StringHelper.persist("IntArray", dout);

				for (int i = 0; i < intArray.length; i++) {
					int data = intArray[i];
					dout.writeInt(data);
				}
			} else if (object instanceof AFSerializer[]) {
				AFSerializer[] afArray = (AFSerializer[]) object;
				int n = afArray.length;
				dout.writeInt(n);
				StringHelper.persist("AFArray", dout);

				AFSerializer obj;
				for (int i = 0; i < n; i++) {
					obj = afArray[i];
					if (obj == null) {
						dout.writeUTF("null");
					} else {
						dout.writeUTF(obj.getClass().getName());
						byte[] data = obj.serialize();
						dout.writeInt(data.length);
						if (data.length > 0) {
							dout.write(data);
						}
					}
				}
			}
		}

		dout.flush();

	} // persist

	/**
	 * Resurrect array of String or byte
	 * 
	 * @param din
	 *            data input stream
	 * @return reference to array
	 * @throws IOException
	 */
	public static Object resurrect(DataInputStream din) throws IOException {
		Object result = null;
		int n = din.readInt();

		if (n != -1) {
			String s = StringHelper.resurrect(din);

			if (s.equals("StringArray")) {
				String[] stringArray = null;
				stringArray = new String[n];

				for (int i = 0; i < n; i++) {
					String sin = StringHelper.resurrect(din);
					stringArray[i] = sin;
				}

				result = stringArray;
			} else if (s.equals("ByteArray")) {
				byte[] byteArray = new byte[n];
				din.readFully(byteArray, 0, n);
				result = byteArray;
			} else if (s.equals("IntArray")) {
				int[] intArray = new int[n];

				for (int i = 0; i < intArray.length; i++) {
					intArray[i] = din.readInt();
				}

				result = intArray;

			} // else if (s.equals("null")) returns null

		}

		return result;
	} // resurrect

	/**
	 * Resurrect array of String or byte
	 * 
	 * @param din
	 *            data input stream
	 * @return reference to array
	 * @throws IOException
	 */
	public static Object resurrect(DataInputStream din, AFClassLoader loader) throws IOException {
		Object result = null;
		int n = din.readInt();

		if (n != -1) {
			String s = StringHelper.resurrect(din);

			if (s.equals("StringArray")) {
				String[] stringArray = null;
				stringArray = new String[n];

				for (int i = 0; i < n; i++) {
					String sin = StringHelper.resurrect(din);
					stringArray[i] = sin;
				}

				result = stringArray;
			} else if (s.equals("ByteArray")) {
				byte[] byteArray = new byte[n];
				din.readFully(byteArray, 0, n);
				result = byteArray;
			} else if (s.equals("IntArray")) {
				int[] intArray = new int[n];

				for (int i = 0; i < intArray.length; i++) {
					intArray[i] = din.readInt();
				}

				result = intArray;
			} else if (s.equals("AFArray")) {
				AFSerializer[] afArray = new AFSerializer[n];
				for (int i = 0; i < n; i++) {
					String cname = din.readUTF();
					if (cname.equals("null")) {
						afArray[i] = null;
					} else {
						int len = din.readInt();
						byte[] tmp = new byte[len];
						din.readFully(tmp);
						try {
							Class cl;
							AFSerializer obj;
							if (loader == null) {
								cl = Class.forName(cname);
								obj = (AFSerializer) cl.newInstance();
								obj.deSerialize(tmp, loader);
							} else {
								cl = loader.loadClass(cname);
								obj = (AFSerializer) cl.newInstance();
								obj.deSerialize(tmp, loader);
							}
							afArray[i] = obj;
						} catch (IOException e) {
							throw e;
						} catch (Exception e) {
							throw new IOException("Exception " + e.toString());
						}
					}
				}
				result = afArray;

			}

		}

		return result;
	} // resurrect
}
