/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.util.serialize;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Hashtable;
import java.util.Vector;

/**
 * User: Knut Eilif
 * 
 * 
 */
public class ObjectHelper {

	public static byte[] persist(Object object) throws Exception {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		persist(object, dout);
		return bout.toByteArray();

	}

	public static void persist(Object object, DataOutputStream dout) throws Exception {

		if (object instanceof Hashtable) {
			dout.writeByte(1);
			dout.write(HashtableHelper.persist((Hashtable) object));
		} else if (object instanceof String) {
			dout.writeByte(2);
			dout.write(StringHelper.persist((String) object));
		} else if (object instanceof Vector) {
			dout.writeByte(3);
			dout.write(VectorHelper.persist((Vector) object));
		} else if (object instanceof byte[] || object instanceof String[]
				|| object instanceof int[] || object instanceof AFSerializer[]) {
			dout.writeByte(4);
			dout.write(ArrayHelper.persist(object));
		} else if (object instanceof AFSerializer) {
			dout.writeByte(5);
			if (object == null) {
				dout.writeInt(-1);
			} else {

				dout.writeUTF(object.getClass().getName());
				byte[] data = ((AFSerializer) object).serialize();
				dout.writeInt(data.length);
				if (data.length > 0) {
					dout.write(data);
				}
			}
			dout.flush();

		} else if (object instanceof Integer) {
			dout.writeByte(6);
			IntegerHelper.persist((Integer) object, dout);
		} else {
			throw new Exception("Cannot serialize object of type " + object.getClass().getName());
		}
	}

	public static Object resurrect(DataInputStream din, AFClassLoader loader) throws Exception {
		byte type = din.readByte();
		switch (type) {
		case 1:
			return HashtableHelper.resurrect(din, loader);
		case 2:
			return StringHelper.resurrect(din);
		case 3:
			return VectorHelper.resurrect(din, loader);
		case 4:
			return ArrayHelper.resurrect(din, loader);
		case 5:
			String cname = din.readUTF();
			int len = din.readInt();
			byte[] tmp = new byte[len];
			din.readFully(tmp);

			Class cl;
			Object o;
			if (loader == null) {
				cl = Class.forName(cname);
				o = cl.newInstance();
				((AFSerializer) o).deSerialize(tmp, loader);
			} else {
				cl = loader.loadClass(cname);
				o = cl.newInstance();
				((AFSerializer) o).deSerialize(tmp, loader);
			}
			return o;
		case 6:
			return IntegerHelper.resurrect(din);
		default:
			throw new Exception("Can not resurrect object of type " + type);

		}
	}
}
