/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.util.macro;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.Map;

/**
 * MacroSubstituteReader class extends BufferedReader to substitue macros in
 * file.
 * 
 * @author Robert Bjarum, Tellu AS
 * 
 */
public class MacroSubstituteReader extends BufferedReader {
	private MacroSubstituter macroSubstituter;

	public MacroSubstituteReader(Reader in, Map<String, String> defaultProperties) {
		super(in);
		macroSubstituter = new MacroSubstituter(defaultProperties);
	}

	public MacroSubstituteReader(Reader in) {
		super(in);
		macroSubstituter = new MacroSubstituter();
	}

	@Override
	public int read() throws IOException {
		throw new IOException("read() not supported");
	}

	@Override
	public int read(char[] cbuf, int off, int len) throws IOException {
		throw new IOException("read(char[] cbuf, int off, int len) not supported");
	}

	@Override
	public String readLine() throws IOException {
		String s = super.readLine();

		if (s != null) {
			return macroSubstituter.process(s);
		}

		return s;
	}

	/**
	 * Read until EOF and scan for macro-definitions.
	 * 
	 * @return
	 * @throws IOException
	 */
	public Map<String, String> scanForMacrosOnly() throws IOException {
		macroSubstituter.setDryRun(true);

		String s = readLine();

		while (s != null) {
			s = readLine();
		}

		return macroSubstituter.getFoundMacros();
	}

	/**
	 * Read until EOF and substitute macro-definitions.
	 * 
	 * @return stream as string
	 * @throws IOException
	 */
	public String substituteMacros() throws IOException {
		StringBuilder sb = new StringBuilder();
		macroSubstituter.setDryRun(false);

		String s;

		while (true) {
			s = readLine();

			if (s == null) {
				break;
			}

			sb.append(s);
			sb.append("\n");
		}

		return sb.toString();
	}

	/**
	 * Get the set of macros found so far.
	 * 
	 * @return sorted set of key values
	 */
	public Map<String, String> getFoundMacros() {
		return macroSubstituter.getFoundMacros();
	}
}
