/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.util.serialize;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Use this class to serialize Vector's
 * 
 * 
 * 
 */
public class VectorHelper {

	public static final int NULL = 0;
	public static final int INTEGER = 1;
	public static final int STRING = 2;
	public static final int LONG = 3;
	public static final int BOOLEAN = 4;
	public static final int AFSERIALIZABLE = 5;
	public static final int VECTOR = 6;
	public static final int HASHTABLE = 7;
	public static final int BYTEARRAY = 8;
	public static final int FLOAT = 9;
	public static final int STRINGARRAY = 10;
	public static final int DOUBLE = 11;
	public static final int DOUBLEARRAY = 12;

	public static byte[] persist(Vector v) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		persist(v, dout);
		return bout.toByteArray();

	}

	public static void persist(Vector v, DataOutputStream dout) throws IOException {

		if (v == null) {
			dout.writeInt(-1);
		} else {
			int n = v.size();
			dout.writeInt(n);

			for (int i = 0; i < n; ++i) {
				Object o = v.elementAt(i);
				if (o instanceof String) {
					dout.writeByte(STRING);
					dout.writeUTF((String) o);
				} else if (o instanceof Integer) {
					dout.writeByte(INTEGER);
					dout.writeInt(((Integer) o).intValue());
				} else if (o instanceof Double) {
					dout.writeByte(DOUBLE);
					dout.writeDouble(((Double) o).doubleValue());
				} else if (o instanceof Long) {
					dout.writeByte(LONG);
					dout.writeLong(((Long) o).longValue());
				} else if (o instanceof Float) {
					dout.writeFloat(FLOAT);
					dout.writeFloat(((Float) o).floatValue());
				} else if (o instanceof Boolean) {
					dout.writeByte(BOOLEAN);
					dout.writeBoolean(((Boolean) o).booleanValue());
				} else if (o instanceof Vector) {
					dout.writeByte(VECTOR);
					VectorHelper.persist((Vector) o, dout);
				} else if (o instanceof Hashtable) {
					dout.writeByte(HASHTABLE);
					HashtableHelper.persist((Hashtable) o, dout);
				} else if (o instanceof AFSerializer) {
					dout.writeByte(AFSERIALIZABLE);
					dout.writeUTF(o.getClass().getName());
					byte[] data = ((AFSerializer) o).serialize();
					dout.writeInt(data.length);
					if (data.length > 0) {
						dout.write(data);
					}
				} else if (o instanceof byte[]) {
					dout.writeByte(BYTEARRAY);
					dout.writeInt(((byte[]) o).length);
					dout.write((byte[]) o);
				} else if (o instanceof String[]) {
					dout.writeByte(STRINGARRAY);
					String[] tmp = (String[]) o;
					dout.writeInt(tmp.length);
					for (int x = 0; x < tmp.length; x++) {
						dout.writeUTF(tmp[x]);
					}
				} else if (o instanceof double[]) {
					dout.writeByte(DOUBLEARRAY);
					double[] tmp = (double[]) o;
					dout.writeInt(tmp.length);
					for (int x = 0; x < tmp.length; x++) {
						dout.writeDouble(tmp[x]);
					}
				} else if (o == null) {

					dout.writeByte(NULL);
				} else {
					throw new IOException("Cannot persist " + "object of type " + o.getClass().getName());
				}
			}
		}
		dout.flush();
	} // persist

	// public static Vector resurrect(byte[] persisted) throws IOException {
	// ByteArrayInputStream bin = new ByteArrayInputStream(persisted);

	// DataInputStream din = new DataInputStream(bin);

	public static Vector resurrect(DataInputStream din, AFClassLoader loader) throws IOException {
		Vector v = new Vector();
		int n = din.readInt();
		if (n == -1) {
			v = null;
		} else {
			for (int i = 0; i < n; ++i) {
				int type = din.readByte();
				if (type == NULL) {
					v.addElement(null);
				} else if (type == INTEGER) {
					v.addElement(new Integer(din.readInt()));
				} else if (type == DOUBLE) {
					v.addElement(new Double(din.readDouble()));
				} else if (type == STRING) {
					v.addElement(din.readUTF());
				} else if (type == LONG) {
					v.addElement(new Long(din.readLong()));
				} else if (type == FLOAT) {
					v.addElement(new Float(din.readFloat()));
				} else if (type == BOOLEAN) {
					v.addElement(new Boolean(din.readBoolean()));
				} else if (type == VECTOR) {
					v.addElement(VectorHelper.resurrect(din, loader));
				} else if (type == HASHTABLE) {
					v.addElement(HashtableHelper.resurrect(din, loader));
				} else if (type == AFSERIALIZABLE) {
					String cname = din.readUTF();
					int len = din.readInt();
					byte[] tmp = new byte[len];
					din.readFully(tmp);
					try {
						Class cl;
						Object o;
						if (loader == null) {
							cl = Class.forName(cname);
							o = cl.newInstance();
							((AFSerializer) o).deSerialize(tmp, loader);
						} else {
							cl = loader.loadClass(cname);
							o = cl.newInstance();
							((AFSerializer) o).deSerialize(tmp, loader);
						}
						/*
						 * o = cl.newInstance(); ((AFSerializer)
						 * o).deSerialize(tmp);
						 */
						v.addElement(o);
					} catch (IOException e) {
						throw e;
					} catch (Exception e) {
						throw new IOException("Exception " + e.toString());
					}
				} else if (type == BYTEARRAY) {
					byte[] ba = new byte[din.readInt()];
					din.readFully(ba);
					v.addElement(ba);
				} else if (type == STRINGARRAY) {
					String[] st = new String[din.readInt()];
					for (int x = 0; x < st.length; x++)
						st[x] = din.readUTF();
					v.addElement(st);
				} else if (type == DOUBLEARRAY) {
					int length = din.readInt();
					double[] tmp = new double[length];
					for (int x = 0; x < length; x++) {
						tmp[x] = din.readDouble();
					}
					v.addElement(tmp);
				} else {
					throw new IOException("Unknown " + "type " + type);
				}
			}
		}
		return v;
	}
}
