package org.crazyyak.demo.jerseyspring.controllers;

import java.util.*;
import javax.annotation.Resource;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.crazyyak.demo.common.app.domain.*;
import org.crazyyak.dev.common.StringUtils;
import org.crazyyak.dev.common.fine.*;
import org.crazyyak.dev.domain.locality.LatLng;
import org.crazyyak.dev.jackson.test.FreeBird;
import org.joda.time.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Resource
@Component
@Path("/api")
public class JerseyApiController {

  private final AccountStore accountStore;

  @Autowired
  public JerseyApiController(AccountStore accountStore) {
    this.accountStore = accountStore;
  }

  @GET
  @Path("/account/{emailAddress}")
  @Consumes(MediaType.APPLICATION_JSON)
  @Produces(MediaType.APPLICATION_JSON)
  public Account getDefault(@PathParam("emailAddress") String emailAddress) {
    return accountStore.getByEmailAddress(emailAddress);
  }

  @GET
  @Path("/date/{date}/month-of-year")
  @Produces(MediaType.APPLICATION_JSON)
  public int getCurrentDate(@PathParam("date") LocalDate localDate) {
    return localDate.getMonthOfYear();
  }

  @GET
  @Path("/date/{date}/dates-in-month")
  @Produces(MediaType.APPLICATION_JSON)
  public List<LocalDate> getDaysInMonth(@PathParam("date") LocalDate localDate) {
    LocalDate firstDate = localDate.dayOfMonth().withMinimumValue();
    int max = localDate.dayOfMonth().withMaximumValue().getDayOfMonth();
    List<LocalDate> dates = new ArrayList<>();

    for (int i = 0; i < max; i++) {
      dates.add(firstDate.plusDays(i));
    }

    return dates;
  }

  @GET
  @Path("/plain-text")
  @Consumes(MediaType.TEXT_PLAIN)
  @Produces(MediaType.TEXT_PLAIN)
  public String getPlainText(@QueryParam("signature") String signature) {

    if (StringUtils.isBlank(signature)) {
      signature = "- One Crazy Yak";
    }

    return "This is plain text.\n" +
        "There really isn't much to it.\n\n" +
        "\t" +
        signature;
  }

  @GET
  @Path("/free-bird")
  @Consumes(MediaType.APPLICATION_JSON)
  @Produces(MediaType.APPLICATION_JSON)
  public FreeBird getFreeBird(@QueryParam("firstMessage") String firstMessage) {
    if (StringUtils.isBlank(firstMessage)) {
      firstMessage = "Hi, my name is Joe!";
    }
    return createFreeBird(firstMessage);
  }

  @POST
  @Path("/null-bird")
  @Produces(MediaType.APPLICATION_JSON)
  public FreeBird getNullBird() {
    return createFreeBird("This is a null bird.");
  }

  @POST
  @Path("/free-bird")
  @Consumes(MediaType.APPLICATION_JSON)
  @Produces(MediaType.APPLICATION_JSON)
  public FreeBird echoFreeBird(FreeBird freeBird) {
    return freeBird;
  }

  private FreeBird createFreeBird(String firstMessage) {
    return new FreeBird(
        "string-value", Long.MAX_VALUE, Integer.MIN_VALUE,
        new LatLng("37.3382030", "-119.7085060"),
        new org.crazyyak.dev.domain.money.Money("19.95"),
        org.joda.money.Money.parse("USD 99.99"),
        new LocalTime("12:32:13"), new LocalDate("1975-05-06"), new LocalDateTime("1975-05-06T12:32:13"),
        new TraitMap("first:yes", "empty-value:", "null-value", "last:ok"),
        FineMessage.withText("This message is fine!"),
        (FineMessageSetImpl) new FineMessageSetBuilder()
            .withText(firstMessage)
            .withAll("I'm Suzie", "999", new TraitMap("girl:true", "boy:false")).build());
  }
}
