package org.crazyyak.demo.webmvc.controllers;

import java.util.*;
import org.crazyyak.demo.common.app.domain.*;
import org.crazyyak.dev.common.StringUtils;
import org.crazyyak.dev.common.fine.*;
import org.crazyyak.dev.common.net.InetMediaType;
import org.crazyyak.dev.domain.locality.LatLng;
import org.crazyyak.dev.jackson.test.FreeBird;
import org.joda.time.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping(value = "/api")
public class MvcApiController {

  private final AccountStore accountStore;

  @Autowired
  public MvcApiController(AccountStore accountStore) {
    this.accountStore = accountStore;
  }

  @RequestMapping(value="/account/{emailAddress}")
  public Account getAccount(@PathVariable String emailAddress) {
    return accountStore.getByEmailAddress(emailAddress);
  }

  @RequestMapping("/date/{date}/month-of-year")
  public int getCurrentDate(@PathVariable LocalDate date) {
    return date.getMonthOfYear();
  }

  @RequestMapping("/date/{date}/dates-in-month")
  public List<LocalDate> getDaysInMonth(@PathVariable LocalDate date) {
    LocalDate firstDate = date.dayOfMonth().withMinimumValue();
    int max = date.dayOfMonth().withMaximumValue().getDayOfMonth();
    List<LocalDate> dates = new ArrayList<>();

    for (int i = 0; i < max; i++) {
      dates.add(firstDate.plusDays(i));
    }

    return dates;
  }

  @RequestMapping(value = "/plain-text", produces = InetMediaType.TEXT_PLAIN_VALUE)
  public String getPlainText(@RequestParam(required=false) String signature) {

    if (StringUtils.isBlank(signature)) {
      signature = "- One Crazy Yak";
    }

    return "This is plain text.\n" +
        "There really isn't much to it.\n\n" +
        "\t" +
        signature;
  }

  @RequestMapping("/free-bird")
  public FreeBird getFreeBird(@RequestParam(required=false) String firstMessage) {
    if (StringUtils.isBlank(firstMessage)) {
      firstMessage = "Hi, my name is Joe!";
    }
    return createFreeBird(firstMessage);
  }

  @RequestMapping(value = "/free-bird", method = RequestMethod.POST)
  public FreeBird echoFreeBird(@RequestBody FreeBird freeBird) {
    return freeBird;
  }

  @RequestMapping(value = "/null-bird", method = RequestMethod.POST)
  public FreeBird getNullBird() {
    return createFreeBird("This is a null bird.");
  }

  private FreeBird createFreeBird(String firstMessage) {
    return new FreeBird(
        "string-value", Long.MAX_VALUE, Integer.MIN_VALUE,
        new LatLng("37.3382030", "-119.7085060"),
        new org.crazyyak.dev.domain.money.Money("19.95"),
        org.joda.money.Money.parse("USD 99.99"),
        new LocalTime("12:32:13"), new LocalDate("1975-05-06"), new LocalDateTime("1975-05-06T12:32:13"),
        new TraitMap("first:yes", "empty-value:", "null-value", "last:ok"),
        FineMessage.withText("This message is fine!"),
        (FineMessageSetImpl) new FineMessageSetBuilder()
            .withText(firstMessage)
            .withAll("I'm Suzie", "999", new TraitMap("girl:true", "boy:false")).build());
  }
}
