


/*
 * Copyright 2014 Harlan Noonkester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.common.fine;

import java.util.*;

/**
 * User: harlann
 * Date: 8/23/13
 * Time: 9:34 PM
 */
public class ImmutableFineMessageSet implements FineMessageSet, Iterable<FineMessage> {

  private static final ImmutableFineMessageSet emptySet = new ImmutableFineMessageSet();

  private final List<FineMessage> messages;

  public static ImmutableFineMessageSet empty() {
    return emptySet;
  }

  public static Builder builder() {
    return new Builder();
  }

  public ImmutableFineMessageSet(FineMessage...messages) {
    this(Arrays.asList(messages));
  }

  public ImmutableFineMessageSet(Collection<FineMessage> messages) {
    // We want an immutable list - first make a defensive copy
    // so we are not effected by changes to the original and
    // then we initialize ourselves with unmodifiable list.
    List<FineMessage> mutableList = new ArrayList<>(messages);
    this.messages = Collections.unmodifiableList(mutableList);
  }

  @Override
  public boolean hasText(String text) {
    for (FineMessage message : messages) {
      if (message.isText(text)) {
        return true;
      }
    }
    return false;
  }

  @Override
  public boolean hasId(String id) {
    for (FineMessage message : messages) {
      if (message.isId(id)) {
        return true;
      }
    }
    return false;
  }

  @Override
  public FineMessage findFirstWithId(String id) {
    for (FineMessage message : messages) {
      if (message.isId(id)) {
        return message;
      }
    }
    return null;
  }

  @Override
  public List<FineMessage> findWithId(String id) {
    List<FineMessage> foundList = new ArrayList<>();
    for (FineMessage message : messages) {
      if (message.isId(id)) {
        foundList.add(message);
      }
    }
    return foundList;
  }

  @Override
  public FineMessage findFirstWithTrait(String trait) {
    for (FineMessage message : messages) {
      if (message.hasTrait(trait)) {
        return message;
      }
    }
    return null;
  }

  @Override
  public List<FineMessage> findWithTrait(String trait) {
    List<FineMessage> foundList = new ArrayList<>();
    for (FineMessage message : messages) {
      if (message.hasTrait(trait)) {
        foundList.add(message);
      }
    }
    return foundList;
  }

  @Override
  public String getDisplay() {
    StringBuilder sb = new StringBuilder();
    for (FineMessage message : messages) {
      sb.append(message.getDisplay());
      sb.append("\n");
    }
    if (sb.length() > 2) {
      sb.delete(sb.length() - 2, sb.length());
    }
    return sb.toString();
  }

  @Override
  public String getText() {
    StringBuilder sb = new StringBuilder();
    for (FineMessage message : messages) {
      sb.append(message.getDisplay());
      sb.append(" ");
    }
    if (sb.length() > 2) {
      sb.delete(sb.length() - 2, sb.length());
    }
    return sb.toString();
  }

  @Override
  public String getSummary() {
    if (isEmpty()) {
      return String.format("%s is empty.", getClass().getSimpleName());

    } else {
      StringBuilder text = new StringBuilder();
      for (FineMessage message : getMessages()) {
        text.append(message.getText());
        text.append(", ");
      }
      return text.substring(0, Math.max(text.length()-2, 0));
    }
  }

  @Override
  public boolean isEmpty() {
    return messages.isEmpty();
  }

  @Override
  public boolean isNotEmpty() {
    return !messages.isEmpty();
  }

  @Override
  public int size() {
    return messages.size();
  }

  @Override
  public Iterator<FineMessage> iterator() {
    return messages.iterator();
  }

  @Override
  public List<FineMessage> getMessages() {
    return messages;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    ImmutableFineMessageSet that = (ImmutableFineMessageSet) o;

    if (!messages.equals(that.messages)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return messages.hashCode();
  }

  @Override
  public String toString() {
    return getDisplay();
  }

  public static class Builder {

    private final List<FineMessage> messages = new ArrayList<>();

    public ImmutableFineMessageSet build() {
      return new ImmutableFineMessageSet(messages);
    }

    public Builder withMessage(FineMessage message) {
      if (message instanceof FineMessage) {
        messages.add((FineMessage)message);
      } else {
        this.messages.add(FineMessage.withAll(
            message.getText(),
            message.getId(),
            message.getTraitMap()));
      }
      return this;
    }

    public Builder withText(String text) {
      messages.add(FineMessage.withAll(text, null, TraitMap.empty()));
      return this;
    }

    public Builder withId(String text, String id) {
      messages.add(FineMessage.withAll(text, id, TraitMap.empty()));
      return this;
    }

    public Builder withTraits(String text, TraitMap traitMap) {
      messages.add(FineMessage.withAll(text, null, traitMap));
      return this;
    }

    public Builder withTraits(String text, Map<String, String> traitMapArg) {
      TraitMap traitMap = (traitMapArg != null && traitMapArg.size() > 0) ? new TraitMap(traitMapArg) : TraitMap.empty();
      messages.add(FineMessage.withAll(text, null, traitMap));
      return this;
    }

    public Builder withTraits(String text, String...traits) {
      TraitMap traitMap = (traits != null && traits.length > 0) ? new TraitMap(traits) : TraitMap.empty();
      messages.add(FineMessage.withAll(text, null, traitMap));
      return this;
    }

    public Builder withAll(String text, String id, TraitMap traitMap) {
      messages.add(FineMessage.withAll(text, id, traitMap));
      return this;
    }

    public Builder withSet(FineMessageSet fineMsgSet) {
      if (fineMsgSet instanceof ImmutableFineMessageSet) {
        return withSet(fineMsgSet);
      }

      for (FineMessage message : fineMsgSet.getMessages()) {
        this.withMessage(message);
      }

      return this;
    }

    public Builder withImmutableSet(ImmutableFineMessageSet fineMsgSet) {
      if (fineMsgSet != null) {
        this.messages.addAll(fineMsgSet.getMessages());
      }
      return this;
    }
  }
}
