/*
 * Copyright 2014 Harlan Noonkester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.common.fine;

import org.crazyyak.dev.common.BeanUtils;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

public class TraitMap  implements Serializable {

  private static final TraitMap empty = new TraitMap();
  public static TraitMap empty() {
    return empty;
  }

  private final SortedMap<String, String> traitMap;

  public TraitMap(Map<?, ?> givenMap) {
    SortedMap<String, String> localMap = new TreeMap<>();
    for (Map.Entry entry : givenMap.entrySet()) {

      Object key = entry.getKey();
      Object value = entry.getValue();

      if (key != null) {
        if (value == null) {
          localMap.put(key.toString(), null);
        } else {
          localMap.put(key.toString(), value.toString());
        }
      }
    }
    this.traitMap = Collections.unmodifiableSortedMap(localMap);
  }

  public TraitMap(String... traitStrings) {
    this(BeanUtils.toMap(traitStrings));
  }

  public Map<String, String> toMap() {
    return traitMap;
  }

  public boolean isEmpty() {
    return traitMap.isEmpty();
  }

  public int getSize() {
    return traitMap.size();
  }

  public boolean isNotEmpty() {
    return !traitMap.isEmpty();
  }

  public boolean hasTrait(String key) {
    return (key != null) && traitMap.containsKey(key);
  }

  public boolean hasValue(String key, String checkValue) {
    String value = traitMap.get(key);
    if (checkValue == null) {
      return value == null;
    } else {
      return checkValue.equals(value);
    }
  }

  public String getValue(String key) {
    return traitMap.get(key);
  }

  public String getText() {
    StringBuilder sb = new StringBuilder();
    for (Map.Entry<String, String> entry : traitMap.entrySet()) {
      sb.append(entry.getKey());
      sb.append(":");

      if (entry.getValue() == null) {
        sb.append("null");
      } else {
        sb.append("\"");
        sb.append(entry.getValue());
        sb.append("\"");
      }

      sb.append(", ");
    }
    if (sb.length() > 2) {
      sb.delete(sb.length() - 2, sb.length());
    }
    return sb.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    } else if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TraitMap that = (TraitMap) o;
    return traitMap.equals(that.traitMap);

  }

  @Override
  public int hashCode() {
    return traitMap.hashCode();
  }

  public String toString() {
    return getText();
  }
}
