/*
 * Copyright 2014 Harlan Noonkester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.common.fine;

import org.crazyyak.dev.common.BeanUtils;

import java.io.Serializable;
import java.util.*;

public class TraitMap implements Serializable {

  public static enum KeyType {
    NATURAL, UPPER, LOWER
  }

  private static final TraitMap empty = new TraitMap();
  public static TraitMap empty() {
    return empty;
  }

  private final KeyType keyType;
  private final Map<String, String> map;

  public TraitMap(Map<?, ?> givenMap) {
    this(null, givenMap);
  }

  public TraitMap(KeyType keyType, Map<?, ?> givenMap) {

    this.keyType = (keyType == null ? KeyType.UPPER : keyType);

    if (givenMap == null) {
      this.map = Collections.emptyMap();
      return;
    }

    SortedMap<String, String> localMap = new TreeMap<>();
    for (Map.Entry entry : givenMap.entrySet()) {

      if (entry.getKey() == null) {
        continue;
      }

      String key = fixCase(entry.getKey().toString());
      Object value = entry.getValue();

      if (value == null) {
        localMap.put(key, null);
      } else {
        localMap.put(key, value.toString());
      }
    }

    this.map = Collections.unmodifiableSortedMap(localMap);
  }

  public TraitMap(String...traitStrings) {
    this(null, traitStrings);
  }
  public TraitMap(KeyType keyType, String...traitStrings) {
    this(keyType, BeanUtils.toMap(traitStrings));
  }

  public TraitMap(Collection<String> traitStrings) {
    this(null, BeanUtils.toMap(traitStrings));
  }
  public TraitMap(KeyType keyType, Collection<String> traitStrings) {
    this(keyType, BeanUtils.toMap(traitStrings));
  }

  public KeyType getKeyType() {
    return keyType;
  }

  public TraitMap add(TraitMap traitMapArg) {
      Map<String, String> localMap = new HashMap<>(map);
      localMap.putAll(traitMapArg.getMap());
      return new TraitMap(keyType, localMap);
  }

  public TraitMap add(Map<?,?> traitMapArg) {
      if (traitMapArg == null || traitMapArg.isEmpty()) {
          return this;
      }
      Map<String, String> localMap = new HashMap<>(map);

      for (Map.Entry<?,?> entry : traitMapArg.entrySet()) {
        if (entry.getKey() == null) {
          // skip it...
        } else if (entry.getValue() == null) {
          localMap.put(entry.getKey().toString(), null);
        } else {
          localMap.put(entry.getKey().toString(), entry.getValue().toString());
        }
      }

      return new TraitMap(keyType, localMap);
  }

  public TraitMap remove(Collection<String> traits) {
      if (traits == null || traits.isEmpty()) {
          return this;
      }
      Map<String, String> localMap = new HashMap<>(map);
      for (String trait : traits) {
          localMap.remove(trait);
      }
      return new TraitMap(keyType, localMap);
  }

  public Map<String, String> getMap() {
  return map;
  }

  public boolean isEmpty() {
    return map.isEmpty();
  }

  public boolean isNotEmpty() {
    return !map.isEmpty();
  }

  public int getSize() {
    return map.size();
  }

  public boolean hasTrait(String key) {
    return (key != null) && map.containsKey(fixCase(key));
  }

  public boolean hasValue(String key, String checkValue) {
    if (key == null) return false;

    String value = map.get(fixCase(key));
    if (checkValue == null) {
      return value == null;
    } else {
      return checkValue.equals(value);
    }
  }

  public String getValue(String key) {
    return (key == null) ? null : map.get(fixCase(key));
  }

  public String getText() {
    StringBuilder sb = new StringBuilder();
    for (Map.Entry<String, String> entry : map.entrySet()) {
      sb.append(entry.getKey());
      sb.append(":");

      if (entry.getValue() == null) {
        sb.append("null");
      } else {
        sb.append("\"");
        sb.append(entry.getValue());
        sb.append("\"");
      }

      sb.append(", ");
    }
    if (sb.length() > 2) {
      sb.delete(sb.length() - 2, sb.length());
    }
    return sb.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    } else if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TraitMap that = (TraitMap) o;
    return map.equals(that.map);

  }

  @Override
  public int hashCode() {
    return map.hashCode();
  }

  public String toString() {
    return getText();
  }

  private String fixCase(String value) {

    if (value == null) {
      return null;

    } else if (KeyType.NATURAL.equals(keyType)) {
      return value;

    } else if (KeyType.UPPER.equals(keyType)) {
      return value.toUpperCase();

    } else if (KeyType.LOWER.equals(keyType)) {
      return value.toLowerCase();

    } else {
      String msg = String.format("The key type \"%s\" is not supported.", keyType);
      throw new UnsupportedOperationException(msg);
    }
  }
}
