package org.crazyyak.dev.couchace;

import com.couchace.core.api.*;
import com.couchace.core.api.meta.*;
import com.couchace.core.api.query.CouchViewQuery;
import com.couchace.core.api.request.*;
import com.couchace.core.api.response.*;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.crazyyak.dev.common.ReflectUtils;
import org.crazyyak.dev.common.exceptions.*;
import org.crazyyak.dev.common.exceptions.ApiException;

public abstract class DefaultCouchStore<T>{

  private final Class<T> entityType;
  protected final CouchDatabase database;

  public abstract String getDesignName();

  public DefaultCouchStore(CouchServer couchServer, String databaseName, Class<T> entityType) {
    this.entityType = ExceptionUtils.assertNotNull(entityType, "entityType");
    ExceptionUtils.assertNotNull(databaseName, "databaseName");
    this.database = couchServer.database(databaseName);

    if (database == null || database.exists() == false) {
      String msg = String.format("The database \"%s\" does not exist.", databaseName);
      throw new IllegalArgumentException(msg);
    }
  }

  public WriteResponse update(T entity) {
    WriteResponse response = database.put().entity(entity).execute();

    if (response.isCreated() == false) {
      java.lang.String msg = String.format("Unexpected DB response %s: %s\n", response.getStatusCode(), response.getErrorContent());
      throw ApiException.internalServerError(msg);
    }

    injectVersion(entity, response);

    return response;
  }

  public T getByDocumentId(String documentId) {
    GetEntityRequest<T> request = database.get().entity(entityType, documentId);
    GetEntityResponse<T> response = request.execute();
    return response.isEmpty() ? null : response.getFirstEntity();
  }

  public HeadResponse headByDocumentId(String documentId) {
    return database.head().id(documentId).execute();
  }

  public WriteResponse delete(T entity) {
    WriteResponse deleteResponse = database.delete().entity(entity).execute();
    return validateDelete(deleteResponse);
  }

  public WriteResponse deleteByDocumentId(String documentId, String revision) {
    WriteResponse deleteResponse = database.delete().document(documentId, revision).execute();
    return validateDelete(deleteResponse);
  }

  private WriteResponse validateDelete(WriteResponse deleteResponse) {
    if (deleteResponse.isOk() == false) {
      String msg = String.format("Unexpected DB response %s: %s\n", deleteResponse.getStatusCode(), deleteResponse.getErrorContent());
      throw ApiException.internalServerError(msg);
    }
    return deleteResponse;
  }

  public WriteResponse create(T entity) {
    WriteResponse putResponse = database.put().entity(entity).execute();

    if (putResponse.isCreated() == false) {
      String msg = String.format("Unexpected DB response %s: %s\n", putResponse.getStatusCode(), putResponse.getErrorContent());
      throw ApiException.internalServerError(msg);
    }

    injectVersion(entity, putResponse);

    return putResponse;
  }

  protected List<T> getEntities(String viewName, String... keyValues) {
    return getEntities(viewName, (Object[])keyValues);
  }

  protected List<T> getEntities(String viewName, Object[] keyValues) {
    return getEntities(viewName, Arrays.asList(keyValues));
  }

  protected List<T> getEntities(String viewName, Collection<?> keyValues) {
    GetEntityResponse<T> response = getEntityResponse(viewName, keyValues);
    return response.getEntityList();
  }

  protected GetEntityResponse<T> getEntityResponse(String viewName, Collection<?> keyValues) {

    CouchViewQuery couchViewQuery = CouchViewQuery
        .builder(getDesignName(), viewName)
        .key(keyValues.toArray())
        .includeDocs(true)
        .build();

    GetEntityResponse<T> response = database.get().entity(entityType, couchViewQuery).execute();

    if (response.isNotFound()) {
      String msg = String.format("The view \"_design/%s:%s\" does not exist.", getDesignName(), viewName);
      throw ApiException.internalServerError(msg);

    } else if (response.isOk() == false) {
      throw ApiException.internalServerError(response.getErrorReason());
    }

    return response;
  }

  protected void injectVersion(T entity, WriteResponse response) {
    String version = response.getDocumentRevision();

    EntityMeta<T> meta = new AnnotationMetaBuilder().buildEntityMeta(entityType);
    String property = meta.getRevisionName();
    Field field = ReflectUtils.getField(entity.getClass(), property);
    ReflectUtils.setPropertyValue(entity, field, version);
  }

  public Class<T> getEntityType() {
    return entityType;
  }

  public CouchDatabase getDatabase() {
    return database;
  }
}
