/*
 * Copyright 2012 Jacob D Parr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.domain.translation.jackson;

import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.InjectableValues;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * Custom InjectableValues implementation which will return null for any value which is not found.
 */
public class NoFailInjectableValues extends InjectableValues implements Serializable {
  private static final long serialVersionUID = 1L;

  protected final Map<String, Object> valueMap;

  public NoFailInjectableValues() {
    valueMap = new HashMap<String, Object>();
  }

  public void addValue(String key, Object value) {
    valueMap.put(key, value);
  }

  public void addValue(Class<?> classKey, Object value) {
    valueMap.put(classKey.getName(), value);
  }

  @Override
  public Object findInjectableValue(Object valueId,
                                    DeserializationContext ctxt,
                                    BeanProperty forProperty,
                                    Object beanInstance) {

    if (!(valueId instanceof String)) {
      String type = (valueId == null) ? "[null]" : valueId.getClass().getName();
      throw new IllegalArgumentException("Unrecognized inject value id type (" + type + "), expecting String");
    }
    String key = (String) valueId;
    return valueMap.get(key);
  }
}