package org.crazyyak.dev.domain.embedded;

import java.io.File;
import java.util.Arrays;
import org.crazyyak.dev.common.StringUtils;

public class EmbeddedArgs {

  private final String webAppRoot;
  private final EmbeddedAction action;

  private String contextPath = "";
  private String workingDirectory;

  private int shutdownPort = 8005;
  private int shutdownTimeout = 120;

  private int httpPort = 8080;
  private int securePort;

  private int ajpHttpPort;
  private int ajpSecurePort;

  private String keystoreFile;
  private String keystorePass;

  private String truststoreFile;
  private String truststorePass;

  private String keyAlias;

  private boolean autoShutDown;

  public EmbeddedArgs(String... args) {
    if (args.length % 2 != 0) {
      String msg = String.format("Expected an even number of arguments, not %s: %s", args.length, Arrays.asList(args));
      throw new IllegalArgumentException(msg);
    }

    String action = null;
    String webAppRoot = null;

    for (int i = 0; i < args.length; i += 2) {
      String key = args[i];
      String value = args[i+1];

      if ("webAppRoot".equalsIgnoreCase(key)) {
        webAppRoot = value;
      } else if ("action".equals(key)) {
        action = value;
      } else {
        parseArgs(key, value);
      }
    }

    this.webAppRoot = webAppRoot;

    try {
      this.action = EmbeddedAction.valueOf(action);
    } catch (NullPointerException e) {
      throw new IllegalArgumentException("The argument \"action\" must be specified");
    } catch (IllegalArgumentException e) {
      String msg = String.format("The value \"%s\" is not a valid value for the argument \"action\"", action);
      throw new IllegalArgumentException(msg);
    }
  }

  public String getErrorMessage() {
    if (EmbeddedAction.status == action) {

      if (shutdownPort == 0) {
        return "The argument \"shutdownPort\" must be specified.";
      }

    } else if (EmbeddedAction.stop == action) {

      if (shutdownPort == 0) {
        return "The argument \"shutdownPort\" must be specified.";
      } else if (shutdownTimeout == 0) {
        return "The argument \"shutdownTimeout\" must be specified.";
      }

    } else if (EmbeddedAction.start == action) {

      if (StringUtils.isBlank(webAppRoot)) {
        return "The argument \"webAppRoot\" must be specified.";
      } else if (StringUtils.isBlank(workingDirectory)) {
        return "The argument \"workingDirectory\" must be specified.";
      } else if (shutdownTimeout == 0) {
        return "The argument \"shutdownTimeout\" must be specified.";
      }
    }
    return null;
  }

  public static void printUsage() {
    System.out.println("Arguments: action status");
    System.out.println("           shutdownPort int (default 8005)\n");

    System.out.println("Arguments: action stop");
    System.out.println("           shutdownPort int (default 8005)");
    System.out.println("           shutdownTimeout int (default 120)\n");

    System.out.println("Arguments: action start");
    System.out.println("           webAppRoot \"{path}\"");
    System.out.println("           [contextPath \"{path}\"]");
    System.out.println("           workingDirectory \"{path}\"");
    System.out.println("           autoShutDown boolean (default false)");
    System.out.println("           shutdownPort int (default 8005)");
    System.out.println("           shutdownTimeout int (default 120)");
    System.out.println("           [httpPort int (default 8080)]");
    System.out.println("           [securePort int]");
    System.out.println("           [ajpHttpPort int]");
    System.out.println("           [ajpSecurePort int]");
    System.out.println("           [keystoreFile \"{path}\" keystorePass string]");
    System.out.println("           [truststoreFile \"{path}\" truststorePass string]");
    System.out.println("           [keyAlias string]");
  }

  private Object parseArgs(String key, String value) {
    try {
      if (key.equalsIgnoreCase("contextPath")) {
        return contextPath = value;
      } else if (key.equalsIgnoreCase("workingDirectory")) {
        return workingDirectory = value;
      } else if (key.equalsIgnoreCase("shutdownPort")) {
        return shutdownPort = Math.max(0, Integer.valueOf(value));
      } else if (key.equalsIgnoreCase("shutdownTimeout")) {
        return shutdownTimeout = Math.max(0, Integer.valueOf(value));
      } else if (key.equalsIgnoreCase("httpPort")) {
        return httpPort = Integer.valueOf(value);
      } else if (key.equalsIgnoreCase("securePort")) {
        return securePort = Integer.valueOf(value);
      } else if (key.equalsIgnoreCase("ajpHttpPort")) {
        return ajpHttpPort = Integer.valueOf(value);
      } else if (key.equalsIgnoreCase("ajpSecurePort")) {
        return ajpSecurePort = Integer.valueOf(value);
      } else if (key.equalsIgnoreCase("keystoreFile")) {
        return keystoreFile = value;
      } else if (key.equalsIgnoreCase("keystorePass")) {
        return keystorePass = value;
      } else if (key.equalsIgnoreCase("truststoreFile")) {
        return truststoreFile = value;
      } else if (key.equalsIgnoreCase("truststorePass")) {
        return truststorePass = value;
      } else if (key.equalsIgnoreCase("keyAlias")) {
        return keyAlias = value;
      } else if (key.equalsIgnoreCase("autoShutDown")) {
        return autoShutDown = Boolean.valueOf(value);
      } else {
        String msg = String.format("The argument \"%s\" (%s) is not supported.", key, value);
        throw new IllegalArgumentException(msg);
      }
    } catch (NumberFormatException e) {
      String msg = String.format("Unable to parse the number \"%s\" for the argument \"%s\".", value, key);
      throw new IllegalArgumentException(msg);
    }
  }

  public EmbeddedArgs(EmbeddedAction action, String webAppRoot) {
    this.action = action;
    this.webAppRoot = webAppRoot;
  }
  public EmbeddedArgs(EmbeddedAction action, File webAppRoot) {
    this.action = action;
    this.webAppRoot = webAppRoot.getAbsolutePath();
  }

  public EmbeddedAction getAction() {
    return action;
  }

  public String getContextPath() {
    return contextPath;
  }

  public void setContextPath(String contextPath) {
    this.contextPath = contextPath;
  }

  public String getWebAppRoot() {
    return webAppRoot;
  }

  public String getWorkingDirectory() {
    return workingDirectory;
  }

  public void setWorkingDirectory(String workingDirectory) {
    this.workingDirectory = workingDirectory;
  }
  public void setWorkingDirectory(File workingDirectory) {
    this.workingDirectory = workingDirectory.getAbsolutePath();
  }

  public int getShutdownPort() {
    return shutdownPort;
  }

  public void setShutdownPort(int shutdownPort) {
    this.shutdownPort = Math.max(0, shutdownPort);
  }

  public int getShutdownTimeout() {
    return shutdownTimeout;
  }

  public void setShutdownTimeout(int shutdownTimeout) {
    this.shutdownTimeout = Math.max(0, shutdownTimeout);
  }

  public int getHttpPort() {
    return httpPort;
  }

  public void setHttpPort(int httpPort) {
    this.httpPort = httpPort;
  }

  public int getSecurePort() {
    return securePort;
  }

  public void setSecurePort(int securePort) {
    this.securePort = securePort;
  }

  public int getAjpHttpPort() {
    return ajpHttpPort;
  }

  public void setAjpHttpPort(int ajpHttpPort) {
    this.ajpHttpPort = ajpHttpPort;
  }

  public int getAjpSecurePort() {
    return ajpSecurePort;
  }

  public void setAjpSecurePort(int ajpSecurePort) {
    this.ajpSecurePort = ajpSecurePort;
  }

  public String getKeystoreFile() {
    return keystoreFile;
  }

  public void setKeystoreFile(String keystoreFile) {
    this.keystoreFile = keystoreFile;
  }

  public String getKeystorePass() {
    return keystorePass;
  }

  public void setKeystorePass(String keystorePass) {
    this.keystorePass = keystorePass;
  }

  public String getTruststoreFile() {
    return truststoreFile;
  }

  public void setTruststoreFile(String truststoreFile) {
    this.truststoreFile = truststoreFile;
  }

  public String getTruststorePass() {
    return truststorePass;
  }

  public void setTruststorePass(String truststorePass) {
    this.truststorePass = truststorePass;
  }

  public String getKeyAlias() {
    return keyAlias;
  }

  public void setKeyAlias(String keyAlias) {
    this.keyAlias = keyAlias;
  }

  public boolean isAutoShutDown() {
    return autoShutDown;
  }

  public void setAutoShutDown(boolean autoShutDown) {
    this.autoShutDown = autoShutDown;
  }
}
