/*
 * Copyright 2012 Jacob D Parr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.domain.translation.jackson;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.datatype.joda.JodaModule;

import java.util.*;

public class YakJacksonObjectMapper extends ObjectMapper {

  private final Collection<? extends Module> modules;
  private final Collection<? extends YakJacksonInjectable> injectables;

  public YakJacksonObjectMapper() {
    this(Arrays.asList(
        new JodaModule(),
        new YakJacksonModule()),
        Collections.<YakJacksonInjectable>emptyList());
  }

  public YakJacksonObjectMapper(Collection<? extends Module> modules, Collection<? extends YakJacksonInjectable> injectables) {

    this.modules = modules;
    this.injectables = injectables;

    // Stops LocalDateTime from being rendered as array.
    configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);

    // Pretty printing
    configure(SerializationFeature.INDENT_OUTPUT, true);

    // Empty strings to null
    configure(DeserializationFeature.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT, false);

    // Allows for extra data.
    configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, true);

    // Register Joda and Dev modules
    for (Module module : modules) {
      registerModule(module);
    }

    YakJacksonInjectableValues injectableValues = new YakJacksonInjectableValues();
    for (YakJacksonInjectable injectable : injectables) {
      injectableValues.addValue(injectable.getKey(), injectable.getValue());
    }
    setInjectableValues(injectableValues);

    setVisibilityChecker(this.getSerializationConfig().getDefaultVisibilityChecker()
        .withFieldVisibility(JsonAutoDetect.Visibility.ANY)
        .withGetterVisibility(JsonAutoDetect.Visibility.ANY)
        .withSetterVisibility(JsonAutoDetect.Visibility.ANY)
        .withCreatorVisibility(JsonAutoDetect.Visibility.ANY));
  }

  @Override
  public ObjectMapper copy() {
    _checkInvalidCopy(YakJacksonObjectMapper.class);
    return new YakJacksonObjectMapper(modules, injectables);
  }
}
