package org.crazyyak.dev.embedded;

import java.io.OutputStream;
import java.net.Socket;
import org.apache.catalina.connector.Connector;
import org.apache.catalina.startup.Tomcat;
import org.apache.coyote.http11.Http11NioProtocol;
import org.crazyyak.dev.common.StringUtils;

public class EmbeddedTomcat {

  public static void main(String...args) {
    try {

      EmbeddedArgs embeddedArgs = new EmbeddedArgs(args);

      if (embeddedArgs.getErrorMessage() != null) {
        System.out.println(embeddedArgs.getErrorMessage());
        embeddedArgs.printUsage();
        System.out.flush();
        return;
      }

      EmbeddedTomcat embedded = new EmbeddedTomcat(embeddedArgs);
      embedded.start(true, embeddedArgs.isAutoShutDown());

    } catch (IllegalArgumentException e) {
      System.out.println(e.getMessage());
      EmbeddedArgs.printUsage();
      System.exit(0);

    } catch (Exception e) {
      e.printStackTrace();
      System.exit(0);
    }
  }

  private final int httpPort;
  private final int shutdownPort;

  private final String contextPath;
  private final String webAppRoot;

  private final Tomcat tomcat = new Tomcat();

  public EmbeddedTomcat(EmbeddedArgs args) throws Exception {

    if (args.getErrorMessage() != null) {
      throw new IllegalArgumentException(args.getErrorMessage());
    }

    // The shutdown port.
    tomcat.getServer().setPort(shutdownPort = args.getShutDownPort());

    // The location of the web app.
    tomcat.addWebapp(contextPath = args.getContextPath(), webAppRoot = args.getWebAppRoot());

    // The location of the work directory.
    tomcat.setBaseDir(args.getWorkingDirectory());

    configureDefaultConnector(httpPort = args.getHttpPort(), args.getSecurePort());

    if (args.getSecurePort() > 0) {
      Connector connector = createSslConnector(args);
      tomcat.getService().addConnector(connector);
    }

    if (args.getAjpHttpPort() > 0) {
      Connector connector = createAjpConnector(args.getAjpHttpPort(), args.getSecurePort());
      tomcat.getService().addConnector(connector);
    }

    if (args.getAjpSecurePort() > 0) {
      Connector connector = createSecureAjpConnector(args.getAjpSecurePort());
      tomcat.getService().addConnector(connector);
    }
  }

  private void configureDefaultConnector(int port, int securePort) {
    System.out.println("Creating default connector on port " + port);

    tomcat.setPort(port);

    if (securePort > 0) {
      tomcat.getConnector().setRedirectPort(securePort);
    }
  }

  private Connector createSslConnector(EmbeddedArgs args) {
    System.out.println("Creating SSL connector on port " + args.getSecurePort());

    Connector connector = new Connector(Http11NioProtocol.class.getName());
    Http11NioProtocol protocol = (Http11NioProtocol)connector.getProtocolHandler();

    connector.setScheme("https");
    connector.setSecure(true);
    connector.setPort(args.getSecurePort());
    protocol.setSSLEnabled(true);

    if (StringUtils.isNotBlank(args.getKeystoreFile())) {
      protocol.setKeystoreFile(args.getKeystoreFile());
    }
    if (StringUtils.isNotBlank(args.getKeystorePass())) {
      protocol.setKeystorePass(args.getKeystorePass());
    }

    if (StringUtils.isNotBlank(args.getTruststoreFile())) {
      protocol.setTruststoreFile(args.getTruststoreFile());
    }
    if (StringUtils.isNotBlank(args.getTruststorePass())) {
      protocol.setTruststorePass(args.getTruststorePass());
    }

    if (StringUtils.isNotBlank(args.getKeyAlias())) {
      protocol.setKeyAlias(args.getKeyAlias());
    }

    return connector;
  }

  private Connector createAjpConnector(int port, int securePort){
    System.out.println("Creating standard AJP connector on port " + port);

    Connector connector = new Connector(org.apache.coyote.ajp.AjpNioProtocol.class.getName());
    connector.setPort(port);
    connector.setScheme("http");
    connector.setSecure(false);

    if (securePort > 0) {
      connector.setRedirectPort(securePort);
    }

    return connector;
  }

  private Connector createSecureAjpConnector(int securePort){
    System.out.println("Creating secured AJP connector on port " + securePort);

    Connector connector = new Connector(org.apache.coyote.ajp.AjpNioProtocol.class.getName());
    connector.setPort(securePort);
    connector.setScheme("https");
    connector.setSecure(true);
    return connector;
  }

  public void start(boolean keepAlive, boolean autoShutDown) throws Exception {
    EmbeddedStatus status = getStatus();

    if (status.isUp()) {
      if (autoShutDown) {
        shutdown();
        while (getStatus().isUp()) {
          /* keep waiting */
        }

      } else {
        System.out.println("Embedded tomcat is already running.");
        return;
      }
    }

    System.out.println(String.format("Starting embedded tomcat on http://localhost:%s/%s using %s", httpPort, contextPath, webAppRoot));

    tomcat.start();

    if (keepAlive) {
      tomcat.getServer().await();
    }
  }

  public void shutdown() throws Exception {
    System.out.println("Shutting down embedded tomcat.");

    try(Socket localSocket = new Socket("localhost", shutdownPort)) {
      try(OutputStream outStream = localSocket.getOutputStream()) {
        outStream.write("SHUTDOWN".getBytes());
        outStream.flush();
      }
    }
  }

  public EmbeddedStatus getStatus() throws Exception {
    try(Socket ignored = new Socket("localhost", shutdownPort)) {
      return EmbeddedStatus.up;
    } catch (Throwable e) {
      return EmbeddedStatus.down;
    }
  }
}
