/*
 * Copyright 2014 Harlan Noonkester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.jackson;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.crazyyak.dev.common.exceptions.ApiException;
import org.crazyyak.dev.common.json.JsonTranslator;

import java.io.IOException;
import java.util.Collection;

/**
 * User: harlann
 * Date: 8/23/13
 * Time: 11:13 PM
 */
public class YakJacksonTranslator implements JsonTranslator {

  private final ObjectMapper jacksonMapper;


  public YakJacksonTranslator() {
    this(new YakJacksonObjectMapper());
  }

  public YakJacksonTranslator(Collection<? extends YakJacksonInjectable> injectables) {
    this(new YakJacksonObjectMapper(injectables));
  }

  public YakJacksonTranslator(Collection<? extends Module> modules, Collection<? extends YakJacksonInjectable> injectables) {
    this(new YakJacksonObjectMapper(modules, injectables));
  }

    public YakJacksonTranslator(ObjectMapper jacksonMapper) {
    this.jacksonMapper = jacksonMapper;
  }

  @Override
  public String toJson(Object domain) {
    try {
      return jacksonMapper.writeValueAsString(domain).replaceAll("\r", "");

    } catch (JsonProcessingException e) {
      throw ApiException.internalServerError(e);
    }
  }

  @Override
  public <T> T fromJson(Class<T> type, String json, Class<?>... parameterTypes) {

    if (parameterTypes != null && parameterTypes.length > 0) {
      try {
        JavaType javaType = jacksonMapper.getTypeFactory().constructParametricType(type, parameterTypes);
        return jacksonMapper.readValue(json, javaType);
      } catch (IOException e) {
        throw ApiException.internalServerError(e);
      }
    } else {
      try {
        return jacksonMapper.readValue(json, type);

      } catch (IOException e) {
        throw ApiException.internalServerError(e);
      }
    }
  }

}
