package org.crazyyak.rest.client;

import java.io.*;
import java.util.*;
import javax.ws.rs.client.*;
import javax.ws.rs.core.*;
import org.crazyyak.dev.common.*;
import org.crazyyak.dev.common.exceptions.ApiException;
import org.crazyyak.dev.common.json.JsonTranslator;
import org.crazyyak.dev.common.net.HttpStatusCode;
import org.glassfish.jersey.client.authentication.HttpAuthenticationFeature;
import org.glassfish.jersey.uri.internal.JerseyUriBuilder;

public class SimpleRestClient {


  public static final Map<String,Object> EMPTY_QUERY = Collections.emptyMap();

  protected final JsonTranslator translator;

  private final String apiUrl;
  private final String userName;
  private final String password;

  public SimpleRestClient(JsonTranslator translator, String apiUrl, String userName, String password) {
    this.apiUrl = apiUrl;
    this.userName = userName;
    this.password = password;
    this.translator = translator;
  }



  public void post(String subUrl) {
    post(null, subUrl, null);
  }

  public void post(String subUrl, Object entity) {
    post(null, subUrl, entity);
  }

  public <T> T post(Class returnType, String subUrl) {
    return post(returnType, subUrl, null);
  }

  public <T> T post(Class returnType, String subUrl, Object entity) {

    Invocation.Builder builder = builder(subUrl, Collections.<String,Object>emptyMap(), MediaType.APPLICATION_JSON);

    String json = (entity == null) ? null : translator.toJson(entity);
    Response response = builder.post(Entity.entity(json, MediaType.APPLICATION_JSON_TYPE));

    assertResponse(response.getStatus());
    json = response.readEntity(String.class);

    // noinspection unchecked
    return (returnType == null) ? null : (T)translator.fromJson(returnType, json);
  }



  public <T> T get(Class returnType, String subUrl, String... queryStrings) {
    Map<String,Object> queryMap = new HashMap<>();
    queryMap.putAll(toMap(queryStrings));
    return get(returnType, subUrl, queryMap, "application/json");
  }

  public <T> T get(Class returnType, String subUrl, Map<String, Object> queryMap) {
    return get(returnType, subUrl, queryMap, "application/json");
  }

  public byte[] getBytes(String subUrl, Map<String, Object> queryMap, String...acceptedResponseTypes) throws IOException {
    Invocation.Builder builder = builder(subUrl, queryMap, acceptedResponseTypes);
    Response response = builder.get(Response.class);

    assertResponse(response.getStatus());
    InputStream in = (InputStream)response.getEntity();
    return IoUtils.toBytes(in);
  }

  public <T> T get(Class returnType, String subUrl, Map<String, Object> queryMap, String...acceptedResponseTypes) {

    Invocation.Builder builder = builder(subUrl, queryMap, acceptedResponseTypes);
    Response response = builder.get(Response.class);

    assertResponse(response.getStatus());
    String content = response.readEntity(String.class);

    Object retValue;

    if (returnType == null) {
      // return object not expected
      return null;

    } else if (returnType.isPrimitive() || (returnType.isArray() && returnType.getComponentType().isPrimitive())) {
      // An array of primitive types
      retValue = content;

    } else if (String.class.equals(returnType)) {
      // A simple string
      retValue = content.replaceAll("\r", "");

    } else {
      // A json object to be translated.
      // noinspection unchecked
      return (T)translator.fromJson(returnType, content);
    }

    // noinspection unchecked
    return (T)retValue;
  }



  public String getApiUrl() {
    return apiUrl;
  }

  public String getUserName() {
    return userName;
  }

  public String getPassword() {
    return password;
  }

  protected void assertResponse(int status) {
    HttpStatusCode statusCode = HttpStatusCode.findByCode(status);
    if (statusCode.isOk() == false) {
      String msg = String.format("Unexpected response: %s %s", status, statusCode.getReason());
      throw new ApiException(statusCode, msg);
    }
  }

  protected Map<String,Object> toMap(String...keyValuePairs) {

    if (keyValuePairs == null) {
      return new HashMap<>();
    }

    Map<String,Object> map = new HashMap<>();
    for (String pair : keyValuePairs) {
      int pos = (pair == null) ? -1 : pair.indexOf("=");
      if (pair == null) {
        map.put(null,null);
      } else if (pos < 0) {
        map.put(pair, null);
      } else {
        String key = pair.substring(0, pos);
        String value = pair.substring(pos+1);
        map.put(key, value);
      }
    }
    return map;
  }

  protected Invocation.Builder builder(String url, Map<String, Object> queryMap, String...acceptedResponseTypes) {
    Client client = ClientBuilder.newBuilder().build();
    UriBuilder uriBuilder = new JerseyUriBuilder().uri(getApiUrl()).path(url);

    for (Map.Entry<String,Object> queryParam : queryMap.entrySet()) {
      uriBuilder.queryParam(queryParam.getKey(), queryParam.getValue());
    }

    WebTarget target = client.target(uriBuilder);
    if (StringUtils.isNotBlank(getUserName())) {
      target = target.register(HttpAuthenticationFeature.basic(getUserName(), getPassword()));
    }

    return target.request(acceptedResponseTypes);
  }
}
