package org.crazyyak.dev.jerseyspring;

import java.util.*;
import javax.servlet.*;
import org.crazyyak.dev.common.*;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.servlet.*;
import org.springframework.core.annotation.Order;
import org.springframework.web.WebApplicationInitializer;
import org.springframework.web.context.*;
import org.springframework.web.context.support.XmlWebApplicationContext;
import org.springframework.web.filter.DelegatingFilterProxy;

// Order required to ensure this initializer is run before Jersey 2.x Spring WebApplicationInitializer
@Order(0)
public abstract class YakWebAppInitializer implements WebApplicationInitializer {

  /** Identifies the Jersey ResourceConfig for this app */
  public abstract Class<? extends ResourceConfig> getApplicationClass();

  /**
   * Defines the name of the system property that is used to
   * identify the application's current environment.
   */
  public abstract String getEnvironmentPropertyName();

  /**
   * Defines the name of the system property that is used to
   * override the list of spring profiles for this application.
   */
  protected abstract String getProfilesPropertyName();

  /** Defines the location of the main spring config file. */
  protected abstract String getSpringConfigLocation();

  public YakWebAppInitializer() {
  }

  @Override
  public void onStartup(ServletContext servletContext) throws ServletException {

    WebApplicationContext appContext = createWebApplicationContext();

    addListeners(servletContext, appContext);

    addFilters(servletContext);

    addServlets(servletContext);
  }

  protected void addFilters(ServletContext servletContext) {

    // Spring security filter chain
    addSpringSecurityFilter(servletContext);

    addJerseyFilter(servletContext);
  }

  protected void addListeners(ServletContext servletContext, WebApplicationContext appContext) {
    // Add context listener with refreshed appContext
    servletContext.addListener(new ContextLoaderListener(appContext));
  }

  protected XmlWebApplicationContext createWebApplicationContext() {

    XmlWebApplicationContext appContext = new XmlWebApplicationContext();
    appContext.setConfigLocation(getSpringConfigLocation());

    String[] profiles = getSpringProfiles();
    appContext.getEnvironment().setActiveProfiles(profiles);

    // Refresh the spring context -- important, Jersey 2.x
    // Spring integration will not work with out it.
    appContext.refresh();

    return appContext;
  }

  /** Configures the Spring Security filter */
  protected void addSpringSecurityFilter(ServletContext servletContext) {
    FilterRegistration.Dynamic securityFilter = servletContext.addFilter("springSecurityFilterChain", DelegatingFilterProxy.class);
    securityFilter.addMappingForUrlPatterns(null, false, "/*");
  }

  /** Configures the Jersey filter */
  protected void addJerseyFilter(ServletContext servletContext) {

    servletContext.setInitParameter("contextConfigLocation", ""); // Prevents Jersey Spring WebInitializer for do any work.

    FilterRegistration.Dynamic jerseyFilter = servletContext.addFilter("jersey-filter", ServletContainer.class);
    jerseyFilter.setInitParameter(ServletProperties.JAXRS_APPLICATION_CLASS, getApplicationClass().getName());
    jerseyFilter.setInitParameter(ServletProperties.FILTER_FORWARD_ON_404, "true");
    jerseyFilter.setInitParameter(ServletProperties.FILTER_STATIC_CONTENT_REGEX, "([^\\s]+(\\.(?i)(html|jpg|png|gif|bmp|css|js|ico))$)");
    jerseyFilter.addMappingForUrlPatterns(null, false, "/*");
  }

  protected void addServlets(ServletContext servletContext) {
    addJspfServlet(servletContext);
  }

  /** Includes jspf as servlet */
  protected void addJspfServlet(ServletContext servletContext) {
    servletContext.getServletRegistration("jsp").addMapping("*.jspf");
  }

  protected String[] getSpringProfiles() {

    String environmentName = System.getProperty(getEnvironmentPropertyName(), "null");
    String envProfileName = "env-"+environmentName;
    String defaultProfiles = StringUtils.concat(",", "main", "live", envProfileName);

    String profilesString = System.getProperty(getProfilesPropertyName(), defaultProfiles);
    List<String> profiles = new ArrayList<>();

    for (String profile : profilesString.split(",")) {
      profiles.add(profile.trim());
    }

    return ReflectUtils.toArray(String.class, profiles);
  }
}