package org.crazyyak.dev.jerseyspring;

import java.io.File;
import java.io.IOException;
import java.util.*;
import javax.servlet.*;

import org.apache.commons.logging.LogConfigurationException;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.*;
import org.crazyyak.dev.common.*;
import org.crazyyak.dev.common.exceptions.ExceptionUtils;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.servlet.*;
import org.springframework.core.annotation.Order;
import org.springframework.util.Log4jConfigurer;
import org.springframework.web.WebApplicationInitializer;
import org.springframework.web.context.*;
import org.springframework.web.context.support.XmlWebApplicationContext;
import org.springframework.web.filter.DelegatingFilterProxy;

// Order required to ensure this initializer is run before Jersey 2.x Spring WebApplicationInitializer
@Order(0)
public abstract class YakWebAppInitializer implements WebApplicationInitializer {

  /** Identifies the Jersey ResourceConfig for this app */
  public abstract Class<? extends ResourceConfig> getApplicationClass(ServletContext servletContext, WebApplicationContext appContext);

  /**
   * Defines the name of the system property that is used to
   * identify the application's current environment.
   */
  public abstract String getEnvironmentPropertyName(ServletContext servletContext, WebApplicationContext appContext);

  /**
   * Defines the name of the system property that is used to
   * override the list of spring profiles for this application.
   */
  protected abstract String getProfilesPropertyName(ServletContext servletContext, WebApplicationContext appContext);

  /** Defines the location of the main spring config file. */
  protected abstract String getSpringConfigLocation(ServletContext servletContext, WebApplicationContext appContext);

  public YakWebAppInitializer() {
  }

  @Override
  public void onStartup(ServletContext servletContext) throws ServletException {

    WebApplicationContext appContext = createWebApplicationContext(servletContext);

    addListeners(servletContext, appContext);

    addFilters(servletContext, appContext);

    addServlets(servletContext, appContext);
  }

  protected void addFilters(ServletContext servletContext, WebApplicationContext appContext) {

    // Spring security filter chain
    addSpringSecurityFilter(servletContext, appContext);

    addJerseyFilter(servletContext, appContext);
  }

  protected void addListeners(ServletContext servletContext, WebApplicationContext appContext) {
    // Add context listener with refreshed appContext
    servletContext.addListener(new ContextLoaderListener(appContext));
  }

  protected XmlWebApplicationContext createWebApplicationContext(ServletContext servletContext) {

    XmlWebApplicationContext appContext = new XmlWebApplicationContext();
    appContext.setConfigLocation(getSpringConfigLocation(servletContext, appContext));

    String[] profiles = getSpringProfiles(servletContext, appContext);
    appContext.getEnvironment().setActiveProfiles(profiles);

    // Refresh the spring context -- important, Jersey 2.x
    // Spring integration will not work with out it.
    appContext.refresh();

    return appContext;
  }

  /** Configures the Spring Security filter */
  protected void addSpringSecurityFilter(ServletContext servletContext, WebApplicationContext appContext) {
    FilterRegistration.Dynamic securityFilter = servletContext.addFilter("springSecurityFilterChain", DelegatingFilterProxy.class);
    securityFilter.addMappingForUrlPatterns(null, false, "/*");
  }

  /** Configures the Jersey filter */
  protected void addJerseyFilter(ServletContext servletContext, WebApplicationContext appContext) {

    servletContext.setInitParameter("contextConfigLocation", ""); // Prevents Jersey Spring WebInitializer for do any work.

    FilterRegistration.Dynamic jerseyFilter = servletContext.addFilter("jersey-filter", ServletContainer.class);
    jerseyFilter.setInitParameter(ServletProperties.JAXRS_APPLICATION_CLASS, getApplicationClass(servletContext, appContext).getName());
    jerseyFilter.setInitParameter(ServletProperties.FILTER_FORWARD_ON_404, "true");
    jerseyFilter.setInitParameter(ServletProperties.FILTER_STATIC_CONTENT_REGEX, "([^\\s]+(\\.(?i)(html|jpg|png|gif|bmp|css|js|ico|pdf|txt))$)");
    jerseyFilter.addMappingForUrlPatterns(null, false, "/*");
  }

  protected void addServlets(ServletContext servletContext, WebApplicationContext appContext) {
    addJspfServlet(servletContext, appContext);
  }

  /** Includes jspf as servlet */
  protected void addJspfServlet(ServletContext servletContext, WebApplicationContext appContext) {
    servletContext.getServletRegistration("jsp").addMapping("*.jspf");
  }

  protected String[] getSpringProfiles(ServletContext servletContext, WebApplicationContext appContext) {

    String environmentName = System.getProperty(getEnvironmentPropertyName(servletContext, appContext), "null");
    String envProfileName = "env-"+environmentName;
    String defaultProfiles = StringUtils.concat(",", "main", "live", envProfileName);

    String profilesString = System.getProperty(getProfilesPropertyName(servletContext, appContext), defaultProfiles);
    List<String> profiles = new ArrayList<>();

    for (String profile : profilesString.split(",")) {
      profiles.add(profile.trim());
    }

    return ReflectUtils.toArray(String.class, profiles);
  }
}