/*
 * Copyright 2012 Jacob D Parr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.jerseyspring.jackson;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.*;
import javax.ws.rs.core.MediaType;
import org.apache.commons.logging.*;
import org.crazyyak.dev.domain.money.Money;
import org.glassfish.jersey.message.internal.AbstractMessageReaderWriterProvider;
import org.joda.time.*;

public abstract class BaseReaderWriterProvider extends AbstractMessageReaderWriterProvider<Object> {

  private static final Log log = LogFactory.getLog(BaseReaderWriterProvider.class);

  protected boolean supportsArrays;
  protected boolean supportsEnums;
  protected boolean supportsPrimitives;

  protected final List<Class> supportedTypes = new ArrayList<Class>();

  protected final List<MediaType> mediaTypes = new ArrayList<MediaType>();

  protected BaseReaderWriterProvider(Collection<MediaType> mediaTypes) {

    this.mediaTypes.addAll(mediaTypes);

    supportsArrays = true;
    supportsEnums = true;
    supportsPrimitives = true;

    supportedTypes.add(String.class);
    supportedTypes.add(Number.class);
    supportedTypes.add(Collection.class);
    supportedTypes.add(Boolean.class);
    supportedTypes.add(Character.class);

    supportedTypes.add(Money.class);

    supportedTypes.add(DateMidnight.class);
    supportedTypes.add(DateTime.class);
    supportedTypes.add(Duration.class);
    supportedTypes.add(Instant.class);
    supportedTypes.add(LocalDateTime.class);
    supportedTypes.add(LocalDate.class);
    supportedTypes.add(LocalTime.class);
    supportedTypes.add(Period.class);
    supportedTypes.add(ReadableDateTime.class);
    supportedTypes.add(ReadableInstant.class);
    supportedTypes.add(Interval.class);
  }

  protected BaseReaderWriterProvider(Collection<Class> supportedTypes, Collection<MediaType> mediaTypes, boolean supportsArrays, boolean supportsEnums, boolean supportsPrimitives) {

    this.supportsArrays = supportsArrays;
    this.supportsEnums = supportsEnums;
    this.supportsPrimitives = supportsPrimitives;

    this.supportedTypes.addAll(supportedTypes);
  }

  @Override
  public final boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
    return isSupported(type, genericType, annotations, mediaType);
  }

  @Override
  public final boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
    return isSupported(type, genericType, annotations, mediaType);
  }

  protected boolean isSupported(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
    if (mediaTypes.contains(mediaType) == false) {
      return false;
    }

    if (type.isArray() && supportsArrays == false) {
      return false;
    } else if (type.isPrimitive() && supportsPrimitives == false) {
      return false;
    } else if (type.isEnum() && supportsEnums == false) {
      return false;
    }

    if (type.isArray()) {
      type = type.getComponentType();
    }

    for (Class supportedType : supportedTypes) {
      // noinspection unchecked
      if (supportedType.isAssignableFrom(type)) {
        return true;
      }
    }

    return false;
  }
}
