package org.crazyyak.dev.jerseyspring;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.*;
import javax.ws.rs.ext.*;
import org.apache.commons.logging.*;
import org.crazyyak.dev.common.exceptions.ApiException;
import org.springframework.security.authentication.BadCredentialsException;

@Provider
public class YakExceptionMapper implements ExceptionMapper<Throwable> {

  private static final Log log = LogFactory.getLog(YakExceptionMapper.class);

  @Context
  protected UriInfo uriInfo;

  public YakExceptionMapper() {
    log.info("Created exception mapper");
  }

  @Override
  public Response toResponse(Throwable ex) {

    Response response;

    if (ex instanceof BadCredentialsException) {
      logException(ex, 403);
      return createResponse_403(ex);

    } else if (ex instanceof ApiException) {
      ApiException apiEx = (ApiException)ex;

      if (apiEx.getHttpStatusCode().isBadRequest()) return createResponse_400(ex);
      if (apiEx.getHttpStatusCode().isForbidden()) return createResponse_403(ex);
      if (apiEx.getHttpStatusCode().isNotFound()) return createResponse_404(ex);
      if (apiEx.getHttpStatusCode().isMethodNotAllowed()) return createResponse_405(ex);

      response = Response.status(apiEx.getStatusCode()).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();

    } else if (ex instanceof WebApplicationException) {
      WebApplicationException wae = (WebApplicationException)ex;

      if (wae.getResponse() == null) {
        response = Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();
      } else {
        response = wae.getResponse();
        int status = response.getStatus();
        if (status == 400) return createResponse_400(ex);
        if (status == 403) return createResponse_403(ex);
        if (status == 404) return createResponse_404(ex);
        if (status == 405) return createResponse_405(ex);
      }

    } else {
      response = Response.status(Response.Status.INTERNAL_SERVER_ERROR).type(MediaType.TEXT_PLAIN_TYPE).entity(ex.getMessage()).build();
    }

    logException(ex, response.getStatus());
    return response;
  }

  protected void logException(Throwable throwable, int status) {
    String msg = "Status " + status;
    if (uriInfo != null) {
      msg += " ";
      msg += uriInfo.getRequestUri();
    }

    if (400 == status) {
      log.info(msg, throwable);
    } else {
      log.error(msg, throwable);
    }
  }

  protected Response createResponse_400(Throwable ex) {
    logException(ex, 400);
    return Response.status(400).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();
  }

  protected Response createResponse_403(Throwable ex) {
    logException(ex, 403);
    return Response.status(403).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();
  }

  protected Response createResponse_404(Throwable ex) {
    logException(ex, 404);
    return Response.status(404).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();
  }

  protected Response createResponse_405(Throwable ex) {
    logException(ex, 405);
    return Response.status(405).entity(ex.getMessage()).type(MediaType.TEXT_PLAIN).build();
  }
}
