/*
 * Copyright 2012 Jacob D Parr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.crazyyak.dev.jerseyspring.jackson;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.*;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.*;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.*;
import org.crazyyak.dev.common.IoUtils;

/**
 * Provided for Jersey applications that wish to use Jackson for marshaling with a
 * specific object mapper to and from JSON. Derived classes need only annotate the
 * class with @Service for spring and @Provider for Jersey.
 */
public abstract class JacksonReaderWriterProvider extends BaseReaderWriterProvider {

  private final ObjectMapper objectMapper;

  public JacksonReaderWriterProvider(ObjectMapper objectMapper, MediaType mediaType) {
    super(Arrays.asList(mediaType));
    this.objectMapper = objectMapper;
  }

  public JacksonReaderWriterProvider(ObjectMapper objectMapper, Collection<MediaType> mediaTypes) {
    super(mediaTypes);
    this.objectMapper = objectMapper;
  }

  public JacksonReaderWriterProvider(ObjectMapper objectMapper, Collection<Class> supportedTypes, Collection<MediaType> mediaTypes, boolean supportsArrays, boolean supportsEnums, boolean supportsPrimitives) {
    super(supportedTypes, mediaTypes, supportsArrays, supportsEnums, supportsPrimitives);
    this.objectMapper = objectMapper;
  }

  @Override
  public Object readFrom(Class<Object> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> map, InputStream stream) throws IOException, WebApplicationException {
    String json = IoUtils.toString(stream);
    return objectMapper.readValue(json, type);
  }

  @Override
  public void writeTo(Object value, Class<?> aClass, Type type, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> map, OutputStream stream) throws IOException, WebApplicationException {
    String json = objectMapper.writeValueAsString(value);
    stream.write(json.getBytes());
  }
}
