/*
 * Copyright (c) 2010-2013, Munchie Monster, LLC.
 */

package org.crazyyak.dev.selenium;

import java.lang.CharSequence;import java.lang.Exception;import java.lang.String;import java.lang.System;import java.lang.Thread;
import java.util.ArrayList;import java.util.List;import java.util.NoSuchElementException;
import java.util.concurrent.TimeUnit;
import org.openqa.selenium.Alert;import org.openqa.selenium.By;import org.openqa.selenium.Dimension;import org.openqa.selenium.Point;import org.openqa.selenium.WebDriver;import org.openqa.selenium.WebDriverException;import org.openqa.selenium.WebElement;import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.interactions.Actions;

public class SeleniumBrowser {

  private final WebDriver driver;
  private final long defaultTimeout;

  public SeleniumBrowser() throws Exception {
    this(500, 10*1000);
  }

  public SeleniumBrowser(long implicitlyWait) throws Exception {
    this(implicitlyWait, 10*1000);
  }

  public SeleniumBrowser(long implicitlyWait, long defaultTimeout) throws Exception {
    this.defaultTimeout = defaultTimeout;
    this.driver = new ChromeDriver();
    this.driver.manage().window().setPosition(new Point(1700,0));
    this.driver.manage().window().maximize();
    this.driver.manage().timeouts().implicitlyWait(implicitlyWait, TimeUnit.MILLISECONDS);
  }

  public long getDefaultTimeout() {
    return defaultTimeout;
  }

  public void get(java.lang.String url) {
    driver.get(url);
  }

  public WebElement findElement(final String key) {
    try {
      if (key.startsWith("#")) {
        String idKey = key.substring(1);
        List<WebElement> elements = driver.findElements(By.id(idKey));
        if (elements.size() > 1) {
          String msg = String.format("More than one (%s) element was found for the id %s.", elements.size(), idKey);
          throw new SeleniumBrowserException(msg);
        }
        return driver.findElement(By.id(idKey));

      } else if (key.startsWith(".")) {
        String classKey = key.substring(1);
        return driver.findElement(By.className(classKey));

      } else {
        String msg = String.format("Cannot get element for key \"%s\".", key);
        throw new SeleniumBrowserException(msg);
      }
    } catch (NoSuchElementException e) {
      String msg = String.format("Unable to locate the element \"%s\".", key);
      throw new SeleniumBrowserException(msg);
    }
  }



  public List<WebElement> findElements(String key) {
    if (key.startsWith("#")) {
      key = key.substring(1);
      List<WebElement> elements = driver.findElements(By.id(key));
      if (elements.size() > 1) {
        String msg = String.format("More than one (%s) element was found for the id %s.", elements.size(), key);
        throw new SeleniumBrowserException(msg);
      }
      return driver.findElements(By.id(key));

    } else if (key.startsWith(".")) {
      key = key.substring(1);
      return driver.findElements(By.className(key));

    } else {
      String msg = String.format("Cannot get element for key %s.", key);
      throw new SeleniumBrowserException(msg);
    }
  }

  public boolean isPresent(String elementName) {
    return findElements(elementName).isEmpty() == false;
  }

  public boolean isNotPresent(String elementName) {
    return findElements(elementName).isEmpty();
  }

  public void click(String elementName) throws Exception {
    click(elementName, defaultTimeout);
  }
  public void click(String elementName, long timeout) throws Exception {
    WebElement element = null;
    try {
      element = waitForElement(elementName, timeout);
      element.click();
    } catch (WebDriverException e) {
      throw new SeleniumBrowserException(e);
    }
  }

  public void moveTo(String elementName) throws Exception {
    moveTo(elementName, defaultTimeout);
  }
  public void moveTo(String elementName, long timeout) throws Exception {
    try {
      WebElement element = waitForElement(elementName, timeout);
      new Actions(driver).moveToElement(element);
    } catch (WebDriverException e) {
      throw new SeleniumBrowserException(e);
    }
  }

  private Point getCenter(WebElement element) {
    Point location = element.getLocation();
    Dimension dimension = element.getSize();

    location.x = location.x + (int)(dimension.width/2d);
    location.y = location.y + (int)(dimension.height/2d);

    return location;
  }

  public void sendKeys(String elementName, CharSequence text) throws Exception {
    sendKeys(elementName, text, defaultTimeout);
  }
  public void sendKeys(String elementName, CharSequence text, long timeout) throws Exception {
    try {
      waitForElement(elementName, timeout).sendKeys(text);
    } catch (WebDriverException e) {
      throw new SeleniumBrowserException(e);
    }
  }

  public String getText(String elementName) throws Exception {
    return getText(elementName, defaultTimeout);
  }
  public String getText(String elementName, long timeout) throws Exception {
    try {
      return waitForElement(elementName, timeout).getText();
    } catch (WebDriverException e) {
      throw new SeleniumBrowserException(e);
    }
  }

  public void quit() {
    driver.quit();
  }

  public WebElement waitForElement(String elementName) throws Exception {
    return waitForElement(elementName, defaultTimeout);
  }
  public WebElement waitForElement(String elementName, long timeout) throws Exception {

    long start = System.currentTimeMillis();
    List<WebElement> elements = new ArrayList<>();

    while (System.currentTimeMillis() - start < timeout) {
      elements = findElements(elementName);

      if (elements.isEmpty()) {
        Thread.sleep(250);
      } else if (elements.get(0).isDisplayed() == false) {
        Thread.sleep(250);
      } else {
        return elements.get(0);
      }
    }

    if (elements.isEmpty()) {
      String msg = String.format("The element \"%s\" was not found after %s milliseconds.", elementName, timeout);
      throw new SeleniumBrowserException(msg);

    } else {
      String msg = String.format("The element \"%s\" was found but not displayed after %s milliseconds.", elementName, timeout);
      throw new SeleniumBrowserException(msg);
    }
  }

  public void waitForNotElement(String elementName) throws Exception {
    waitForNotElement(elementName, defaultTimeout);
  }
  public void waitForNotElement(String elementName, long timeout) throws Exception {

    long start = System.currentTimeMillis();
    List<WebElement> elements = new ArrayList<>();

    while (System.currentTimeMillis() - start < timeout) {
      elements = findElements(elementName);
      if (elements.isEmpty()) {
        return;
      } else {
        Thread.sleep(250);      }
    }

    if (elements.isEmpty() == false) {
      String msg = String.format("The element \"%s\" was found after %s milliseconds.", elementName, timeout);
      throw new SeleniumBrowserException(msg);
    }
  }

  public void waitForText(String elementName, String message) throws Exception {
    waitForText(elementName, message, defaultTimeout);
  }
  public void waitForText(String elementName, String message, long timeout) throws Exception {
    long start = System.currentTimeMillis();
    while (System.currentTimeMillis() - start < timeout) {
      String text = findElement(elementName).getText();
      if (text.equals(message)) {
        return;
      }
      Thread.sleep(250);
    }
    String msg = String.format("The element \"%s\" did not have a value of \"%s\" after %s milliseconds.", elementName, message, timeout);
    throw new SeleniumBrowserException(msg);
  }

  public void acceptAlert() {
    Alert alert = driver.switchTo().alert();
    alert.accept();
  }
}
