package org.crazyyak.dev.webmvc;

import java.lang.*;import java.lang.Class;import java.lang.Override;import java.lang.String;import java.lang.System;import java.util.*;
import java.util.ArrayList;import java.util.Arrays;import java.util.Collection;import java.util.List;import javax.servlet.*;
import javax.servlet.FilterRegistration;import javax.servlet.ServletContext;import javax.servlet.ServletException;import javax.servlet.ServletRegistration;import org.crazyyak.dev.common.*;
import org.crazyyak.dev.common.ReflectUtils;import org.crazyyak.dev.common.StringUtils;import org.springframework.core.annotation.Order;
import org.springframework.web.WebApplicationInitializer;
import org.springframework.web.context.*;
import org.springframework.web.context.ContextLoaderListener;import org.springframework.web.context.WebApplicationContext;import org.springframework.web.context.support.*;
import org.springframework.web.context.support.AnnotationConfigWebApplicationContext;import org.springframework.web.filter.DelegatingFilterProxy;
import org.springframework.web.servlet.DispatcherServlet;

// Order required to ensure this initializer is run before Jersey 2.x Spring WebApplicationInitializer
@Order(0)
public abstract class YakSpringWebAppInitializer implements WebApplicationInitializer {

  /**
   * Defines the name of the system property that is used to
   * identify the application's current environment.
   */
  public abstract String getEnvironmentPropertyName(ServletContext servletContext, WebApplicationContext appContext);

  /**
   * Defines the name of the system property that is used to
   * override the list of spring profiles for this application.
   */
  protected abstract String getProfilesPropertyName(ServletContext servletContext, WebApplicationContext appContext);

  /** Defines the spring config classes. */
  protected abstract java.lang.Class<?>[] getSpringConfigClasses(ServletContext servletContext, WebApplicationContext appContext);

  public YakSpringWebAppInitializer() {
  }

  @Override
  public void onStartup(ServletContext servletContext) throws ServletException {

    WebApplicationContext appContext = createWebApplicationContext(servletContext);

    addListeners(servletContext, appContext);

    addFilters(servletContext, appContext);

    addServlets(servletContext, appContext);
  }

  protected void addFilters(ServletContext servletContext, WebApplicationContext appContext) {
    // Spring security filter chain
    addSpringSecurityFilter(servletContext, appContext);
  }

  protected void addListeners(ServletContext servletContext, WebApplicationContext appContext) {
    // Add context listener with refreshed appContext
    servletContext.addListener(new ContextLoaderListener(appContext));
  }

  protected WebApplicationContext createWebApplicationContext(ServletContext servletContext) {

    AnnotationConfigWebApplicationContext appContext = new AnnotationConfigWebApplicationContext();

    Class<?>[] configClasses = getSpringConfigClasses(servletContext, appContext);
    if (configClasses != null) {
      appContext.register(configClasses);
    }

    String[] profiles = getSpringProfiles(servletContext, appContext);
    appContext.getEnvironment().setActiveProfiles(profiles);

    // Refresh the spring context -- important, Jersey 2.x
    // Spring integration will not work with out it.
    appContext.setServletContext(servletContext);
    appContext.refresh();

    return appContext;
  }

  /** Configures the Spring Security filter */
  protected void addSpringSecurityFilter(ServletContext servletContext, WebApplicationContext appContext) {
    FilterRegistration.Dynamic securityFilter = servletContext.addFilter("springSecurityFilterChain", DelegatingFilterProxy.class);
    securityFilter.addMappingForUrlPatterns(null, false, "/*");
  }

  protected void addServlets(ServletContext servletContext, WebApplicationContext appContext) {
    addDispatcherServlet(servletContext, appContext);
    addJspfServlet(servletContext, appContext);
    registerStaticResourceUrlPatterns(servletContext, appContext);
  }

  protected void registerStaticResourceUrlPatterns(ServletContext servletContext, WebApplicationContext appContext) {
    for (String urlPattern : getStaticResourceUrlPatterns()) {
      urlPattern = "*."+urlPattern;
      servletContext.getServletRegistration("default").addMapping(urlPattern);
    }
  }

  protected Collection<String> getStaticResourceUrlPatterns() {
    return Arrays.asList("html","jpg","png","gif","bmp","css","js","ico","pdf","txt");
  }

  protected void addDispatcherServlet(ServletContext servletContext, WebApplicationContext appContext) {
    ServletRegistration.Dynamic servlet = servletContext.addServlet("dispatcherServlet", new DispatcherServlet(appContext));
    servlet.addMapping("/");
    servlet.setLoadOnStartup(1);
  }

  /** Includes jspf as servlet */
  protected void addJspfServlet(ServletContext servletContext, WebApplicationContext appContext) {
    servletContext.getServletRegistration("jsp").addMapping("*.jspf");
  }

  protected String[] getSpringProfiles(ServletContext servletContext, WebApplicationContext appContext) {

    String environmentName = System.getProperty(getEnvironmentPropertyName(servletContext, appContext), "null");
    String envProfileName = "env-"+environmentName;
    String defaultProfiles = StringUtils.concat(",", "main", "live", envProfileName);

    String profilesString = System.getProperty(getProfilesPropertyName(servletContext, appContext), defaultProfiles);
    List<String> profiles = new ArrayList<>();

    for (String profile : profilesString.split(",")) {
      profiles.add(profile.trim());
    }

    return ReflectUtils.toArray(String.class, profiles);
  }
}