package org.crazyyak.lib.security.providers;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import org.apache.commons.logging.*;
import org.crazyyak.apis.google.users.*;
import org.crazyyak.dev.common.exceptions.ApiException;
import org.crazyyak.dev.domain.account.*;
import org.crazyyak.lib.security.CurrentUser;
import org.springframework.security.authentication.*;
import org.springframework.security.authentication.dao.AbstractUserDetailsAuthenticationProvider;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.*;

public class GooglePlusSecurityProvider extends AbstractUserDetailsAuthenticationProvider {

  private static final Log log = LogFactory.getLog(GooglePlusSecurityProvider.class);

  private final String clientSecret;
  private final String clientId;

  private final CurrentUserStore store;
  private final ObjectMapper objectMapper;

  public GooglePlusSecurityProvider(CurrentUserStore store, ObjectMapper objectMapper, String clientId, String clientSecret) {
    this.store = store;
    this.objectMapper = objectMapper;

    this.clientId = clientId;
    this.clientSecret = clientSecret;
  }

  @Override
  protected void additionalAuthenticationChecks(UserDetails userDetails, UsernamePasswordAuthenticationToken authentication) throws AuthenticationException {
    if (ProviderUtils.isGoogleAuthentication(authentication) == false || userDetails == null) {
      // We cannot really authenticate the password with Google but this is better than nothing.
      throw new BadCredentialsException(CurrentUser.INVALID_USER_NAME_OR_PASSWORD);
    }
  }

  @Override
  protected UserDetails retrieveUser(String userName, UsernamePasswordAuthenticationToken authentication) throws AuthenticationException {
    try {
      if (ProviderUtils.isGoogleAuthentication(authentication) == false) {
        throw new UsernameNotFoundException("Not Goolge Authentication");
      }

      String code = authentication.getCredentials().toString();
      GoogleAuthentication googleAuth = GoogleUsersUtils.getAuthResponse(code, clientId, clientSecret);
      UserInfo userInfo = GoogleUsersUtils.getUserInfo(googleAuth.getAccessToken());
      CurrentUserSource source = store.getCurrentUserSourceByEmail(userInfo.getEmail());

      if (source == null) {
        throw new BadCredentialsException(CurrentUser.INVALID_USER_NAME_OR_PASSWORD);
      }
      return new CurrentUser(source);

    } catch (IOException e) {
      throw ApiException.internalServerError("Exception during Google-Authentication", e);
    }
  }
}
