package doodle
package interact
package examples

import cats.implicits._
import doodle.language.Basic
import doodle.algebra.Picture
import doodle.core._
import doodle.syntax._
import doodle.interact.animation.Transducer
import doodle.interact.syntax._
import monix.reactive.Observable
import scala.concurrent.duration._
import scala.math.{exp, pow}

object CompositionalCreativity {
  val unitCircle = (a: Angle) => Point(1.0, a)
  val scale = (factor: Double) => (pt: Point) => pt.scaleLength(factor)
  val largeCircle = unitCircle.andThen(scale(100))

  def dot[Alg[x[_]] <: Basic[x], F[_]]: Picture[Alg, F, Unit] =
    circle[Alg,F](10)
      .fillColor(Color.deepPink)
      .strokeColor(Color.deepPink.spin(-15.degrees))
      .strokeWidth(3.0)

  def sample[Alg[x[_]] <: Basic[x], F[_]](stop: Angle, steps: Int)(
      f: Angle => Point
  ): Transducer[Picture[Alg, F, Unit]] =
    (0.degrees)
      .upToIncluding(stop)
      .forSteps(steps.toLong)
      .scanLeft(empty[Alg, F]) { (picture, angle) =>
        val pt = f(angle)
        picture.on(dot.at(pt))
      }

  def circleAnimation[Alg[x[_]] <: Basic[x], F[_]]
      : Observable[Picture[Alg, F, Unit]] =
    sample(360.degrees, 17)(largeCircle)
      .repeat(5)
      .toObservable
      .withFrameRate(100.millis)
  // .animate(Frame.size(300, 300).background(Color.midnightBlue))

  val linear = (pt: Point) => pt.scaleLength(pt.angle.toTurns)

  val archimedeanSpiral = unitCircle.andThen(linear)

  val largeArchimedeanSpiral =
    archimedeanSpiral.andThen(scale(100))

  def archimedeanSpiralAnimation[Alg[x[_]] <: Basic[x], F[_]]
      : Observable[Picture[Alg, F, Unit]] =
    sample(1080.degrees, 49)(largeArchimedeanSpiral)
      .repeat(5)
      .toObservable
      .withFrameRate(100.millis)

  val power = (p: Double) =>
    (pt: Point) => pt.scaleLength(pow(pt.angle.toTurns, p))
  val exponential = (pt: Point) => pt.scaleLength(exp(pt.angle.toTurns))

  val logarithmicSpiral = unitCircle.andThen(exponential).andThen(scale(50))
  val quadraticSpiral = unitCircle.andThen(power(2)).andThen(scale(50))

  def spiralAnimation[Alg[x[_]] <: Basic[x], F[_]]
      : Observable[Picture[Alg, F, Unit]] =
    sample[Alg, F](1080.degrees, 97)(logarithmicSpiral)
      .product(sample[Alg, F](1080.degrees, 97)(quadraticSpiral))
      .map {
        case (log, quad) =>
          log
            .fillColor(Color.hotpink)
            .strokeColor(Color.hotpink.spin(-15.degrees))
            .on(quad)
      }
      .repeat(5)
      .toObservable
      .withFrameRate(100.millis)
}
