/*
 * Copyright 2015-2020 Noel Welsh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package doodle
package plot

import cats.Contravariant
import cats.syntax.contravariant._
import doodle.core.{Point, Transform}

/**
  * A Scale translates data to a location. It usually indicates where data should
  * be placed on a plot.
  */
trait Scale[A] extends Function[A, Point] {
  import AsPoint._

  def fromAsPoint[B](implicit ev: Point =:= A, asPoint: AsPoint[B]): Scale[B] =
    this.contramap(b => ev(b.asPoint))
}
object Scale {
  final case class BasicScale[A](f: A => Point) extends Scale[A] {
    def apply(v1: A): Point = f(v1)
  }

  implicit val scaleFunctorInstance: Contravariant[Scale] =
    new Contravariant[Scale] {
      def contramap[A, B](fa: Scale[A])(f: B => A): Scale[B] =
        BasicScale[B](b => fa(f(b)))
    }

  def fromTransform(tx: Transform): Scale[Point] =
    BasicScale[Point](pt => tx(pt))

  def apply[A](f: A => Point): Scale[A] =
    BasicScale(f)

  /**
    * Create a Scale that linearly maps (minX, minY) to (0, 0) and (maxX, maxY)
    * to (xRange, yRange)
    */
  def linear(minX: Double, maxX: Double, minY: Double, maxY: Double)(
      xRange: Double,
      yRange: Double
  ): Scale[Point] = {
    assert(minX <= maxX, s"Linear scale must have minimum X <= maximum X")
    assert(minY <= maxY, s"Linear scale must have minimum Y <= maximum Y")

    val lengthX = maxX - minX
    val lengthY = maxY - minY
    Scale { pt =>
      Point(
        ((pt.x - minX) / lengthX) * xRange,
        ((pt.y - minY) / lengthY) * yRange
      )
    }
  }
}
