/*
 * Copyright 2015-2020 Noel Welsh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package doodle
package plot
package geometry

import doodle.algebra.{Algebra, Picture}

/**
  * Geometry describes how to draw data within a plot (e.g. a scatter plot or a
  * line).
  *
  * The type
  * - `Alg` represents the `Algebra` needed to render this geometry;
  * - `F` is the effect type the geometry is rendered to; and
  * - `A` is the type of the data.
  */
trait Geometry[Alg[x[_]] <: Algebra[x], F[_], A] {

  /**
    * Render this geometry to a picture using the given scale to determine
    * layout.
    *
    * The scale converts from whatever coordinates the data uses to the
    * coordinates for the rendered output. The convention for the scale is that
    * (0,0) represents the bottom left corner of the area where the data is
    * drawn (though the actual picture may extend beyond this point due to the
    * glyphs representing the data taking up space). Getting the bounding box of
    * the picture allows the user to determine if this has occurred (as the
    * bounding box will extend beyond the origin).
    */
  def render(scale: Scale[A]): Picture[Alg, F, Unit]
}
