package doodle
package reactor
package examples

import doodle.core._
import doodle.java2d._
import doodle.image._
import doodle.image.syntax.core._
import doodle.reactor._
import scala.concurrent.duration._

sealed trait Shape {
  import Shape._

  def lifespan: Int
  def next: Shape =
    this match {
      case Star(lifespan, points, color) =>
        if (lifespan == 0) Square(60, 0.degrees)
        else Star(lifespan - 1, points, color.spin(5.degrees))
      case Square(lifespan, angle) =>
        if (lifespan == 0) Circle(60, 10)
        else Square(lifespan - 1, angle + 5.degrees)
      case Circle(lifespan, diameter) =>
        if (lifespan == 0) Star(60, 5, Color.blue)
        else Circle(lifespan - 1, diameter + 5)
    }
}
object Shape {
  val initial: Shape = Star(60, 5, Color.blue)

  final case class Star(lifespan: Int, points: Int, color: Color) extends Shape
  final case class Square(lifespan: Int, angle: Angle) extends Shape
  final case class Circle(lifespan: Int, diameter: Double) extends Shape
}

object ShapeExample {
  val animation =
    Reactor
      .init(Shape.initial)
      .tickRate(20.millis)
      .onTick(shape => shape.next)
      .render { (shape: Shape) =>
        shape match {
          case Shape.Star(_, points, color) =>
            Image.star(points, 200, 100, 0.degrees).fillColor(color)
          case Shape.Square(_, angle)    =>
            Image.square(200).rotate(angle)
          case Shape.Circle(_, diameter) =>
            Image.circle(diameter)
        }
      }

  def go() =
    animation.run(Frame.size(600, 600))
}
