/*
 * Copyright 2023 Creative Scala
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package chartreuse.examples

import scala.collection.immutable.ArraySeq

// HadCRUT5 temperature anomaly data
// https://www.metoffice.gov.uk/hadobs/hadcrut5/index.html
object HadCrut5 {

  final case class Record(
      year: Int,
      month: Int,
      anomaly: Double,
      lower: Double,
      upper: Double
  )

  val dateRe = raw"(\d{4})-(\d{2})".r
  def parse(
      date: String,
      anomaly: Double,
      lower: Double,
      upper: Double
  ): Record = {
    date match {
      case dateRe(year, month) =>
        Record(year.toInt, month.toInt, anomaly, lower, upper)
    }
  }

  val data: ArraySeq[Record] =
    ArraySeq(
      parse("1850-01", -0.67456436, -0.98177195, -0.3673568),
      parse("1850-02", -0.333416, -0.700901, 0.034069024),
      parse("1850-03", -0.59132266, -0.9339191, -0.2487262),
      parse("1850-04", -0.58872116, -0.8982594, -0.27918297),
      parse("1850-05", -0.5081851, -0.7612137, -0.2551565),
      parse("1850-06", -0.34424013, -0.60947233, -0.079007894),
      parse("1850-07", -0.15979019, -0.39714956, 0.07756917),
      parse("1850-08", -0.2076543, -0.4648224, 0.04951382),
      parse("1850-09", -0.3847069, -0.6477238, -0.121690065),
      parse("1850-10", -0.53312653, -0.78265744, -0.28359562),
      parse("1850-11", -0.2825075, -0.54502493, -0.01999013),
      parse("1850-12", -0.40367043, -0.6826584, -0.12468251),
      parse("1851-01", -0.20073938, -0.51260436, 0.11112556),
      parse("1851-02", -0.46933234, -0.8057982, -0.13286653),
      parse("1851-03", -0.6460531, -0.95056796, -0.34153816),
      parse("1851-04", -0.5421326, -0.8318831, -0.25238213),
      parse("1851-05", -0.19762275, -0.47129744, 0.07605195),
      parse("1851-06", -0.13670355, -0.41463488, 0.1412278),
      parse("1851-07", -0.0968156, -0.3358151, 0.1421839),
      parse("1851-08", -0.10175375, -0.36319193, 0.15968445),
      parse("1851-09", -0.091244884, -0.36001372, 0.17752396),
      parse("1851-10", -0.00836263, -0.26999545, 0.2532702),
      parse("1851-11", -0.08189084, -0.38667485, 0.22289316),
      parse("1851-12", -0.22754622, -0.551656, 0.09656359),
      parse("1852-01", -0.37509567, -0.6837687, -0.06642263),
      parse("1852-02", -0.47672674, -0.82690173, -0.12655175),
      parse("1852-03", -0.5597683, -0.8769547, -0.24258201),
      parse("1852-04", -0.5853704, -0.8682941, -0.30244678),
      parse("1852-05", -0.12741269, -0.39380231, 0.13897693),
      parse("1852-06", -0.083659135, -0.33952817, 0.17220989),
      parse("1852-07", 0.005029385, -0.22846359, 0.23852237),
      parse("1852-08", -0.13558698, -0.39293692, 0.121762976),
      parse("1852-09", -0.0019016695, -0.26057866, 0.25677535),
      parse("1852-10", -0.172434, -0.41274902, 0.06788101),
      parse("1852-11", -0.3052479, -0.58084655, -0.029649276),
      parse("1852-12", 0.065385304, -0.21881793, 0.34958854),
      parse("1853-01", -0.23259853, -0.51683515, 0.0516381),
      parse("1853-02", -0.40372267, -0.70065147, -0.10679385),
      parse("1853-03", -0.28042167, -0.5714255, 0.010582115),
      parse("1853-04", -0.38631994, -0.65944624, -0.11319364),
      parse("1853-05", -0.2678886, -0.5075292, -0.02824804),
      parse("1853-06", -0.1423762, -0.3862959, 0.101543516),
      parse("1853-07", -0.08344058, -0.297061, 0.13017985),
      parse("1853-08", -0.056647103, -0.28783756, 0.17454337),
      parse("1853-09", -0.2502504, -0.47014314, -0.030357674),
      parse("1853-10", -0.391768, -0.6159777, -0.16755827),
      parse("1853-11", -0.41142052, -0.67802095, -0.14482014),
      parse("1853-12", -0.33739907, -0.6387964, -0.036001712),
      parse("1854-01", -0.3813734, -0.6314074, -0.1313394),
      parse("1854-02", -0.36109406, -0.6438892, -0.078298904),
      parse("1854-03", -0.24269956, -0.5189016, 0.033502445),
      parse("1854-04", -0.3338324, -0.55951685, -0.10814798),
      parse("1854-05", -0.29079238, -0.49502686, -0.08655792),
      parse("1854-06", -0.29902688, -0.5297665, -0.068287276),
      parse("1854-07", -0.17864197, -0.38943616, 0.03215221),
      parse("1854-08", -0.23944739, -0.4525426, -0.026352169),
      parse("1854-09", -0.21674344, -0.42721775, -0.006269128),
      parse("1854-10", -0.09526511, -0.27856913, 0.08803892),
      parse("1854-11", -0.41029203, -0.680082, -0.14050205),
      parse("1854-12", -0.4503518, -0.72317326, -0.17753038),
      parse("1855-01", -0.15919624, -0.42137975, 0.102987275),
      parse("1855-02", -0.31344938, -0.59707975, -0.02981898),
      parse("1855-03", -0.34401497, -0.6289762, -0.059053726),
      parse("1855-04", -0.21820337, -0.44758487, 0.011178128),
      parse("1855-05", -0.3877071, -0.6007164, -0.17469783),
      parse("1855-06", -0.3329626, -0.5576329, -0.108292304),
      parse("1855-07", -0.3329893, -0.53221554, -0.1337631),
      parse("1855-08", -0.19952463, -0.40983552, 0.010786269),
      parse("1855-09", -0.27071294, -0.45980525, -0.0816206),
      parse("1855-10", -0.19544719, -0.40497714, 0.0140827345),
      parse("1855-11", -0.38930947, -0.6501514, -0.1284676),
      parse("1855-12", -0.41989732, -0.6853395, -0.15445511),
      parse("1856-01", -0.075628236, -0.31484526, 0.16358879),
      parse("1856-02", -0.33713824, -0.60642755, -0.0678489),
      parse("1856-03", -0.35769272, -0.63118136, -0.08420405),
      parse("1856-04", -0.3159384, -0.531346, -0.100530796),
      parse("1856-05", -0.28508008, -0.49249765, -0.07766249),
      parse("1856-06", -0.21416615, -0.4383176, 0.009985293),
      parse("1856-07", -0.22469248, -0.42359686, -0.02578809),
      parse("1856-08", -0.23770598, -0.45777294, -0.01763901),
      parse("1856-09", -0.38821492, -0.5981285, -0.17830133),
      parse("1856-10", -0.4381357, -0.6417849, -0.2344865),
      parse("1856-11", -0.55232793, -0.80731004, -0.29734582),
      parse("1856-12", -0.4175237, -0.7001903, -0.13485713),
      parse("1857-01", -0.49476224, -0.7212615, -0.26826298),
      parse("1857-02", -0.43487996, -0.7002412, -0.16951871),
      parse("1857-03", -0.47860673, -0.7623405, -0.19487299),
      parse("1857-04", -0.6443354, -0.8869147, -0.40175614),
      parse("1857-05", -0.65871185, -0.86846006, -0.4489636),
      parse("1857-06", -0.30414528, -0.5227023, -0.085588254),
      parse("1857-07", -0.4113695, -0.6166561, -0.2060829),
      parse("1857-08", -0.36897412, -0.5925666, -0.14538166),
      parse("1857-09", -0.37968943, -0.5874363, -0.17194258),
      parse("1857-10", -0.47263545, -0.6786636, -0.2666073),
      parse("1857-11", -0.6851335, -0.9487671, -0.42149988),
      parse("1857-12", -0.27351704, -0.5484415, 0.0014074869),
      parse("1858-01", -0.29548538, -0.56474644, -0.026224334),
      parse("1858-02", -0.47690183, -0.7623272, -0.19147645),
      parse("1858-03", -0.5476575, -0.84601724, -0.24929774),
      parse("1858-04", -0.42799428, -0.6573944, -0.19859417),
      parse("1858-05", -0.5320951, -0.74805903, -0.31613111),
      parse("1858-06", -0.37952122, -0.60970783, -0.14933462),
      parse("1858-07", -0.3355508, -0.54754287, -0.12355878),
      parse("1858-08", -0.27754986, -0.4933736, -0.0617261),
      parse("1858-09", -0.28318587, -0.49380964, -0.07256212),
      parse("1858-10", -0.2887329, -0.4937586, -0.083707206),
      parse("1858-11", -0.36552474, -0.6381552, -0.09289427),
      parse("1858-12", -0.4549887, -0.7361435, -0.17383386),
      parse("1859-01", -0.38148466, -0.62637764, -0.1365917),
      parse("1859-02", -0.34352297, -0.6039441, -0.08310183),
      parse("1859-03", -0.31371728, -0.590663, -0.036771517),
      parse("1859-04", -0.15715948, -0.3835844, 0.06926547),
      parse("1859-05", -0.15164812, -0.35725668, 0.053960454),
      parse("1859-06", -0.24957645, -0.46606663, -0.033086263),
      parse("1859-07", -0.31209, -0.49856845, -0.12561159),
      parse("1859-08", -0.24464844, -0.46309808, -0.026198825),
      parse("1859-09", -0.393858, -0.5993753, -0.18834068),
      parse("1859-10", -0.25472045, -0.4681661, -0.04127479),
      parse("1859-11", -0.2875482, -0.54473567, -0.030360727),
      parse("1859-12", -0.28437135, -0.56793344, -0.0008092772),
      parse("1860-01", -0.39058298, -0.6584608, -0.12270518),
      parse("1860-02", -0.4682912, -0.7842476, -0.15233481),
      parse("1860-03", -0.7864609, -1.1232259, -0.44969583),
      parse("1860-04", -0.30610728, -0.5380226, -0.074192025),
      parse("1860-05", -0.36362675, -0.5794941, -0.14775938),
      parse("1860-06", -0.21145956, -0.45290247, 0.029983336),
      parse("1860-07", -0.1847646, -0.40834823, 0.038819034),
      parse("1860-08", -0.1142386, -0.37883177, 0.15035456),
      parse("1860-09", -0.13131975, -0.36443615, 0.10179664),
      parse("1860-10", -0.3459595, -0.5577351, -0.13418393),
      parse("1860-11", -0.60494, -0.8451416, -0.36473843),
      parse("1860-12", -0.774231, -1.0370945, -0.51136756),
      parse("1861-01", -0.91803956, -1.1972613, -0.6388177),
      parse("1861-02", -0.6245886, -0.99098545, -0.25819176),
      parse("1861-03", -0.41567025, -0.8173928, -0.0139476545),
      parse("1861-04", -0.38623625, -0.7137754, -0.05869713),
      parse("1861-05", -0.6607519, -0.9770289, -0.34447485),
      parse("1861-06", -0.25202128, -0.53403753, 0.029994994),
      parse("1861-07", -0.29750192, -0.550664, -0.044339824),
      parse("1861-08", -0.041093126, -0.29339635, 0.2112101),
      parse("1861-09", -0.27033228, -0.5080883, -0.03257631),
      parse("1861-10", -0.47664645, -0.67291355, -0.28037935),
      parse("1861-11", -0.45758948, -0.70344317, -0.21173581),
      parse("1861-12", -0.35085425, -0.6705306, -0.031177888),
      parse("1862-01", -0.82483715, -1.1399956, -0.5096788),
      parse("1862-02", -0.7155356, -1.0883836, -0.34268767),
      parse("1862-03", -0.5353963, -0.9173056, -0.15348701),
      parse("1862-04", -0.52208453, -0.8259743, -0.21819483),
      parse("1862-05", -0.3478659, -0.6414074, -0.054324463),
      parse("1862-06", -0.3066114, -0.554004, -0.05921875),
      parse("1862-07", -0.3448706, -0.5859212, -0.10381997),
      parse("1862-08", -0.4561808, -0.72345316, -0.18890849),
      parse("1862-09", -0.3624242, -0.58751315, -0.13733529),
      parse("1862-10", -0.44867027, -0.66414434, -0.23319624),
      parse("1862-11", -0.6777555, -0.93400127, -0.42150965),
      parse("1862-12", -0.8945406, -1.2084035, -0.58067775),
      parse("1863-01", -0.07634223, -0.40096194, 0.24827747),
      parse("1863-02", -0.24953309, -0.66414887, 0.16508271),
      parse("1863-03", -0.3549987, -0.7145881, 0.004590684),
      parse("1863-04", -0.28827205, -0.5977736, 0.021229519),
      parse("1863-05", -0.3840041, -0.66699326, -0.10101496),
      parse("1863-06", -0.36912903, -0.6320461, -0.106212005),
      parse("1863-07", -0.3123347, -0.5947716, -0.029897762),
      parse("1863-08", -0.29738235, -0.57519704, -0.019567715),
      parse("1863-09", -0.35204113, -0.61949664, -0.08458564),
      parse("1863-10", -0.5043272, -0.742704, -0.26595044),
      parse("1863-11", -0.44055805, -0.68936455, -0.19175154),
      parse("1863-12", -0.50319576, -0.817736, -0.1886555),
      parse("1864-01", -0.8532364, -1.2133667, -0.493106),
      parse("1864-02", -0.5934415, -1.0275025, -0.15938036),
      parse("1864-03", -0.49939686, -0.8818438, -0.11694993),
      parse("1864-04", -0.54358864, -0.87270206, -0.21447514),
      parse("1864-05", -0.32347065, -0.6318895, -0.015051826),
      parse("1864-06", -0.11254485, -0.38457897, 0.15948927),
      parse("1864-07", -0.25960582, -0.530554, 0.011342296),
      parse("1864-08", -0.3079488, -0.57993907, -0.035958502),
      parse("1864-09", -0.4425317, -0.6966836, -0.18837978),
      parse("1864-10", -0.6029094, -0.8257851, -0.38003364),
      parse("1864-11", -0.45623875, -0.70915705, -0.20332046),
      parse("1864-12", -0.5903272, -0.9233383, -0.25731608),
      parse("1865-01", -0.17609702, -0.4980693, 0.14587525),
      parse("1865-02", -0.46668038, -0.926016, -0.0073448196),
      parse("1865-03", -0.66122115, -1.0886513, -0.23379107),
      parse("1865-04", -0.4481533, -0.77840537, -0.117901206),
      parse("1865-05", -0.37767282, -0.678569, -0.07677658),
      parse("1865-06", -0.1958778, -0.45268297, 0.060927358),
      parse("1865-07", -0.21560228, -0.49204847, 0.06084391),
      parse("1865-08", -0.21177977, -0.47747025, 0.053910702),
      parse("1865-09", -0.17093047, -0.43420225, 0.0923413),
      parse("1865-10", -0.26942545, -0.51537, -0.023480851),
      parse("1865-11", -0.33705235, -0.59656864, -0.077536084),
      parse("1865-12", -0.4605614, -0.79645264, -0.124670126),
      parse("1866-01", -0.1320992, -0.47347298, 0.2092746),
      parse("1866-02", -0.40108874, -0.778535, -0.023642521),
      parse("1866-03", -0.6821522, -1.0710651, -0.29323936),
      parse("1866-04", -0.4543501, -0.7766433, -0.132057),
      parse("1866-05", -0.5719213, -0.8795296, -0.26431298),
      parse("1866-06", -0.0985114, -0.35920775, 0.16218495),
      parse("1866-07", -0.00851037, -0.26040974, 0.24338901),
      parse("1866-08", -0.2494715, -0.49963456, 0.00069158967),
      parse("1866-09", -0.37775257, -0.62569046, -0.12981468),
      parse("1866-10", -0.4206272, -0.6282327, -0.21302171),
      parse("1866-11", -0.29829076, -0.52906585, -0.06751567),
      parse("1866-12", -0.40035242, -0.7012374, -0.09946747),
      parse("1867-01", -0.40518752, -0.74488693, -0.065488115),
      parse("1867-02", -0.3535976, -0.78066856, 0.07347331),
      parse("1867-03", -0.6812089, -1.0579066, -0.3045111),
      parse("1867-04", -0.4213023, -0.7407834, -0.1018212),
      parse("1867-05", -0.57436174, -0.8913324, -0.25739112),
      parse("1867-06", -0.35023195, -0.6303714, -0.07009248),
      parse("1867-07", -0.25301105, -0.5183183, 0.01229622),
      parse("1867-08", -0.33047715, -0.60988814, -0.051066175),
      parse("1867-09", -0.14789358, -0.40636784, 0.11058068),
      parse("1867-10", -0.12525536, -0.35672548, 0.106214754),
      parse("1867-11", -0.19566038, -0.43645424, 0.045133475),
      parse("1867-12", -0.44537255, -0.7262891, -0.16445602),
      parse("1868-01", -0.6848776, -1.044338, -0.32541716),
      parse("1868-02", -0.69327545, -1.1240178, -0.2625332),
      parse("1868-03", -0.33465612, -0.727329, 0.058016773),
      parse("1868-04", -0.36299402, -0.66513675, -0.06085131),
      parse("1868-05", -0.28559557, -0.5878955, 0.016704334),
      parse("1868-06", -0.29764694, -0.54980946, -0.045484435),
      parse("1868-07", -0.059621982, -0.30119848, 0.18195452),
      parse("1868-08", -0.1932709, -0.45685738, 0.070315555),
      parse("1868-09", -0.34477365, -0.5571461, -0.13240121),
      parse("1868-10", -0.22966905, -0.4261849, -0.033153232),
      parse("1868-11", -0.42318085, -0.66923624, -0.17712545),
      parse("1868-12", -0.31396636, -0.57192636, -0.05600635),
      parse("1869-01", -0.33569098, -0.6297284, -0.041653544),
      parse("1869-02", 0.13131645, -0.19830833, 0.46094123),
      parse("1869-03", -0.6257712, -0.955734, -0.29580843),
      parse("1869-04", -0.40569928, -0.7054616, -0.10593699),
      parse("1869-05", -0.3099707, -0.57051814, -0.049423277),
      parse("1869-06", -0.38451383, -0.6465563, -0.12247129),
      parse("1869-07", -0.41707024, -0.6630914, -0.17104907),
      parse("1869-08", -0.16793925, -0.4234084, 0.08752989),
      parse("1869-09", -0.17162752, -0.3800871, 0.036832076),
      parse("1869-10", -0.2934824, -0.4622508, -0.124713965),
      parse("1869-11", -0.2818132, -0.47505817, -0.08856823),
      parse("1869-12", -0.5365831, -0.7834703, -0.289696),
      parse("1870-01", -0.21106681, -0.4358849, 0.013751276),
      parse("1870-02", -0.4544139, -0.6885369, -0.22029088),
      parse("1870-03", -0.4983347, -0.75912595, -0.23754346),
      parse("1870-04", -0.33294073, -0.56307286, -0.102808595),
      parse("1870-05", -0.2977442, -0.54020154, -0.055286862),
      parse("1870-06", -0.23123583, -0.4569157, -0.005555956),
      parse("1870-07", -0.073357955, -0.2875345, 0.1408186),
      parse("1870-08", -0.2582274, -0.5095156, -0.0069392435),
      parse("1870-09", -0.2023172, -0.4136001, 0.0089657),
      parse("1870-10", -0.44285795, -0.65396285, -0.2317531),
      parse("1870-11", -0.2658218, -0.48923132, -0.042412296),
      parse("1870-12", -0.6663718, -0.9382973, -0.3944463),
      parse("1871-01", -0.55831134, -0.82570755, -0.29091513),
      parse("1871-02", -0.6715548, -0.9617495, -0.3813601),
      parse("1871-03", -0.17169894, -0.47935697, 0.13595909),
      parse("1871-04", -0.22192766, -0.45470837, 0.010853066),
      parse("1871-05", -0.3622017, -0.61092186, -0.113481514),
      parse("1871-06", -0.2864155, -0.5228108, -0.05002018),
      parse("1871-07", -0.082433164, -0.3044722, 0.13960588),
      parse("1871-08", -0.32236677, -0.5468733, -0.09786023),
      parse("1871-09", -0.36107233, -0.5652693, -0.15687534),
      parse("1871-10", -0.37327537, -0.55728364, -0.18926708),
      parse("1871-11", -0.5226968, -0.75467384, -0.29071972),
      parse("1871-12", -0.48901418, -0.7645088, -0.21351954),
      parse("1872-01", -0.59878874, -0.85335076, -0.3442267),
      parse("1872-02", -0.61987036, -0.89861023, -0.34113047),
      parse("1872-03", -0.523702, -0.8182255, -0.22917853),
      parse("1872-04", -0.3854148, -0.6421561, -0.12867345),
      parse("1872-05", -0.14151993, -0.37056223, 0.08752235),
      parse("1872-06", -0.27950048, -0.5010192, -0.057981815),
      parse("1872-07", -0.14631678, -0.35728696, 0.064653404),
      parse("1872-08", 0.008637112, -0.22310326, 0.24037747),
      parse("1872-09", -0.17648244, -0.37449455, 0.021529682),
      parse("1872-10", -0.2035192, -0.37991014, -0.027128257),
      parse("1872-11", -0.32202852, -0.5166016, -0.12745541),
      parse("1872-12", -0.5479975, -0.78612375, -0.30987123),
      parse("1873-01", -0.26165044, -0.48202783, -0.04127305),
      parse("1873-02", -0.28293252, -0.57342094, 0.0075558876),
      parse("1873-03", -0.4087238, -0.65150857, -0.16593905),
      parse("1873-04", -0.4805553, -0.6951894, -0.26592112),
      parse("1873-05", -0.3509207, -0.5651311, -0.13671026),
      parse("1873-06", -0.32054913, -0.53528905, -0.10580918),
      parse("1873-07", -0.17570111, -0.39297614, 0.04157393),
      parse("1873-08", -0.22783396, -0.4349165, -0.020751415),
      parse("1873-09", -0.40525547, -0.60818976, -0.20232113),
      parse("1873-10", -0.3906487, -0.57215583, -0.20914152),
      parse("1873-11", -0.46331128, -0.66464955, -0.261973),
      parse("1873-12", -0.3279057, -0.59806544, -0.057746004),
      parse("1874-01", -0.19854628, -0.45300004, 0.055907503),
      parse("1874-02", -0.43842116, -0.6789994, -0.1978429),
      parse("1874-03", -0.6729315, -0.92940164, -0.4164614),
      parse("1874-04", -0.4370358, -0.6482773, -0.22579432),
      parse("1874-05", -0.33399135, -0.5918376, -0.07614513),
      parse("1874-06", -0.56584716, -0.79844147, -0.33325285),
      parse("1874-07", -0.16706698, -0.38213357, 0.047999617),
      parse("1874-08", -0.2960711, -0.52731186, -0.064830355),
      parse("1874-09", -0.16819394, -0.35974884, 0.023360983),
      parse("1874-10", -0.47115755, -0.6433688, -0.29894635),
      parse("1874-11", -0.41774487, -0.6313548, -0.20413499),
      parse("1874-12", -0.31200656, -0.5668208, -0.057192285),
      parse("1875-01", -0.5711182, -0.79736364, -0.3448727),
      parse("1875-02", -0.4792721, -0.71974623, -0.23879792),
      parse("1875-03", -0.56430274, -0.7930157, -0.33558974),
      parse("1875-04", -0.35805592, -0.5733307, -0.14278117),
      parse("1875-05", -0.2728362, -0.47623995, -0.069432504),
      parse("1875-06", -0.27996176, -0.4962079, -0.06371561),
      parse("1875-07", -0.18206008, -0.38412824, 0.02000808),
      parse("1875-08", -0.24648568, -0.4747995, -0.018171854),
      parse("1875-09", -0.23427878, -0.44139335, -0.027164232),
      parse("1875-10", -0.23321441, -0.40914226, -0.057286542),
      parse("1875-11", -0.50920546, -0.7304494, -0.28796157),
      parse("1875-12", -0.57672006, -0.8087105, -0.3447295),
      parse("1876-01", -0.51370406, -0.732396, -0.29501218),
      parse("1876-02", -0.33739862, -0.56518865, -0.10960863),
      parse("1876-03", -0.36232424, -0.6376168, -0.08703163),
      parse("1876-04", -0.42427677, -0.65553075, -0.19302277),
      parse("1876-05", -0.59042513, -0.8079633, -0.3728869),
      parse("1876-06", -0.35227904, -0.57574004, -0.12881808),
      parse("1876-07", -0.21372749, -0.41223747, -0.015217529),
      parse("1876-08", -0.32724735, -0.5655297, -0.08896501),
      parse("1876-09", -0.41784137, -0.63995034, -0.19573241),
      parse("1876-10", -0.29983562, -0.47799063, -0.12168063),
      parse("1876-11", -0.60684675, -0.8142865, -0.39940697),
      parse("1876-12", -0.64341265, -0.89493567, -0.3918896),
      parse("1877-01", -0.41108242, -0.6209636, -0.20120125),
      parse("1877-02", -0.20926146, -0.4276892, 0.009166266),
      parse("1877-03", -0.091358565, -0.31345627, 0.13073914),
      parse("1877-04", -0.3459171, -0.5490583, -0.14277588),
      parse("1877-05", -0.41127652, -0.61740625, -0.20514679),
      parse("1877-06", -0.13819988, -0.35527813, 0.078878365),
      parse("1877-07", -0.016253214, -0.22757587, 0.19506943),
      parse("1877-08", 0.21073052, -0.011481554, 0.4329426),
      parse("1877-09", 0.10758122, -0.07691583, 0.29207826),
      parse("1877-10", 0.11331653, -0.0732508, 0.29988387),
      parse("1877-11", 0.03397783, -0.15571973, 0.2236754),
      parse("1877-12", -0.05556291, -0.2958206, 0.18469477),
      parse("1878-01", -0.06979711, -0.26203996, 0.12244575),
      parse("1878-02", 0.22940955, 0.011935646, 0.44688344),
      parse("1878-03", 0.36132926, 0.11839555, 0.604263),
      parse("1878-04", 0.18853739, -0.0070909946, 0.38416576),
      parse("1878-05", -0.122401394, -0.33232173, 0.08751895),
      parse("1878-06", 0.010291125, -0.20308198, 0.22366422),
      parse("1878-07", -0.0772364, -0.28020662, 0.12573381),
      parse("1878-08", -0.04655335, -0.26132223, 0.16821553),
      parse("1878-09", 0.020835022, -0.16271828, 0.20438832),
      parse("1878-10", -0.1650159, -0.33420274, 0.0041709216),
      parse("1878-11", -0.14255811, -0.34657767, 0.061461445),
      parse("1878-12", -0.3226224, -0.5510177, -0.09422704),
      parse("1879-01", -0.23778471, -0.4489502, -0.026619231),
      parse("1879-02", -0.26602048, -0.49155372, -0.040487222),
      parse("1879-03", -0.2583884, -0.48758295, -0.029193843),
      parse("1879-04", -0.38240048, -0.5881249, -0.17667612),
      parse("1879-05", -0.34387106, -0.53022873, -0.15751341),
      parse("1879-06", -0.28325993, -0.47607452, -0.09044534),
      parse("1879-07", -0.2641225, -0.46030587, -0.0679391),
      parse("1879-08", -0.26534295, -0.48025763, -0.05042825),
      parse("1879-09", -0.23556925, -0.42141894, -0.04971956),
      parse("1879-10", -0.15905875, -0.33532667, 0.017209174),
      parse("1879-11", -0.397338, -0.6246113, -0.17006464),
      parse("1879-12", -0.55045533, -0.81308776, -0.2878229),
      parse("1880-01", -0.39386344, -0.60645384, -0.181273),
      parse("1880-02", -0.4498938, -0.6612949, -0.23849276),
      parse("1880-03", -0.19438674, -0.4402409, 0.051467415),
      parse("1880-04", -0.33803284, -0.56909233, -0.106973305),
      parse("1880-05", -0.3253502, -0.5249128, -0.1257876),
      parse("1880-06", -0.38053036, -0.57394856, -0.18711212),
      parse("1880-07", -0.3148068, -0.5000183, -0.12959526),
      parse("1880-08", -0.10718303, -0.31699923, 0.10263317),
      parse("1880-09", -0.29703382, -0.48228338, -0.11178426),
      parse("1880-10", -0.39094922, -0.54256344, -0.23933502),
      parse("1880-11", -0.28068957, -0.48973435, -0.07164481),
      parse("1880-12", -0.3172651, -0.57946, -0.055070166),
      parse("1881-01", -0.21463032, -0.42456862, -0.004692002),
      parse("1881-02", -0.22458804, -0.4410218, -0.008154285),
      parse("1881-03", -0.24820606, -0.4754435, -0.02096861),
      parse("1881-04", -0.06252021, -0.24772863, 0.1226882),
      parse("1881-05", -0.12189843, -0.31805423, 0.074257374),
      parse("1881-06", -0.39908272, -0.60632294, -0.19184254),
      parse("1881-07", -0.19404487, -0.38651505, -0.0015747005),
      parse("1881-08", -0.13816847, -0.34755346, 0.07121651),
      parse("1881-09", -0.3007603, -0.48109457, -0.120426044),
      parse("1881-10", -0.2305081, -0.3861735, -0.0748427),
      parse("1881-11", -0.38121367, -0.57584876, -0.18657859),
      parse("1881-12", -0.27132502, -0.49155435, -0.05109568),
      parse("1882-01", -0.025385773, -0.21284647, 0.16207492),
      parse("1882-02", -0.11726177, -0.29848227, 0.06395874),
      parse("1882-03", -0.17708035, -0.37251085, 0.018350175),
      parse("1882-04", -0.31474164, -0.4901041, -0.13937919),
      parse("1882-05", -0.32572997, -0.50725716, -0.14420277),
      parse("1882-06", -0.39225483, -0.5854527, -0.19905701),
      parse("1882-07", -0.34102684, -0.524619, -0.15743467),
      parse("1882-08", -0.24510053, -0.45590413, -0.03429691),
      parse("1882-09", -0.15044588, -0.33297318, 0.03208143),
      parse("1882-10", -0.3990836, -0.56060207, -0.23756516),
      parse("1882-11", -0.40350994, -0.59063745, -0.2163824),
      parse("1882-12", -0.65474, -0.8827449, -0.426735),
      parse("1883-01", -0.47277278, -0.6465994, -0.2989461),
      parse("1883-02", -0.4854322, -0.6739853, -0.29687917),
      parse("1883-03", -0.26636964, -0.44524446, -0.08749482),
      parse("1883-04", -0.43903488, -0.6062966, -0.27177316),
      parse("1883-05", -0.3400283, -0.5237819, -0.15627469),
      parse("1883-06", -0.1901544, -0.38033026, 2.1439157e-05),
      parse("1883-07", -0.21858145, -0.38680118, -0.05036173),
      parse("1883-08", -0.28788334, -0.48319572, -0.092570975),
      parse("1883-09", -0.3062682, -0.46854252, -0.14399385),
      parse("1883-10", -0.3400996, -0.49213037, -0.1880688),
      parse("1883-11", -0.39625064, -0.58992153, -0.20257974),
      parse("1883-12", -0.41481745, -0.6323676, -0.19726728),
      parse("1884-01", -0.44355372, -0.63514644, -0.25196096),
      parse("1884-02", -0.33285657, -0.5167038, -0.14900933),
      parse("1884-03", -0.50546813, -0.7000096, -0.31092662),
      parse("1884-04", -0.5089616, -0.67386585, -0.34405738),
      parse("1884-05", -0.56562126, -0.7247028, -0.40653968),
      parse("1884-06", -0.5137893, -0.6937361, -0.3338425),
      parse("1884-07", -0.5658637, -0.7341247, -0.39760268),
      parse("1884-08", -0.429122, -0.6203238, -0.23792021),
      parse("1884-09", -0.5043303, -0.6659263, -0.34273428),
      parse("1884-10", -0.463346, -0.614259, -0.31243297),
      parse("1884-11", -0.5852065, -0.7801215, -0.3902915),
      parse("1884-12", -0.4897219, -0.7129149, -0.2665289),
      parse("1885-01", -0.65614164, -0.83794105, -0.47434217),
      parse("1885-02", -0.5527572, -0.73603135, -0.36948302),
      parse("1885-03", -0.42101642, -0.5955314, -0.24650145),
      parse("1885-04", -0.63497037, -0.78652006, -0.4834207),
      parse("1885-05", -0.60631216, -0.77378696, -0.43883735),
      parse("1885-06", -0.63664883, -0.8227926, -0.45050508),
      parse("1885-07", -0.42047033, -0.5939572, -0.24698347),
      parse("1885-08", -0.4050678, -0.5947867, -0.21534885),
      parse("1885-09", -0.35673314, -0.52351606, -0.18995023),
      parse("1885-10", -0.36150765, -0.51169276, -0.21132258),
      parse("1885-11", -0.38425598, -0.55152065, -0.2169913),
      parse("1885-12", -0.21760142, -0.4325158, -0.0026870253),
      parse("1886-01", -0.479745, -0.6322591, -0.3272309),
      parse("1886-02", -0.47879001, -0.62834597, -0.32923406),
      parse("1886-03", -0.48723003, -0.63084835, -0.34361172),
      parse("1886-04", -0.3615867, -0.5048325, -0.21834087),
      parse("1886-05", -0.29257125, -0.45943964, -0.1257029),
      parse("1886-06", -0.5019767, -0.6763906, -0.32756293),
      parse("1886-07", -0.3021988, -0.45671389, -0.14768374),
      parse("1886-08", -0.45738658, -0.6496851, -0.26508808),
      parse("1886-09", -0.39079213, -0.54562986, -0.2359544),
      parse("1886-10", -0.41297862, -0.56080145, -0.26515582),
      parse("1886-11", -0.49078947, -0.65871966, -0.3228593),
      parse("1886-12", -0.39479798, -0.6054431, -0.18415283),
      parse("1887-01", -0.74377, -0.90897405, -0.57856596),
      parse("1887-02", -0.65888214, -0.81991065, -0.49785364),
      parse("1887-03", -0.44538546, -0.60558623, -0.28518468),
      parse("1887-04", -0.48280382, -0.63975, -0.32585767),
      parse("1887-05", -0.52945954, -0.7083593, -0.35055977),
      parse("1887-06", -0.52584267, -0.7173657, -0.33431968),
      parse("1887-07", -0.38890877, -0.55927885, -0.21853866),
      parse("1887-08", -0.44554248, -0.64114463, -0.24994037),
      parse("1887-09", -0.3885525, -0.55485785, -0.22224714),
      parse("1887-10", -0.4446146, -0.59794164, -0.29128754),
      parse("1887-11", -0.43516165, -0.6090515, -0.26127174),
      parse("1887-12", -0.49650568, -0.70510846, -0.2879029),
      parse("1888-01", -0.6184704, -0.7836917, -0.45324904),
      parse("1888-02", -0.58728784, -0.76135224, -0.41322348),
      parse("1888-03", -0.59725916, -0.77896965, -0.41554868),
      parse("1888-04", -0.35193178, -0.5080489, -0.19581465),
      parse("1888-05", -0.41130793, -0.5864251, -0.23619075),
      parse("1888-06", -0.3217171, -0.50169754, -0.14173672),
      parse("1888-07", -0.3386262, -0.5072375, -0.17001486),
      parse("1888-08", -0.31051376, -0.5074154, -0.11361212),
      parse("1888-09", -0.32469457, -0.49043873, -0.15895042),
      parse("1888-10", -0.16644995, -0.32206133, -0.010838576),
      parse("1888-11", -0.26496226, -0.4413725, -0.08855199),
      parse("1888-12", -0.2593257, -0.47495842, -0.043693),
      parse("1889-01", -0.2966627, -0.47627586, -0.11704954),
      parse("1889-02", -0.006579389, -0.19397688, 0.1808181),
      parse("1889-03", -0.08734918, -0.28068063, 0.10598228),
      parse("1889-04", -0.08679018, -0.25234994, 0.078769594),
      parse("1889-05", -0.15811297, -0.3331547, 0.016928775),
      parse("1889-06", -0.21282648, -0.40597382, -0.019679131),
      parse("1889-07", -0.28090227, -0.46454594, -0.09725862),
      parse("1889-08", -0.30713442, -0.5067254, -0.10754347),
      parse("1889-09", -0.34166074, -0.51179236, -0.17152908),
      parse("1889-10", -0.37001047, -0.5306747, -0.20934623),
      parse("1889-11", -0.5064449, -0.67347455, -0.3394152),
      parse("1889-12", -0.34427297, -0.55167353, -0.1368724),
      parse("1890-01", -0.53390056, -0.68983096, -0.37797013),
      parse("1890-02", -0.46501267, -0.64362186, -0.28640348),
      parse("1890-03", -0.6111193, -0.79531926, -0.42691922),
      parse("1890-04", -0.49407724, -0.6826467, -0.3055078),
      parse("1890-05", -0.53646934, -0.71954244, -0.3533962),
      parse("1890-06", -0.49480304, -0.6899875, -0.2996186),
      parse("1890-07", -0.48914096, -0.6675704, -0.31071147),
      parse("1890-08", -0.46699485, -0.6746316, -0.25935808),
      parse("1890-09", -0.482208, -0.65571135, -0.30870467),
      parse("1890-10", -0.4391856, -0.60413635, -0.27423483),
      parse("1890-11", -0.67008716, -0.8370257, -0.5031486),
      parse("1890-12", -0.3992993, -0.5958885, -0.20271009),
      parse("1891-01", -0.580199, -0.77196914, -0.3884289),
      parse("1891-02", -0.59757084, -0.80028284, -0.3948589),
      parse("1891-03", -0.36251077, -0.55440897, -0.1706126),
      parse("1891-04", -0.4480748, -0.64362466, -0.25252488),
      parse("1891-05", -0.34619382, -0.5355189, -0.15686876),
      parse("1891-06", -0.375095, -0.57121325, -0.17897676),
      parse("1891-07", -0.39966992, -0.5892808, -0.21005905),
      parse("1891-08", -0.3548557, -0.5726065, -0.13710487),
      parse("1891-09", -0.25669602, -0.44973028, -0.06366173),
      parse("1891-10", -0.3429681, -0.5111194, -0.17481679),
      parse("1891-11", -0.5339126, -0.708038, -0.35978726),
      parse("1891-12", -0.21803272, -0.42916322, -0.0069021964),
      parse("1892-01", -0.4647994, -0.65507704, -0.2745218),
      parse("1892-02", -0.2896702, -0.4841962, -0.095144175),
      parse("1892-03", -0.532998, -0.72427106, -0.34172493),
      parse("1892-04", -0.5456076, -0.75018156, -0.34103373),
      parse("1892-05", -0.4568904, -0.64939237, -0.2643884),
      parse("1892-06", -0.43644238, -0.63285375, -0.24003102),
      parse("1892-07", -0.5867592, -0.790647, -0.3828715),
      parse("1892-08", -0.508325, -0.72042894, -0.296221),
      parse("1892-09", -0.3824375, -0.5656438, -0.19923118),
      parse("1892-10", -0.43382177, -0.6145778, -0.25306576),
      parse("1892-11", -0.66069424, -0.84201086, -0.47937757),
      parse("1892-12", -0.7922568, -1.000348, -0.5841656),
      parse("1893-01", -1.0448953, -1.2234055, -0.86638504),
      parse("1893-02", -0.84499276, -1.0261025, -0.6638829),
      parse("1893-03", -0.42680097, -0.62173426, -0.23186766),
      parse("1893-04", -0.5148424, -0.7004083, -0.32927647),
      parse("1893-05", -0.5721378, -0.76604015, -0.3782353),
      parse("1893-06", -0.44231105, -0.6416443, -0.24297778),
      parse("1893-07", -0.24375539, -0.43675295, -0.050757837),
      parse("1893-08", -0.2869543, -0.5068631, -0.06704551),
      parse("1893-09", -0.36513734, -0.5501072, -0.18016747),
      parse("1893-10", -0.30426577, -0.47791195, -0.13061959),
      parse("1893-11", -0.404811, -0.5897336, -0.21988842),
      parse("1893-12", -0.4845268, -0.70177835, -0.26727527),
      parse("1894-01", -0.6276136, -0.8077845, -0.44744277),
      parse("1894-02", -0.42949706, -0.6282197, -0.2307744),
      parse("1894-03", -0.51159126, -0.71238536, -0.3107972),
      parse("1894-04", -0.53747475, -0.7314728, -0.34347677),
      parse("1894-05", -0.5191684, -0.7246258, -0.3137109),
      parse("1894-06", -0.583648, -0.7914774, -0.3758187),
      parse("1894-07", -0.39212567, -0.59586215, -0.1883892),
      parse("1894-08", -0.35357937, -0.570022, -0.13713676),
      parse("1894-09", -0.4396966, -0.6310632, -0.24833001),
      parse("1894-10", -0.3990114, -0.5790243, -0.21899848),
      parse("1894-11", -0.460066, -0.64320624, -0.27692574),
      parse("1894-12", -0.551695, -0.75928974, -0.3441002),
      parse("1895-01", -0.6359875, -0.8076031, -0.46437198),
      parse("1895-02", -0.6417986, -0.83066016, -0.45293707),
      parse("1895-03", -0.6178789, -0.82091254, -0.4148453),
      parse("1895-04", -0.46109405, -0.67555755, -0.24663055),
      parse("1895-05", -0.48253113, -0.68159723, -0.28346506),
      parse("1895-06", -0.4253601, -0.62747884, -0.22324134),
      parse("1895-07", -0.3349007, -0.52899164, -0.14080974),
      parse("1895-08", -0.37724775, -0.5895025, -0.16499303),
      parse("1895-09", -0.39428148, -0.57385796, -0.21470495),
      parse("1895-10", -0.3257419, -0.48164272, -0.16984108),
      parse("1895-11", -0.3213687, -0.48260084, -0.16013654),
      parse("1895-12", -0.36682844, -0.5629477, -0.1707092),
      parse("1896-01", -0.3152051, -0.49154812, -0.13886207),
      parse("1896-02", -0.33662146, -0.5216296, -0.1516134),
      parse("1896-03", -0.58836246, -0.7725723, -0.40415257),
      parse("1896-04", -0.5460314, -0.7311858, -0.3608771),
      parse("1896-05", -0.35611972, -0.57647145, -0.13576801),
      parse("1896-06", -0.23009133, -0.434404, -0.025778683),
      parse("1896-07", -0.18243536, -0.36182913, -0.0030415945),
      parse("1896-08", -0.1715636, -0.3854166, 0.042289402),
      parse("1896-09", -0.15527536, -0.34389502, 0.033344295),
      parse("1896-10", -0.117941946, -0.2807071, 0.044823196),
      parse("1896-11", -0.31405124, -0.48207262, -0.14602986),
      parse("1896-12", -0.094388485, -0.3038262, 0.115049236),
      parse("1897-01", -0.3000984, -0.49080732, -0.10938945),
      parse("1897-02", -0.3486073, -0.5483834, -0.14883117),
      parse("1897-03", -0.41547, -0.61278546, -0.21815456),
      parse("1897-04", -0.1176401, -0.3024626, 0.0671824),
      parse("1897-05", -0.11408224, -0.3108811, 0.08271663),
      parse("1897-06", -0.1627357, -0.36229974, 0.036828335),
      parse("1897-07", -0.18704693, -0.37309894, -0.0009949328),
      parse("1897-08", -0.1854231, -0.40435466, 0.03350848),
      parse("1897-09", -0.18626954, -0.36248967, -0.010049412),
      parse("1897-10", -0.21232262, -0.39015022, -0.03449505),
      parse("1897-11", -0.4227442, -0.5879673, -0.25752115),
      parse("1897-12", -0.46516183, -0.6517685, -0.27855515),
      parse("1898-01", -0.19924784, -0.38865504, -0.009840637),
      parse("1898-02", -0.55612665, -0.71944946, -0.39280388),
      parse("1898-03", -0.8427487, -1.0089464, -0.676551),
      parse("1898-04", -0.663013, -0.84589446, -0.4801315),
      parse("1898-05", -0.55958927, -0.7435597, -0.37561888),
      parse("1898-06", -0.33471388, -0.52292496, -0.1465028),
      parse("1898-07", -0.39475092, -0.58443546, -0.20506638),
      parse("1898-08", -0.3894281, -0.59876347, -0.18009277),
      parse("1898-09", -0.37076962, -0.5485977, -0.19294158),
      parse("1898-10", -0.5009573, -0.6702813, -0.3316333),
      parse("1898-11", -0.5286319, -0.68540967, -0.3718542),
      parse("1898-12", -0.4895283, -0.67102706, -0.30802953),
      parse("1899-01", -0.3598041, -0.5269037, -0.1927045),
      parse("1899-02", -0.5540309, -0.7154645, -0.39259735),
      parse("1899-03", -0.5889544, -0.7476041, -0.43030468),
      parse("1899-04", -0.4393416, -0.61094743, -0.26773578),
      parse("1899-05", -0.35891113, -0.5424037, -0.17541856),
      parse("1899-06", -0.5010004, -0.69523287, -0.306768),
      parse("1899-07", -0.36553654, -0.55425465, -0.17681848),
      parse("1899-08", -0.23979759, -0.45446852, -0.025126649),
      parse("1899-09", -0.1515833, -0.32894266, 0.025776051),
      parse("1899-10", -0.18313703, -0.35176262, -0.014511443),
      parse("1899-11", -0.056940768, -0.2123845, 0.09850298),
      parse("1899-12", -0.466166, -0.6495811, -0.28275096),
      parse("1900-01", -0.5065199, -0.6711187, -0.34192112),
      parse("1900-02", -0.2028605, -0.3608538, -0.044867214),
      parse("1900-03", -0.19967249, -0.36565012, -0.03369488),
      parse("1900-04", -0.22413696, -0.4042493, -0.044024616),
      parse("1900-05", -0.30258486, -0.48975345, -0.11541622),
      parse("1900-06", -0.21710625, -0.4151495, -0.019062985),
      parse("1900-07", -0.19088483, -0.37040052, -0.01136914),
      parse("1900-08", -0.21078074, -0.42371136, 0.002149894),
      parse("1900-09", -0.19568677, -0.38076556, -0.010607976),
      parse("1900-10", -0.0647567, -0.23704362, 0.10753022),
      parse("1900-11", -0.26252335, -0.42287436, -0.10217235),
      parse("1900-12", -0.23623504, -0.41886643, -0.053603664),
      parse("1901-01", -0.28059345, -0.43856356, -0.122623324),
      parse("1901-02", -0.26819772, -0.42274347, -0.113651976),
      parse("1901-03", -0.20081724, -0.36103362, -0.040600866),
      parse("1901-04", -0.1753031, -0.35032505, -0.00028113974),
      parse("1901-05", -0.24934465, -0.44425672, -0.05443258),
      parse("1901-06", -0.23662645, -0.43364984, -0.039603047),
      parse("1901-07", -0.3289457, -0.51798207, -0.13990928),
      parse("1901-08", -0.2050631, -0.42178407, 0.0116578685),
      parse("1901-09", -0.35517338, -0.53581136, -0.1745354),
      parse("1901-10", -0.3407429, -0.5152294, -0.16625641),
      parse("1901-11", -0.39582872, -0.5590094, -0.23264809),
      parse("1901-12", -0.48450628, -0.65788275, -0.3111298),
      parse("1902-01", -0.43956748, -0.6000706, -0.27906436),
      parse("1902-02", -0.23221202, -0.38552827, -0.078895785),
      parse("1902-03", -0.47400042, -0.63162196, -0.31637895),
      parse("1902-04", -0.4305058, -0.6007136, -0.260298),
      parse("1902-05", -0.43023685, -0.6306319, -0.22984181),
      parse("1902-06", -0.44518706, -0.6403324, -0.25004172),
      parse("1902-07", -0.38915876, -0.56928974, -0.2090278),
      parse("1902-08", -0.42510903, -0.63204134, -0.21817671),
      parse("1902-09", -0.37280047, -0.5562161, -0.18938479),
      parse("1902-10", -0.5005036, -0.67379946, -0.32720774),
      parse("1902-11", -0.56261593, -0.72212607, -0.40310577),
      parse("1902-12", -0.565914, -0.7425985, -0.38922948),
      parse("1903-01", -0.4344225, -0.58624995, -0.282595),
      parse("1903-02", -0.24027054, -0.38465306, -0.095888),
      parse("1903-03", -0.39843398, -0.5441603, -0.25270766),
      parse("1903-04", -0.58799607, -0.7538771, -0.42211506),
      parse("1903-05", -0.5160877, -0.7040209, -0.3281545),
      parse("1903-06", -0.5476815, -0.73070943, -0.3646535),
      parse("1903-07", -0.5052094, -0.6767439, -0.33367482),
      parse("1903-08", -0.60140806, -0.81704575, -0.3857704),
      parse("1903-09", -0.55975336, -0.73646456, -0.38304216),
      parse("1903-10", -0.66481555, -0.8263604, -0.50327075),
      parse("1903-11", -0.62978834, -0.781631, -0.4779457),
      parse("1903-12", -0.71404976, -0.8792802, -0.5488193),
      parse("1904-01", -0.806605, -0.9617955, -0.6514145),
      parse("1904-02", -0.769914, -0.92639124, -0.61343676),
      parse("1904-03", -0.6964779, -0.8489669, -0.5439888),
      parse("1904-04", -0.6468306, -0.81834394, -0.47531733),
      parse("1904-05", -0.6093934, -0.7907733, -0.4280136),
      parse("1904-06", -0.5641545, -0.7518275, -0.37648153),
      parse("1904-07", -0.5530446, -0.72540253, -0.38068676),
      parse("1904-08", -0.4784086, -0.68795663, -0.26886058),
      parse("1904-09", -0.59661376, -0.78148705, -0.41174045),
      parse("1904-10", -0.55304295, -0.72068864, -0.3853973),
      parse("1904-11", -0.39660597, -0.5533854, -0.23982657),
      parse("1904-12", -0.49964583, -0.6772394, -0.32205227),
      parse("1905-01", -0.46341595, -0.6148158, -0.31201616),
      parse("1905-02", -0.72016865, -0.8743835, -0.5659538),
      parse("1905-03", -0.52405393, -0.6707999, -0.377308),
      parse("1905-04", -0.5589943, -0.7221077, -0.3958809),
      parse("1905-05", -0.43423653, -0.6155831, -0.25288996),
      parse("1905-06", -0.34258264, -0.52910966, -0.15605558),
      parse("1905-07", -0.37310305, -0.5452026, -0.20100348),
      parse("1905-08", -0.30447677, -0.51597244, -0.09298106),
      parse("1905-09", -0.34519514, -0.5172048, -0.17318553),
      parse("1905-10", -0.3351493, -0.49376118, -0.17653738),
      parse("1905-11", -0.21728525, -0.37075344, -0.06381706),
      parse("1905-12", -0.27435452, -0.45295084, -0.09575821),
      parse("1906-01", -0.29668435, -0.45274007, -0.14062859),
      parse("1906-02", -0.3885811, -0.534501, -0.24266116),
      parse("1906-03", -0.289529, -0.4427584, -0.1362996),
      parse("1906-04", -0.10402012, -0.27304992, 0.06500969),
      parse("1906-05", -0.3765242, -0.55912966, -0.19391878),
      parse("1906-06", -0.23253845, -0.4197746, -0.045302324),
      parse("1906-07", -0.34295923, -0.52357775, -0.1623407),
      parse("1906-08", -0.27118164, -0.48389447, -0.05846878),
      parse("1906-09", -0.35302594, -0.5320011, -0.17405081),
      parse("1906-10", -0.32229662, -0.47954115, -0.16505207),
      parse("1906-11", -0.53179437, -0.68340665, -0.38018218),
      parse("1906-12", -0.32053673, -0.48989052, -0.15118293),
      parse("1907-01", -0.52930236, -0.67252135, -0.3860834),
      parse("1907-02", -0.6185913, -0.76509285, -0.4720898),
      parse("1907-03", -0.43793064, -0.5846249, -0.29123637),
      parse("1907-04", -0.4779983, -0.63595945, -0.32003713),
      parse("1907-05", -0.6450355, -0.8174265, -0.47264454),
      parse("1907-06", -0.5440585, -0.7285098, -0.35960725),
      parse("1907-07", -0.42508972, -0.5989156, -0.25126383),
      parse("1907-08", -0.4910157, -0.7048997, -0.2771317),
      parse("1907-09", -0.3986793, -0.56648517, -0.2308734),
      parse("1907-10", -0.33594123, -0.4869077, -0.18497479),
      parse("1907-11", -0.55637306, -0.7036612, -0.40908492),
      parse("1907-12", -0.58987683, -0.7728595, -0.40689418),
      parse("1908-01", -0.46647665, -0.6135111, -0.3194422),
      parse("1908-02", -0.4139059, -0.55907226, -0.26873952),
      parse("1908-03", -0.7040725, -0.84940577, -0.55873924),
      parse("1908-04", -0.5884082, -0.7462024, -0.43061388),
      parse("1908-05", -0.45306396, -0.6383044, -0.26782355),
      parse("1908-06", -0.41279954, -0.60589045, -0.21970867),
      parse("1908-07", -0.40103948, -0.5724443, -0.22963466),
      parse("1908-08", -0.48125657, -0.7003825, -0.26213074),
      parse("1908-09", -0.3872215, -0.5591857, -0.21525727),
      parse("1908-10", -0.5747282, -0.7261497, -0.42330676),
      parse("1908-11", -0.6453218, -0.7852707, -0.5053729),
      parse("1908-12", -0.63815445, -0.80027384, -0.476035),
      parse("1909-01", -0.7629406, -0.8951454, -0.6307358),
      parse("1909-02", -0.53553337, -0.67432994, -0.39673677),
      parse("1909-03", -0.6521415, -0.79210275, -0.5121803),
      parse("1909-04", -0.63281554, -0.7831975, -0.48243353),
      parse("1909-05", -0.5918205, -0.77018195, -0.413459),
      parse("1909-06", -0.50006735, -0.6869661, -0.31316864),
      parse("1909-07", -0.5237631, -0.6864449, -0.36108136),
      parse("1909-08", -0.3369181, -0.5479727, -0.12586349),
      parse("1909-09", -0.4156719, -0.58178246, -0.24956137),
      parse("1909-10", -0.49451607, -0.64189625, -0.34713593),
      parse("1909-11", -0.38665202, -0.5246769, -0.24862713),
      parse("1909-12", -0.59633845, -0.7501332, -0.44254363),
      parse("1910-01", -0.4327842, -0.5747264, -0.29084206),
      parse("1910-02", -0.5290626, -0.67527723, -0.38284808),
      parse("1910-03", -0.6189373, -0.771704, -0.46617064),
      parse("1910-04", -0.5349184, -0.6914202, -0.37841663),
      parse("1910-05", -0.5368389, -0.71915257, -0.35452524),
      parse("1910-06", -0.5593434, -0.7473471, -0.37133965),
      parse("1910-07", -0.43475804, -0.6054494, -0.2640667),
      parse("1910-08", -0.38878688, -0.6054517, -0.17212208),
      parse("1910-09", -0.4229382, -0.59759563, -0.24828072),
      parse("1910-10", -0.48481956, -0.63360226, -0.33603686),
      parse("1910-11", -0.672429, -0.81315607, -0.531702),
      parse("1910-12", -0.7566741, -0.91549015, -0.59785813),
      parse("1911-01", -0.66804653, -0.8043503, -0.5317427),
      parse("1911-02", -0.70603365, -0.838752, -0.5733153),
      parse("1911-03", -0.7639851, -0.89808047, -0.62988967),
      parse("1911-04", -0.64364564, -0.80100137, -0.48628986),
      parse("1911-05", -0.6272078, -0.805938, -0.44847766),
      parse("1911-06", -0.55432284, -0.74320203, -0.36544362),
      parse("1911-07", -0.49260998, -0.6690713, -0.3161486),
      parse("1911-08", -0.5064451, -0.7171781, -0.29571208),
      parse("1911-09", -0.4982411, -0.680748, -0.3157342),
      parse("1911-10", -0.34511527, -0.49507046, -0.19516012),
      parse("1911-11", -0.34931326, -0.49439368, -0.20423283),
      parse("1911-12", -0.315495, -0.47613657, -0.1548535),
      parse("1912-01", -0.32940087, -0.4633201, -0.19548161),
      parse("1912-02", -0.31491742, -0.43766788, -0.19216692),
      parse("1912-03", -0.5543457, -0.68546075, -0.4232307),
      parse("1912-04", -0.33788237, -0.4915007, -0.184264),
      parse("1912-05", -0.3352509, -0.503716, -0.1667858),
      parse("1912-06", -0.28508794, -0.4639832, -0.106192656),
      parse("1912-07", -0.50730854, -0.6681659, -0.34645116),
      parse("1912-08", -0.57247293, -0.78099567, -0.36395016),
      parse("1912-09", -0.6437677, -0.80863905, -0.47889635),
      parse("1912-10", -0.6791174, -0.82347953, -0.53475523),
      parse("1912-11", -0.55792326, -0.6956708, -0.42017576),
      parse("1912-12", -0.59060127, -0.74457186, -0.43663067),
      parse("1913-01", -0.5636491, -0.70592016, -0.42137802),
      parse("1913-02", -0.55008626, -0.6813429, -0.41882965),
      parse("1913-03", -0.60579985, -0.7407136, -0.47088605),
      parse("1913-04", -0.5340449, -0.69886494, -0.36922488),
      parse("1913-05", -0.54116863, -0.717569, -0.36476827),
      parse("1913-06", -0.5514844, -0.73894113, -0.36402768),
      parse("1913-07", -0.47697932, -0.649064, -0.30489463),
      parse("1913-08", -0.39560005, -0.605102, -0.18609805),
      parse("1913-09", -0.41690665, -0.5830773, -0.25073597),
      parse("1913-10", -0.43836418, -0.5882755, -0.28845286),
      parse("1913-11", -0.36772293, -0.50663286, -0.22881301),
      parse("1913-12", -0.16402407, -0.3207319, -0.007316216),
      parse("1914-01", -0.078474745, -0.21613824, 0.05918875),
      parse("1914-02", -0.27010897, -0.4004639, -0.13975406),
      parse("1914-03", -0.39748344, -0.5322073, -0.26275954),
      parse("1914-04", -0.4089052, -0.5571737, -0.26063666),
      parse("1914-05", -0.30789974, -0.47892362, -0.13687588),
      parse("1914-06", -0.30798316, -0.49093956, -0.12502678),
      parse("1914-07", -0.2969515, -0.4692706, -0.124632396),
      parse("1914-08", -0.24459992, -0.4556622, -0.033537664),
      parse("1914-09", -0.30140656, -0.48765177, -0.11516135),
      parse("1914-10", -0.110274434, -0.2672808, 0.046731923),
      parse("1914-11", -0.19029672, -0.33736014, -0.04323329),
      parse("1914-12", -0.23672462, -0.39467424, -0.078774974),
      parse("1915-01", -0.28745866, -0.43606412, -0.13885316),
      parse("1915-02", -0.22042616, -0.37010035, -0.070751965),
      parse("1915-03", -0.19399805, -0.35105386, -0.036942244),
      parse("1915-04", -0.082622275, -0.24781929, 0.08257474),
      parse("1915-05", -0.16808057, -0.36465895, 0.028497813),
      parse("1915-06", -0.17210917, -0.37255874, 0.028340416),
      parse("1915-07", -0.16903177, -0.36015847, 0.02209495),
      parse("1915-08", -0.06681827, -0.29493347, 0.16129693),
      parse("1915-09", -0.120297745, -0.30047518, 0.05987967),
      parse("1915-10", -0.28610688, -0.44976383, -0.12244993),
      parse("1915-11", -0.17281866, -0.3218076, -0.023829734),
      parse("1915-12", -0.3623588, -0.50942546, -0.21529216),
      parse("1916-01", -0.33786005, -0.4876082, -0.1881119),
      parse("1916-02", -0.26403636, -0.41823655, -0.10983616),
      parse("1916-03", -0.43909237, -0.5887705, -0.28941423),
      parse("1916-04", -0.37517795, -0.5474576, -0.20289832),
      parse("1916-05", -0.41811523, -0.62675697, -0.20947349),
      parse("1916-06", -0.49483928, -0.70778334, -0.28189525),
      parse("1916-07", -0.38488427, -0.58387554, -0.18589298),
      parse("1916-08", -0.29676017, -0.52024907, -0.07327126),
      parse("1916-09", -0.3218673, -0.5078389, -0.13589564),
      parse("1916-10", -0.3744158, -0.54190564, -0.20692603),
      parse("1916-11", -0.5142024, -0.6644174, -0.3639875),
      parse("1916-12", -0.8212685, -0.9761314, -0.6664057),
      parse("1917-01", -0.6892141, -0.84988326, -0.528545),
      parse("1917-02", -0.715982, -0.8849225, -0.54704154),
      parse("1917-03", -0.78429514, -0.94445986, -0.6241304),
      parse("1917-04", -0.5461108, -0.726568, -0.3656537),
      parse("1917-05", -0.6442835, -0.8495297, -0.43903726),
      parse("1917-06", -0.47886908, -0.698331, -0.2594072),
      parse("1917-07", -0.24527456, -0.455212, -0.035337124),
      parse("1917-08", -0.33282208, -0.5884459, -0.07719827),
      parse("1917-09", -0.2979707, -0.49015912, -0.10578233),
      parse("1917-10", -0.53561026, -0.7070309, -0.36418962),
      parse("1917-11", -0.45924768, -0.6210569, -0.29743847),
      parse("1917-12", -0.7865544, -0.9670814, -0.6060274),
      parse("1918-01", -0.5179739, -0.6850022, -0.3509456),
      parse("1918-02", -0.67216516, -0.8356719, -0.5086584),
      parse("1918-03", -0.57906806, -0.751414, -0.4067221),
      parse("1918-04", -0.6234592, -0.8156268, -0.4312916),
      parse("1918-05", -0.5252177, -0.75086904, -0.29956633),
      parse("1918-06", -0.38651752, -0.6103386, -0.16269644),
      parse("1918-07", -0.36958522, -0.56567657, -0.17349385),
      parse("1918-08", -0.42982036, -0.66980463, -0.18983606),
      parse("1918-09", -0.25825897, -0.4637045, -0.052813478),
      parse("1918-10", -0.11888721, -0.31273952, 0.07496512),
      parse("1918-11", -0.18980259, -0.38338757, 0.0037823957),
      parse("1918-12", -0.42425612, -0.6094721, -0.23904014),
      parse("1919-01", -0.25126135, -0.435014, -0.067508675),
      parse("1919-02", -0.24207273, -0.416542, -0.06760347),
      parse("1919-03", -0.39209825, -0.58520657, -0.19898996),
      parse("1919-04", -0.21141803, -0.41892788, -0.0039082),
      parse("1919-05", -0.30793867, -0.5225244, -0.093352936),
      parse("1919-06", -0.35769194, -0.5655215, -0.14986245),
      parse("1919-07", -0.29989257, -0.51161784, -0.08816732),
      parse("1919-08", -0.2381391, -0.4837357, 0.0074575483),
      parse("1919-09", -0.28606114, -0.48273113, -0.08939117),
      parse("1919-10", -0.25880492, -0.44467, -0.07293986),
      parse("1919-11", -0.5575256, -0.71599007, -0.39906105),
      parse("1919-12", -0.5033144, -0.67652005, -0.3301087),
      parse("1920-01", -0.25863704, -0.43835065, -0.07892343),
      parse("1920-02", -0.545883, -0.69912046, -0.39264554),
      parse("1920-03", -0.17207259, -0.3437741, -0.00037105498),
      parse("1920-04", -0.33658424, -0.5365327, -0.13663577),
      parse("1920-05", -0.25627613, -0.46498504, -0.047567245),
      parse("1920-06", -0.23811027, -0.45812166, -0.018098898),
      parse("1920-07", -0.30046934, -0.5018703, -0.09906845),
      parse("1920-08", -0.22659715, -0.45900223, 0.005807938),
      parse("1920-09", -0.1964555, -0.3914321, -0.0014788725),
      parse("1920-10", -0.21776049, -0.38453242, -0.050988548),
      parse("1920-11", -0.3456471, -0.52150667, -0.16978753),
      parse("1920-12", -0.4884766, -0.6523202, -0.32463297),
      parse("1921-01", -0.21760732, -0.37502497, -0.06018965),
      parse("1921-02", -0.29521245, -0.44902927, -0.14139564),
      parse("1921-03", -0.34263408, -0.50093555, -0.18433265),
      parse("1921-04", -0.29096943, -0.46709645, -0.11484241),
      parse("1921-05", -0.30159548, -0.5012052, -0.10198575),
      parse("1921-06", -0.15440017, -0.3658895, 0.057089157),
      parse("1921-07", -0.15540703, -0.3545471, 0.043733045),
      parse("1921-08", -0.32134694, -0.5458187, -0.09687517),
      parse("1921-09", -0.20710628, -0.39442888, -0.019783666),
      parse("1921-10", -0.13444184, -0.30367956, 0.034795877),
      parse("1921-11", -0.19592439, -0.3460785, -0.04577027),
      parse("1921-12", -0.27147904, -0.424055, -0.11890304),
      parse("1922-01", -0.419077, -0.56817025, -0.2699838),
      parse("1922-02", -0.40880406, -0.5457496, -0.2718585),
      parse("1922-03", -0.344148, -0.4943077, -0.19398831),
      parse("1922-04", -0.29111138, -0.45628148, -0.12594129),
      parse("1922-05", -0.46964964, -0.6579521, -0.28134724),
      parse("1922-06", -0.29270366, -0.48048604, -0.104921274),
      parse("1922-07", -0.2590808, -0.44124886, -0.07691275),
      parse("1922-08", -0.355893, -0.5667185, -0.14506747),
      parse("1922-09", -0.32432133, -0.49675274, -0.15188992),
      parse("1922-10", -0.34514487, -0.49858475, -0.19170502),
      parse("1922-11", -0.25539744, -0.39435166, -0.116443194),
      parse("1922-12", -0.30540624, -0.4507393, -0.16007322),
      parse("1923-01", -0.40372258, -0.54395944, -0.26348573),
      parse("1923-02", -0.4944323, -0.6337809, -0.3550837),
      parse("1923-03", -0.45562357, -0.5954459, -0.3158013),
      parse("1923-04", -0.41038665, -0.57101375, -0.24975957),
      parse("1923-05", -0.4038749, -0.5846617, -0.22308807),
      parse("1923-06", -0.25269112, -0.4406215, -0.06476072),
      parse("1923-07", -0.41963744, -0.5942383, -0.2450366),
      parse("1923-08", -0.32384917, -0.537738, -0.10996033),
      parse("1923-09", -0.34093276, -0.5162632, -0.16560236),
      parse("1923-10", -0.1764545, -0.33028832, -0.022620698),
      parse("1923-11", -0.03351226, -0.17447731, 0.107452795),
      parse("1923-12", -0.10004945, -0.24315248, 0.043053582),
      parse("1924-01", -0.32711104, -0.47252598, -0.18169609),
      parse("1924-02", -0.31528232, -0.45562285, -0.17494178),
      parse("1924-03", -0.17595449, -0.3153586, -0.036550395),
      parse("1924-04", -0.34247932, -0.5014861, -0.18347253),
      parse("1924-05", -0.21757317, -0.39559406, -0.03955228),
      parse("1924-06", -0.27555224, -0.46537247, -0.085732006),
      parse("1924-07", -0.29391536, -0.46117327, -0.12665746),
      parse("1924-08", -0.3172419, -0.5321844, -0.102299385),
      parse("1924-09", -0.33069578, -0.5084952, -0.15289639),
      parse("1924-10", -0.36585054, -0.5206516, -0.2110495),
      parse("1924-11", -0.2907367, -0.4303612, -0.15111223),
      parse("1924-12", -0.49235374, -0.63418114, -0.35052633),
      parse("1925-01", -0.49163565, -0.6360688, -0.34720248),
      parse("1925-02", -0.48531267, -0.62783724, -0.34278813),
      parse("1925-03", -0.37375176, -0.5155008, -0.23200274),
      parse("1925-04", -0.30694255, -0.47034937, -0.14353575),
      parse("1925-05", -0.3056574, -0.5006012, -0.11071356),
      parse("1925-06", -0.33413643, -0.53678447, -0.13148837),
      parse("1925-07", -0.33086437, -0.5057325, -0.15599626),
      parse("1925-08", -0.16828516, -0.38384804, 0.04727771),
      parse("1925-09", -0.25893372, -0.44479072, -0.0730767),
      parse("1925-10", -0.28853402, -0.45646706, -0.120601006),
      parse("1925-11", -0.059098415, -0.20052361, 0.082326785),
      parse("1925-12", 0.014048895, -0.13270758, 0.16080537),
      parse("1926-01", 0.08253241, -0.06263089, 0.22769572),
      parse("1926-02", -0.035031483, -0.17242493, 0.102361955),
      parse("1926-03", 0.023573108, -0.11576544, 0.16291165),
      parse("1926-04", -0.1579929, -0.327664, 0.01167819),
      parse("1926-05", -0.27003446, -0.4538862, -0.08618276),
      parse("1926-06", -0.18281585, -0.3733924, 0.0077607087),
      parse("1926-07", -0.2084477, -0.38436303, -0.03253235),
      parse("1926-08", -0.10001693, -0.32225892, 0.12222505),
      parse("1926-09", -0.11788381, -0.2965996, 0.06083199),
      parse("1926-10", -0.06996329, -0.22726479, 0.08733821),
      parse("1926-11", -0.115788996, -0.25999144, 0.028413441),
      parse("1926-12", -0.32215574, -0.46747088, -0.17684063),
      parse("1927-01", -0.31123906, -0.45435637, -0.16812176),
      parse("1927-02", -0.21210012, -0.33594212, -0.08825814),
      parse("1927-03", -0.41895348, -0.5508106, -0.28709638),
      parse("1927-04", -0.3145047, -0.4677973, -0.16121212),
      parse("1927-05", -0.23004514, -0.40716198, -0.05292829),
      parse("1927-06", -0.24831119, -0.43623513, -0.06038725),
      parse("1927-07", -0.20484532, -0.37125188, -0.038438786),
      parse("1927-08", -0.21330415, -0.42017597, -0.0064323),
      parse("1927-09", -0.11964897, -0.28789678, 0.04859885),
      parse("1927-10", -0.02967228, -0.18330196, 0.1239574),
      parse("1927-11", -0.12667355, -0.25778046, 0.0044333646),
      parse("1927-12", -0.32356292, -0.46620953, -0.18091631),
      parse("1928-01", -0.017104585, -0.1668746, 0.13266544),
      parse("1928-02", -0.124669254, -0.2695675, 0.020228995),
      parse("1928-03", -0.3195826, -0.46694115, -0.17222404),
      parse("1928-04", -0.23520902, -0.39391342, -0.076504625),
      parse("1928-05", -0.3014689, -0.48644242, -0.11649541),
      parse("1928-06", -0.32672966, -0.51603055, -0.1374288),
      parse("1928-07", -0.1883196, -0.35774884, -0.018890347),
      parse("1928-08", -0.17851628, -0.38754088, 0.030508325),
      parse("1928-09", -0.20668688, -0.38965768, -0.02371608),
      parse("1928-10", -0.1840448, -0.34064317, -0.027446417),
      parse("1928-11", -0.17209516, -0.31970295, -0.024487386),
      parse("1928-12", -0.22671194, -0.37439686, -0.07902704),
      parse("1929-01", -0.4364513, -0.5833095, -0.28959316),
      parse("1929-02", -0.6036853, -0.74768084, -0.45968977),
      parse("1929-03", -0.48715258, -0.63443154, -0.33987364),
      parse("1929-04", -0.4373352, -0.60708904, -0.2675813),
      parse("1929-05", -0.4345294, -0.62124, -0.24781875),
      parse("1929-06", -0.40885338, -0.5975827, -0.22012407),
      parse("1929-07", -0.4396209, -0.6168994, -0.26234248),
      parse("1929-08", -0.2663079, -0.48695058, -0.04566523),
      parse("1929-09", -0.27733904, -0.46059266, -0.09408544),
      parse("1929-10", -0.22126578, -0.38139418, -0.06113739),
      parse("1929-11", -0.14017749, -0.28305233, 0.0026973686),
      parse("1929-12", -0.5603613, -0.7042419, -0.4164808),
      parse("1930-01", -0.3631465, -0.49787483, -0.22841819),
      parse("1930-02", -0.35584554, -0.4784241, -0.23326696),
      parse("1930-03", -0.24732758, -0.3744386, -0.12021657),
      parse("1930-04", -0.23128887, -0.38058066, -0.08199705),
      parse("1930-05", -0.25067642, -0.4180869, -0.08326598),
      parse("1930-06", -0.20374553, -0.3809626, -0.026528457),
      parse("1930-07", -0.1882644, -0.36365265, -0.012876141),
      parse("1930-08", -0.09765256, -0.3049344, 0.10962929),
      parse("1930-09", -0.092231415, -0.25912735, 0.07466452),
      parse("1930-10", -0.090577535, -0.23271994, 0.05156486),
      parse("1930-11", 0.06376379, -0.063620925, 0.1911485),
      parse("1930-12", -0.06467221, -0.18759245, 0.058248024),
      parse("1931-01", -0.13693942, -0.2671767, -0.006702163),
      parse("1931-02", -0.2586906, -0.36686972, -0.15051147),
      parse("1931-03", -0.10919364, -0.23315944, 0.014772162),
      parse("1931-04", -0.17070459, -0.31463963, -0.026769554),
      parse("1931-05", -0.19289996, -0.36193624, -0.023863683),
      parse("1931-06", -0.02544678, -0.2039998, 0.15310624),
      parse("1931-07", -0.008178203, -0.17845777, 0.16210136),
      parse("1931-08", -0.026662722, -0.23960268, 0.18627724),
      parse("1931-09", -0.051968556, -0.2076939, 0.10375679),
      parse("1931-10", -0.026985472, -0.17087266, 0.11690171),
      parse("1931-11", -0.13579522, -0.26063263, -0.010957824),
      parse("1931-12", -0.09730702, -0.2172313, 0.022617267),
      parse("1932-01", 0.07975415, -0.043561757, 0.20307006),
      parse("1932-02", -0.17805362, -0.29942062, -0.056686606),
      parse("1932-03", -0.1947655, -0.3203073, -0.06922365),
      parse("1932-04", -0.061219946, -0.198662, 0.076222114),
      parse("1932-05", -0.15982142, -0.3201136, 0.00047073714),
      parse("1932-06", -0.25541136, -0.4258154, -0.0850073),
      parse("1932-07", -0.12971984, -0.29410174, 0.034662075),
      parse("1932-08", -0.18982057, -0.39686376, 0.017222626),
      parse("1932-09", -0.00055066624, -0.15410176, 0.15300043),
      parse("1932-10", -0.09439127, -0.23334253, 0.044559985),
      parse("1932-11", -0.29486138, -0.41643777, -0.17328496),
      parse("1932-12", -0.26667866, -0.38838974, -0.14496759),
      parse("1933-01", -0.31109115, -0.4301322, -0.1920501),
      parse("1933-02", -0.35099408, -0.46522236, -0.23676579),
      parse("1933-03", -0.3851551, -0.5024361, -0.2678741),
      parse("1933-04", -0.28470516, -0.42601293, -0.14339739),
      parse("1933-05", -0.2769364, -0.45054585, -0.10332694),
      parse("1933-06", -0.32360965, -0.5103935, -0.13682581),
      parse("1933-07", -0.2903949, -0.4524689, -0.1283209),
      parse("1933-08", -0.15375972, -0.3546877, 0.047168247),
      parse("1933-09", -0.32664743, -0.48648426, -0.1668106),
      parse("1933-10", -0.245828, -0.38623607, -0.10541995),
      parse("1933-11", -0.40483868, -0.51923364, -0.29044378),
      parse("1933-12", -0.5141726, -0.63299805, -0.3953472),
      parse("1934-01", -0.32326755, -0.43162593, -0.21490915),
      parse("1934-02", -0.1451731, -0.24469295, -0.045653235),
      parse("1934-03", -0.4310569, -0.5399451, -0.32216865),
      parse("1934-04", -0.37176266, -0.50687045, -0.23665486),
      parse("1934-05", -0.14018929, -0.29952848, 0.019149903),
      parse("1934-06", -0.09232751, -0.28172815, 0.09707313),
      parse("1934-07", -0.17809097, -0.33932257, -0.016859364),
      parse("1934-08", -0.11440116, -0.3252075, 0.09640516),
      parse("1934-09", -0.13450967, -0.29044768, 0.021428343),
      parse("1934-10", -0.08177011, -0.21969576, 0.056155562),
      parse("1934-11", 0.031826723, -0.08049537, 0.14414883),
      parse("1934-12", -0.111320004, -0.22109856, -0.0015414461),
      parse("1935-01", -0.3479727, -0.46686283, -0.22908255),
      parse("1935-02", 0.04972145, -0.053259466, 0.15270236),
      parse("1935-03", -0.21446298, -0.31988317, -0.10904279),
      parse("1935-04", -0.3228376, -0.460522, -0.18515322),
      parse("1935-05", -0.31434223, -0.48563308, -0.1430514),
      parse("1935-06", -0.2092866, -0.38938758, -0.029185612),
      parse("1935-07", -0.14094204, -0.2998374, 0.017953362),
      parse("1935-08", -0.14877552, -0.34538043, 0.04782942),
      parse("1935-09", -0.19052882, -0.3428319, -0.03822576),
      parse("1935-10", -0.055213235, -0.18450885, 0.07408237),
      parse("1935-11", -0.32153568, -0.42853072, -0.21454066),
      parse("1935-12", -0.25653476, -0.36094522, -0.15212429),
      parse("1936-01", -0.39848116, -0.5024208, -0.2945415),
      parse("1936-02", -0.34311983, -0.44220343, -0.24403626),
      parse("1936-03", -0.32916442, -0.43601322, -0.22231562),
      parse("1936-04", -0.1883213, -0.31698588, -0.059656713),
      parse("1936-05", -0.17232911, -0.33633974, -0.008318501),
      parse("1936-06", -0.14899857, -0.31470138, 0.016704235),
      parse("1936-07", -0.06987007, -0.22382125, 0.084081106),
      parse("1936-08", -0.07327964, -0.27147797, 0.12491867),
      parse("1936-09", -0.108766146, -0.25690895, 0.03937666),
      parse("1936-10", -0.041382577, -0.17508227, 0.092317104),
      parse("1936-11", -0.054110777, -0.15767384, 0.049452282),
      parse("1936-12", -0.10642759, -0.21628995, 0.0034347898),
      parse("1937-01", -0.12685171, -0.23948237, -0.014221046),
      parse("1937-02", 0.066486835, -0.03919336, 0.17216702),
      parse("1937-03", -0.26192847, -0.37137395, -0.15248302),
      parse("1937-04", -0.121213466, -0.255947, 0.013520067),
      parse("1937-05", -0.091207266, -0.25534177, 0.07292723),
      parse("1937-06", -0.008428331, -0.18466923, 0.16781256),
      parse("1937-07", 0.019690081, -0.13141476, 0.17079492),
      parse("1937-08", 0.036828596, -0.15785031, 0.2315075),
      parse("1937-09", 0.14490855, -0.010420173, 0.30023727),
      parse("1937-10", 0.12801065, -0.011122891, 0.2671442),
      parse("1937-11", 0.061703082, -0.050341, 0.17374717),
      parse("1937-12", -0.0783857, -0.19385788, 0.037086464),
      parse("1938-01", 0.04551444, -0.05937577, 0.15040465),
      parse("1938-02", -0.009824214, -0.11000702, 0.090358585),
      parse("1938-03", -0.015279184, -0.117580466, 0.087022096),
      parse("1938-04", 0.04000326, -0.0976628, 0.17766932),
      parse("1938-05", -0.09591497, -0.2707099, 0.07887996),
      parse("1938-06", -0.09364482, -0.26718155, 0.0798919),
      parse("1938-07", -0.04622565, -0.20220377, 0.109752476),
      parse("1938-08", -0.0028444168, -0.20122032, 0.19553149),
      parse("1938-09", 0.01991738, -0.13494068, 0.17477545),
      parse("1938-10", 0.16282216, 0.024466746, 0.30117756),
      parse("1938-11", 0.054737236, -0.053614, 0.16308847),
      parse("1938-12", -0.20567, -0.32208803, -0.089251965),
      parse("1939-01", -0.121787794, -0.22934918, -0.014226404),
      parse("1939-02", -0.117604576, -0.21543846, -0.019770693),
      parse("1939-03", -0.28873166, -0.39380893, -0.18365441),
      parse("1939-04", -0.106402785, -0.23740795, 0.024602382),
      parse("1939-05", -0.03005215, -0.19047527, 0.13037097),
      parse("1939-06", 0.014760026, -0.16028467, 0.18980472),
      parse("1939-07", -0.037748978, -0.20127435, 0.1257764),
      parse("1939-08", -0.04066755, -0.25702047, 0.17568538),
      parse("1939-09", -0.017160367, -0.20016287, 0.16584213),
      parse("1939-10", -0.09934573, -0.26764223, 0.06895078),
      parse("1939-11", 0.021302136, -0.11902354, 0.16162781),
      parse("1939-12", 0.33387345, 0.18944654, 0.47830033),
      parse("1940-01", -0.052654818, -0.18006782, 0.07475819),
      parse("1940-02", 0.08632718, -0.0359212, 0.20857556),
      parse("1940-03", -0.035766255, -0.1625376, 0.09100509),
      parse("1940-04", 0.14895241, -0.034468964, 0.33237377),
      parse("1940-05", 0.0502412, -0.14184676, 0.24232917),
      parse("1940-06", 0.044663727, -0.1491368, 0.23846425),
      parse("1940-07", 0.1482532, -0.028301956, 0.32480836),
      parse("1940-08", 0.09961082, -0.115770265, 0.31499192),
      parse("1940-09", 0.13118677, -0.03203977, 0.2944133),
      parse("1940-10", 0.09584135, -0.059646063, 0.25132877),
      parse("1940-11", -0.0046561854, -0.13411191, 0.12479955),
      parse("1940-12", 0.19923072, 0.06407699, 0.33438444),
      parse("1941-01", -0.039560955, -0.25157484, 0.17245293),
      parse("1941-02", 0.11823889, -0.08517223, 0.32165),
      parse("1941-03", -0.11652528, -0.32683504, 0.09378447),
      parse("1941-04", 0.036228664, -0.19644469, 0.268902),
      parse("1941-05", 0.06123504, -0.19559848, 0.31806856),
      parse("1941-06", 0.0596246, -0.18859485, 0.30784404),
      parse("1941-07", 0.16075781, -0.07209745, 0.39361307),
      parse("1941-08", 0.039676894, -0.22813907, 0.30749285),
      parse("1941-09", -0.14452067, -0.3784344, 0.08939303),
      parse("1941-10", 0.21426645, -0.0057034367, 0.43423632),
      parse("1941-11", -0.007257905, -0.22418799, 0.20967218),
      parse("1941-12", 0.07538852, -0.13360447, 0.2843815),
      parse("1942-01", 0.25783694, 0.053692356, 0.4619815),
      parse("1942-02", 0.016801072, -0.18281478, 0.21641691),
      parse("1942-03", -0.041579403, -0.24019195, 0.15703313),
      parse("1942-04", -0.015153615, -0.24485067, 0.21454343),
      parse("1942-05", 0.12379065, -0.13554372, 0.38312504),
      parse("1942-06", 0.030325836, -0.2195993, 0.28025097),
      parse("1942-07", -0.059752498, -0.2964828, 0.1769778),
      parse("1942-08", -0.05740612, -0.3351613, 0.22034906),
      parse("1942-09", -0.013465349, -0.25318065, 0.22624996),
      parse("1942-10", -0.08694347, -0.30974555, 0.13585861),
      parse("1942-11", -0.057510037, -0.27184075, 0.15682067),
      parse("1942-12", -0.080070905, -0.2823254, 0.122183576),
      parse("1943-01", -0.18267323, -0.41188934, 0.046542883),
      parse("1943-02", -0.03061434, -0.24937429, 0.18814561),
      parse("1943-03", -0.16772735, -0.39681655, 0.061361846),
      parse("1943-04", 0.022969417, -0.20856993, 0.25450876),
      parse("1943-05", 0.03550913, -0.23148823, 0.30250648),
      parse("1943-06", -0.07487101, -0.33023354, 0.1804915),
      parse("1943-07", 0.018335527, -0.22184484, 0.2585159),
      parse("1943-08", 0.031610876, -0.24861293, 0.31183466),
      parse("1943-09", -0.013441508, -0.24750064, 0.22061762),
      parse("1943-10", 0.2277136, -0.0019893488, 0.45741653),
      parse("1943-11", 0.10388872, -0.115436934, 0.32321438),
      parse("1943-12", 0.10626907, -0.10569832, 0.31823647),
      parse("1944-01", 0.31816214, 0.112214126, 0.52411014),
      parse("1944-02", 0.11862034, -0.08507213, 0.32231283),
      parse("1944-03", 0.12006746, -0.09471434, 0.33484927),
      parse("1944-04", 0.04225588, -0.1885682, 0.27307996),
      parse("1944-05", 0.119925626, -0.13840115, 0.3782524),
      parse("1944-06", 0.10735916, -0.15293637, 0.3676547),
      parse("1944-07", 0.14871696, -0.0872223, 0.3846562),
      parse("1944-08", 0.20666182, -0.079099305, 0.49242294),
      parse("1944-09", 0.16769068, -0.056805864, 0.39218724),
      parse("1944-10", 0.20805594, -0.0102054365, 0.42631733),
      parse("1944-11", 0.08847658, -0.1139502, 0.29090336),
      parse("1944-12", 0.083269, -0.112689644, 0.27922764),
      parse("1945-01", 0.09508047, -0.11957911, 0.30974007),
      parse("1945-02", 0.04839935, -0.16548169, 0.2622804),
      parse("1945-03", 0.03865824, -0.17728569, 0.25460216),
      parse("1945-04", 0.20985286, -0.02450203, 0.44420776),
      parse("1945-05", -0.061685707, -0.33644745, 0.21307604),
      parse("1945-06", -0.03652022, -0.29726994, 0.22422948),
      parse("1945-07", 0.00042324737, -0.24768649, 0.24853298),
      parse("1945-08", 0.2362124, -0.053284284, 0.5257091),
      parse("1945-09", 0.1028386, -0.122961916, 0.32863912),
      parse("1945-10", 0.094233386, -0.13283221, 0.321299),
      parse("1945-11", -0.010639086, -0.22343569, 0.20215751),
      parse("1945-12", -0.19979319, -0.40156344, 0.0019770549),
      parse("1946-01", 0.046495322, -0.12599254, 0.21898319),
      parse("1946-02", -0.11574643, -0.2901115, 0.058618642),
      parse("1946-03", -0.1215551, -0.2931895, 0.050079297),
      parse("1946-04", -0.001070216, -0.18591501, 0.18377458),
      parse("1946-05", -0.082718804, -0.28555256, 0.120114945),
      parse("1946-06", -0.27002892, -0.4794503, -0.06060755),
      parse("1946-07", -0.109389625, -0.31101307, 0.09223383),
      parse("1946-08", -0.15412526, -0.38707858, 0.07882805),
      parse("1946-09", -0.09819822, -0.28867555, 0.09227911),
      parse("1946-10", -0.06812635, -0.24190964, 0.10565692),
      parse("1946-11", -0.10808773, -0.26652718, 0.05035172),
      parse("1946-12", -0.34320223, -0.48636523, -0.20003924),
      parse("1947-01", -0.15051101, -0.29601136, -0.0050106677),
      parse("1947-02", -0.14106397, -0.27542192, -0.006706007),
      parse("1947-03", -0.07618291, -0.22355233, 0.07118651),
      parse("1947-04", -0.054869834, -0.23009242, 0.120352745),
      parse("1947-05", -0.07080105, -0.27533346, 0.13373137),
      parse("1947-06", -0.092732966, -0.3124098, 0.12694384),
      parse("1947-07", -0.06481824, -0.25643027, 0.12679379),
      parse("1947-08", -0.11288264, -0.34558374, 0.11981848),
      parse("1947-09", -0.16043018, -0.35566252, 0.034802165),
      parse("1947-10", 0.033811674, -0.14890239, 0.21652573),
      parse("1947-11", 0.02912334, -0.12750946, 0.18575613),
      parse("1947-12", -0.23310879, -0.38460296, -0.08161461),
      parse("1948-01", -0.01899665, -0.15659407, 0.11860077),
      parse("1948-02", -0.23857601, -0.37797922, -0.09917279),
      parse("1948-03", -0.28302348, -0.4262124, -0.13983452),
      parse("1948-04", -0.114602394, -0.287611, 0.05840621),
      parse("1948-05", 0.057578247, -0.14908782, 0.26424432),
      parse("1948-06", -0.008215971, -0.20537749, 0.18894555),
      parse("1948-07", -0.12959683, -0.32280955, 0.0636159),
      parse("1948-08", -0.047574744, -0.27386165, 0.17871216),
      parse("1948-09", -0.13380374, -0.31497514, 0.047367666),
      parse("1948-10", -0.05680375, -0.2210213, 0.1074138),
      parse("1948-11", -0.17333214, -0.3244153, -0.022249011),
      parse("1948-12", -0.3489878, -0.49336633, -0.20460922),
      parse("1949-01", -0.018578093, -0.14231202, 0.10515583),
      parse("1949-02", -0.24225992, -0.3602003, -0.12431957),
      parse("1949-03", -0.10957818, -0.2236049, 0.0044485494),
      parse("1949-04", -0.14671299, -0.2947693, 0.0013433092),
      parse("1949-05", -0.11125475, -0.29777372, 0.07526421),
      parse("1949-06", -0.23316567, -0.41350195, -0.052829392),
      parse("1949-07", -0.18007946, -0.34748125, -0.01267766),
      parse("1949-08", -0.07750365, -0.29055914, 0.13555184),
      parse("1949-09", -0.123270296, -0.28533342, 0.03879283),
      parse("1949-10", -0.08973549, -0.22897226, 0.04950127),
      parse("1949-11", -0.1810419, -0.30046904, -0.061614756),
      parse("1949-12", -0.21244651, -0.32703057, -0.09786245),
      parse("1950-01", -0.30004355, -0.41899204, -0.18109505),
      parse("1950-02", -0.37003636, -0.4831021, -0.2569706),
      parse("1950-03", -0.21643834, -0.3307996, -0.10207707),
      parse("1950-04", -0.24361573, -0.39951792, -0.08771354),
      parse("1950-05", -0.13729778, -0.31324044, 0.038644888),
      parse("1950-06", -0.105043374, -0.29483223, 0.084745474),
      parse("1950-07", -0.10670231, -0.26735002, 0.05394541),
      parse("1950-08", -0.15261763, -0.35921878, 0.053983524),
      parse("1950-09", -0.24503586, -0.4036668, -0.086404935),
      parse("1950-10", -0.20104446, -0.33889845, -0.063190475),
      parse("1950-11", -0.38081577, -0.49243763, -0.26919392),
      parse("1950-12", -0.26077038, -0.3802367, -0.14130408),
      parse("1951-01", -0.36034262, -0.4589177, -0.26176754),
      parse("1951-02", -0.5093785, -0.6001588, -0.41859823),
      parse("1951-03", -0.2620192, -0.3542802, -0.16975816),
      parse("1951-04", -0.1080777, -0.24150318, 0.025347786),
      parse("1951-05", 0.032269213, -0.151453, 0.21599144),
      parse("1951-06", 0.060181145, -0.11182762, 0.23218991),
      parse("1951-07", 0.03866412, -0.12042866, 0.1977569),
      parse("1951-08", 0.12133113, -0.07829491, 0.32095715),
      parse("1951-09", 0.13226272, -0.019023193, 0.28354865),
      parse("1951-10", 0.12436143, -0.0045943954, 0.25331727),
      parse("1951-11", -0.07229671, -0.17430036, 0.029706946),
      parse("1951-12", 0.0691973, -0.032534182, 0.17092878),
      parse("1952-01", 0.09402644, -0.00812421, 0.1961771),
      parse("1952-02", 0.068350874, -0.028965494, 0.16566725),
      parse("1952-03", -0.16353902, -0.25835308, -0.06872494),
      parse("1952-04", 0.04558809, -0.09313565, 0.18431182),
      parse("1952-05", 0.022243695, -0.15909414, 0.20358154),
      parse("1952-06", 0.023484357, -0.1614921, 0.20846081),
      parse("1952-07", 0.09006012, -0.07709257, 0.25721282),
      parse("1952-08", 0.096942395, -0.1105835, 0.30446827),
      parse("1952-09", 0.09621102, -0.06066274, 0.25308478),
      parse("1952-10", -0.016534438, -0.14808096, 0.11501208),
      parse("1952-11", -0.13228506, -0.23842481, -0.026145296),
      parse("1952-12", -0.04029371, -0.14592908, 0.06534167),
      parse("1953-01", -0.009005178, -0.11308341, 0.09507305),
      parse("1953-02", 0.09501191, 0.0007980468, 0.18922578),
      parse("1953-03", 0.018006958, -0.083248086, 0.119262),
      parse("1953-04", 0.22455436, 0.07946404, 0.36964467),
      parse("1953-05", 0.14901619, -0.02643255, 0.32446492),
      parse("1953-06", 0.1404282, -0.04348267, 0.32433906),
      parse("1953-07", 0.043144584, -0.12013697, 0.20642613),
      parse("1953-08", 0.08707856, -0.12004952, 0.29420665),
      parse("1953-09", 0.08630111, -0.07270259, 0.24530481),
      parse("1953-10", 0.08342033, -0.048025105, 0.21486577),
      parse("1953-11", -0.0058099576, -0.11363081, 0.10201089),
      parse("1953-12", 0.01942187, -0.08567341, 0.12451715),
      parse("1954-01", -0.2434586, -0.33929825, -0.14761893),
      parse("1954-02", -0.089074455, -0.18093091, 0.002781996),
      parse("1954-03", -0.18339783, -0.27912733, -0.08766833),
      parse("1954-04", -0.11477188, -0.24236786, 0.0128240995),
      parse("1954-05", -0.17059736, -0.33720312, -0.0039916215),
      parse("1954-06", -0.1296339, -0.30016065, 0.040892836),
      parse("1954-07", -0.19616233, -0.35168973, -0.040634945),
      parse("1954-08", -0.10182279, -0.28616172, 0.08251613),
      parse("1954-09", -0.11330111, -0.26177186, 0.035169613),
      parse("1954-10", -0.011538466, -0.13309364, 0.1100167),
      parse("1954-11", 0.11638409, 0.01995511, 0.21281308),
      parse("1954-12", -0.16362791, -0.2633629, -0.0638929),
      parse("1955-01", 0.10597788, 0.014951693, 0.19700406),
      parse("1955-02", -0.19271314, -0.27525374, -0.110172555),
      parse("1955-03", -0.3963696, -0.48520008, -0.30753914),
      parse("1955-04", -0.22909744, -0.35830337, -0.099891536),
      parse("1955-05", -0.23882908, -0.40164378, -0.07601436),
      parse("1955-06", -0.17836486, -0.34488028, -0.0118494425),
      parse("1955-07", -0.20829603, -0.36318785, -0.05340421),
      parse("1955-08", -0.08842416, -0.2744727, 0.097624384),
      parse("1955-09", -0.14314792, -0.27703187, -0.0092639625),
      parse("1955-10", -0.11328784, -0.22637765, -0.0001980232),
      parse("1955-11", -0.32083783, -0.41694224, -0.22473344),
      parse("1955-12", -0.36432907, -0.45405668, -0.27460146),
      parse("1956-01", -0.2403825, -0.3247751, -0.15598993),
      parse("1956-02", -0.28243503, -0.3625925, -0.20227754),
      parse("1956-03", -0.35645252, -0.43977112, -0.27313393),
      parse("1956-04", -0.30645922, -0.40755266, -0.20536578),
      parse("1956-05", -0.33606264, -0.45112088, -0.22100437),
      parse("1956-06", -0.22652255, -0.33725312, -0.11579198),
      parse("1956-07", -0.2106352, -0.32375464, -0.097515784),
      parse("1956-08", -0.28964865, -0.42783394, -0.15146336),
      parse("1956-09", -0.2734754, -0.38730374, -0.15964706),
      parse("1956-10", -0.26315844, -0.35439906, -0.17191784),
      parse("1956-11", -0.21516415, -0.29393166, -0.13639663),
      parse("1956-12", -0.1575909, -0.23783027, -0.077351525),
      parse("1957-01", -0.22153524, -0.29409447, -0.14897601),
      parse("1957-02", -0.15856388, -0.23127134, -0.085856415),
      parse("1957-03", -0.1975386, -0.27344698, -0.121630214),
      parse("1957-04", -0.041761864, -0.12591177, 0.042388048),
      parse("1957-05", 0.03177997, -0.05156566, 0.115125604),
      parse("1957-06", 0.06454678, -0.027708724, 0.15680228),
      parse("1957-07", -0.06267592, -0.16108057, 0.035728723),
      parse("1957-08", 0.07285775, -0.025583688, 0.17129919),
      parse("1957-09", 0.041351955, -0.050062116, 0.13276602),
      parse("1957-10", -0.0007920379, -0.07993971, 0.07835563),
      parse("1957-11", -0.00397586, -0.08217196, 0.07422025),
      parse("1957-12", 0.05228784, -0.014773578, 0.119349256),
      parse("1958-01", 0.25472188, 0.18477017, 0.3246736),
      parse("1958-02", 0.09280772, 0.020115048, 0.16550039),
      parse("1958-03", 0.0052554575, -0.06892984, 0.07944076),
      parse("1958-04", -0.06950849, -0.14541516, 0.0063981838),
      parse("1958-05", 0.016476592, -0.06756154, 0.100514725),
      parse("1958-06", -0.15305896, -0.2407915, -0.06532643),
      parse("1958-07", -0.020030653, -0.122118644, 0.082057334),
      parse("1958-08", -0.083085604, -0.18378095, 0.017609738),
      parse("1958-09", -0.11479923, -0.20449965, -0.025098816),
      parse("1958-10", -0.019471264, -0.10863628, 0.06969376),
      parse("1958-11", -0.038575903, -0.110695235, 0.033543434),
      parse("1958-12", -0.08232217, -0.15505922, -0.009585136),
      parse("1959-01", 0.013691866, -0.057967845, 0.08535158),
      parse("1959-02", -0.005084035, -0.06930822, 0.059140153),
      parse("1959-03", 0.016677765, -0.055086143, 0.08844168),
      parse("1959-04", 0.026267799, -0.05526906, 0.107804656),
      parse("1959-05", -0.03853143, -0.12668422, 0.04962136),
      parse("1959-06", -0.0018165166, -0.09611195, 0.09247892),
      parse("1959-07", -0.043648165, -0.1407121, 0.05341577),
      parse("1959-08", -0.041345503, -0.12845452, 0.04576352),
      parse("1959-09", -0.14124896, -0.22689748, -0.05560044),
      parse("1959-10", -0.071337834, -0.14982772, 0.007152045),
      parse("1959-11", -0.18634169, -0.25757015, -0.11511324),
      parse("1959-12", -0.10334121, -0.16692382, -0.039758593),
      parse("1960-01", -0.11465857, -0.17818601, -0.051131118),
      parse("1960-02", 0.02281142, -0.038939342, 0.08456218),
      parse("1960-03", -0.44688365, -0.51143664, -0.38233066),
      parse("1960-04", -0.26193932, -0.33308205, -0.1907966),
      parse("1960-05", -0.17928644, -0.2596041, -0.09896878),
      parse("1960-06", -0.1261228, -0.2180584, -0.034187194),
      parse("1960-07", -0.11407162, -0.20836379, -0.019779457),
      parse("1960-08", -0.006310927, -0.09468046, 0.0820586),
      parse("1960-09", 0.0032144422, -0.077028275, 0.08345716),
      parse("1960-10", -0.031158555, -0.11090458, 0.048587475),
      parse("1960-11", -0.21504581, -0.28288013, -0.14721149),
      parse("1960-12", 0.0836075, 0.024021316, 0.14319369),
      parse("1961-01", -0.052229382, -0.10604209, 0.0015833256),
      parse("1961-02", 0.09082804, 0.03480867, 0.1468474),
      parse("1961-03", -0.0616479, -0.121089086, -0.0022067144),
      parse("1961-04", 0.06100269, -0.005988584, 0.12799396),
      parse("1961-05", 0.067880936, -0.0045889886, 0.14035086),
      parse("1961-06", 0.08760505, 0.005416348, 0.16979375),
      parse("1961-07", -0.03974989, -0.124212116, 0.044712335),
      parse("1961-08", -0.016852155, -0.093706384, 0.060002074),
      parse("1961-09", -0.030176053, -0.110131025, 0.04977892),
      parse("1961-10", -0.06376159, -0.12409272, -0.0034304585),
      parse("1961-11", -0.031068904, -0.0864477, 0.024309888),
      parse("1961-12", -0.25179952, -0.30733567, -0.19626334),
      parse("1962-01", -0.05704969, -0.111643106, -0.0024562806),
      parse("1962-02", 0.012157755, -0.047518242, 0.07183375),
      parse("1962-03", 0.018748457, -0.036765415, 0.07426233),
      parse("1962-04", -0.026072158, -0.095235944, 0.04309163),
      parse("1962-05", -0.18164511, -0.2606271, -0.102663144),
      parse("1962-06", -0.056267038, -0.1399578, 0.027423719),
      parse("1962-07", -0.10915736, -0.2047529, -0.013561823),
      parse("1962-08", -0.10973202, -0.19394721, -0.02551682),
      parse("1962-09", -0.07493823, -0.14678304, -0.003093423),
      parse("1962-10", -0.038202044, -0.112969555, 0.036565468),
      parse("1962-11", -0.020875225, -0.07987935, 0.038128905),
      parse("1962-12", -0.12562062, -0.18604958, -0.06519166),
      parse("1963-01", -0.121948205, -0.17286605, -0.07103036),
      parse("1963-02", 0.040044732, -0.013600229, 0.093689695),
      parse("1963-03", -0.3409997, -0.4023474, -0.27965203),
      parse("1963-04", -0.17912978, -0.24301982, -0.11523975),
      parse("1963-05", -0.1466549, -0.2207718, -0.072538),
      parse("1963-06", -0.06680599, -0.15230548, 0.018693516),
      parse("1963-07", 0.05953688, -0.033529285, 0.15260305),
      parse("1963-08", 0.14581573, 0.05786402, 0.23376743),
      parse("1963-09", 0.12205969, 0.039692715, 0.20442666),
      parse("1963-10", 0.06423311, -0.006535083, 0.1350013),
      parse("1963-11", 0.04470364, -0.017914778, 0.10732206),
      parse("1963-12", -0.062525906, -0.119097084, -0.005954728),
      parse("1964-01", -0.19452871, -0.24590258, -0.14315486),
      parse("1964-02", -0.22300385, -0.2707023, -0.17530538),
      parse("1964-03", -0.38280848, -0.4354939, -0.33012307),
      parse("1964-04", -0.44159797, -0.49986494, -0.38333103),
      parse("1964-05", -0.37424895, -0.44108826, -0.30740967),
      parse("1964-06", -0.16659828, -0.23335688, -0.09983969),
      parse("1964-07", -0.20062846, -0.27401683, -0.12724008),
      parse("1964-08", -0.30053315, -0.36973116, -0.23133513),
      parse("1964-09", -0.4045793, -0.46920264, -0.33995596),
      parse("1964-10", -0.3114203, -0.3756508, -0.24718978),
      parse("1964-11", -0.2988282, -0.3533131, -0.24434334),
      parse("1964-12", -0.3716253, -0.41772586, -0.32552475),
      parse("1965-01", -0.16228765, -0.21544793, -0.10912736),
      parse("1965-02", -0.26308617, -0.31864673, -0.20752563),
      parse("1965-03", -0.2597472, -0.314058, -0.20543642),
      parse("1965-04", -0.29863888, -0.36014748, -0.23713025),
      parse("1965-05", -0.19161598, -0.2604217, -0.12281029),
      parse("1965-06", -0.20754607, -0.2859227, -0.12916943),
      parse("1965-07", -0.26655743, -0.34850556, -0.1846093),
      parse("1965-08", -0.1619956, -0.2357662, -0.08822501),
      parse("1965-09", -0.24531643, -0.32013673, -0.17049614),
      parse("1965-10", -0.10577728, -0.16904464, -0.042509913),
      parse("1965-11", -0.1437657, -0.20105892, -0.08647249),
      parse("1965-12", -0.14632036, -0.19985811, -0.09278261),
      parse("1966-01", -0.28320545, -0.33601388, -0.23039699),
      parse("1966-02", -0.16757779, -0.22072573, -0.114429854),
      parse("1966-03", -0.1708448, -0.22562246, -0.11606713),
      parse("1966-04", -0.17045191, -0.22957575, -0.11132808),
      parse("1966-05", -0.21843779, -0.27984244, -0.15703316),
      parse("1966-06", -0.10499999, -0.18416446, -0.025835512),
      parse("1966-07", 0.0030392085, -0.08309536, 0.08917377),
      parse("1966-08", -0.10166964, -0.18408725, -0.019252032),
      parse("1966-09", -0.08658218, -0.16101395, -0.012150404),
      parse("1966-10", -0.23768093, -0.3007167, -0.17464516),
      parse("1966-11", -0.10768294, -0.16457853, -0.050787363),
      parse("1966-12", -0.14052084, -0.19718638, -0.083855316),
      parse("1967-01", -0.21960148, -0.26585704, -0.17334592),
      parse("1967-02", -0.32817873, -0.38193217, -0.2744253),
      parse("1967-03", -0.09374347, -0.14432028, -0.043166663),
      parse("1967-04", -0.13242696, -0.18680602, -0.078047894),
      parse("1967-05", 0.057162356, -0.005516053, 0.11984076),
      parse("1967-06", -0.108169004, -0.17946222, -0.036875788),
      parse("1967-07", -0.067892365, -0.14901736, 0.01323264),
      parse("1967-08", -0.099165745, -0.17351398, -0.024817519),
      parse("1967-09", -0.12451488, -0.19525743, -0.05377232),
      parse("1967-10", -0.020286117, -0.0855479, 0.044975664),
      parse("1967-11", -0.119650505, -0.17039524, -0.06890577),
      parse("1967-12", -0.1537288, -0.1991738, -0.10828379),
      parse("1968-01", -0.32466504, -0.37137488, -0.27795523),
      parse("1968-02", -0.26771057, -0.3169445, -0.21847664),
      parse("1968-03", -0.003727732, -0.05343113, 0.045975666),
      parse("1968-04", -0.18043014, -0.23411447, -0.1267458),
      parse("1968-05", -0.22587559, -0.28420305, -0.16754813),
      parse("1968-06", -0.15176927, -0.22612664, -0.077411875),
      parse("1968-07", -0.19500259, -0.27286038, -0.1171448),
      parse("1968-08", -0.12997681, -0.21054135, -0.049412273),
      parse("1968-09", -0.22645247, -0.29917043, -0.1537345),
      parse("1968-10", 0.008175721, -0.059301265, 0.07565271),
      parse("1968-11", -0.13496026, -0.18726788, -0.08265263),
      parse("1968-12", -0.1911928, -0.24149217, -0.14089343),
      parse("1969-01", -0.16237703, -0.20836867, -0.11638539),
      parse("1969-02", -0.1970934, -0.24815017, -0.14603664),
      parse("1969-03", -0.13277553, -0.18331009, -0.08224098),
      parse("1969-04", 0.049255937, -0.0007941228, 0.099305995),
      parse("1969-05", 0.06528902, 0.0093505345, 0.12122751),
      parse("1969-06", 0.0056816083, -0.06108295, 0.07244617),
      parse("1969-07", -0.1091712, -0.17552002, -0.042822383),
      parse("1969-08", -0.07289227, -0.14569598, -8.85525e-05),
      parse("1969-09", -0.013110088, -0.07264362, 0.046423446),
      parse("1969-10", 0.00090877194, -0.06066966, 0.062487204),
      parse("1969-11", 0.023137517, -0.028535709, 0.07481074),
      parse("1969-12", 0.16674611, 0.12138196, 0.21211027),
      parse("1970-01", 0.014005779, -0.024967588, 0.05297915),
      parse("1970-02", 0.063759334, 0.017752824, 0.10976585),
      parse("1970-03", -0.08814179, -0.13351265, -0.04277095),
      parse("1970-04", -0.07942118, -0.1272537, -0.031588662),
      parse("1970-05", -0.14084856, -0.19922996, -0.08246716),
      parse("1970-06", -0.1370101, -0.204421, -0.0695992),
      parse("1970-07", -0.11218003, -0.18914407, -0.035215985),
      parse("1970-08", -0.1992122, -0.27766284, -0.12076156),
      parse("1970-09", -0.030715683, -0.09970784, 0.03827647),
      parse("1970-10", -0.06031036, -0.12693785, 0.0063171284),
      parse("1970-11", -0.058197107, -0.110719755, -0.005674464),
      parse("1970-12", -0.1930069, -0.23651735, -0.14949644),
      parse("1971-01", -0.14365958, -0.18466523, -0.10265391),
      parse("1971-02", -0.2726155, -0.31656718, -0.22866382),
      parse("1971-03", -0.33664557, -0.38328534, -0.2900058),
      parse("1971-04", -0.23293439, -0.28357276, -0.18229601),
      parse("1971-05", -0.1979984, -0.26606554, -0.12993127),
      parse("1971-06", -0.2552373, -0.32123104, -0.18924357),
      parse("1971-07", -0.18479517, -0.2544347, -0.11515566),
      parse("1971-08", -0.1812137, -0.25926685, -0.10316055),
      parse("1971-09", -0.12637776, -0.19992936, -0.05282617),
      parse("1971-10", -0.15617602, -0.20850164, -0.103850394),
      parse("1971-11", -0.15279908, -0.20081574, -0.104782425),
      parse("1971-12", -0.23074044, -0.27251953, -0.18896136),
      parse("1972-01", -0.37904298, -0.41819, -0.33989596),
      parse("1972-02", -0.2778282, -0.320703, -0.2349534),
      parse("1972-03", -0.14302962, -0.18787768, -0.09818155),
      parse("1972-04", -0.12759815, -0.17795895, -0.07723736),
      parse("1972-05", -0.082474515, -0.14861475, -0.016334282),
      parse("1972-06", -0.031909496, -0.100287944, 0.036468957),
      parse("1972-07", -0.09311811, -0.1700235, -0.016212704),
      parse("1972-08", 0.061222523, -0.019026173, 0.14147122),
      parse("1972-09", -0.052398738, -0.120318, 0.015520524),
      parse("1972-10", -0.056839254, -0.112568185, -0.0011103234),
      parse("1972-11", -0.051805325, -0.09928263, -0.0043280204),
      parse("1972-12", 0.10889668, 0.06640539, 0.15138797),
      parse("1973-01", 0.100870065, 0.05921677, 0.14252336),
      parse("1973-02", 0.15933001, 0.120916836, 0.19774319),
      parse("1973-03", 0.11498204, 0.069975875, 0.15998821),
      parse("1973-04", 0.19915819, 0.1508453, 0.2474711),
      parse("1973-05", 0.10814754, 0.044189963, 0.17210512),
      parse("1973-06", 0.106045224, 0.045468733, 0.16662171),
      parse("1973-07", 0.01621605, -0.049754184, 0.08218629),
      parse("1973-08", -0.027661854, -0.10094168, 0.04561797),
      parse("1973-09", -0.035152175, -0.11000057, 0.03969623),
      parse("1973-10", 0.0108220205, -0.047591276, 0.06923532),
      parse("1973-11", -0.025164139, -0.071579844, 0.021251563),
      parse("1973-12", -0.1283927, -0.16965793, -0.08712748),
      parse("1974-01", -0.28239968, -0.32175258, -0.24304679),
      parse("1974-02", -0.4465835, -0.4916486, -0.40151843),
      parse("1974-03", -0.23716211, -0.2810448, -0.19327942),
      parse("1974-04", -0.20168932, -0.25082228, -0.15255636),
      parse("1974-05", -0.1265853, -0.18765014, -0.06552047),
      parse("1974-06", -0.11528032, -0.18392684, -0.04663381),
      parse("1974-07", -0.13573898, -0.20700736, -0.0644706),
      parse("1974-08", 0.023082737, -0.05511237, 0.10127784),
      parse("1974-09", -0.09269041, -0.16663258, -0.01874824),
      parse("1974-10", -0.11529485, -0.17675921, -0.053830486),
      parse("1974-11", -0.12621354, -0.17617388, -0.07625319),
      parse("1974-12", -0.21389274, -0.25553343, -0.17225204),
      parse("1975-01", -0.0966292, -0.13978224, -0.053476162),
      parse("1975-02", -0.020880949, -0.06348584, 0.021723943),
      parse("1975-03", -0.01112924, -0.05811524, 0.035856765),
      parse("1975-04", -0.051968537, -0.10094516, -0.0029919175),
      parse("1975-05", 0.053007245, -0.009473412, 0.1154879),
      parse("1975-06", -0.060783383, -0.13291909, 0.011352318),
      parse("1975-07", -0.10466969, -0.18719059, -0.022148784),
      parse("1975-08", -0.23242185, -0.30827186, -0.15657184),
      parse("1975-09", -0.139501, -0.21193255, -0.06706945),
      parse("1975-10", -0.15371256, -0.22347201, -0.083953075),
      parse("1975-11", -0.26358986, -0.31321743, -0.21396226),
      parse("1975-12", -0.24677186, -0.29279146, -0.20075226),
      parse("1976-01", -0.16959351, -0.21492438, -0.12426263),
      parse("1976-02", -0.23186125, -0.27618515, -0.18753733),
      parse("1976-03", -0.33692366, -0.38385162, -0.28999567),
      parse("1976-04", -0.18590008, -0.24028902, -0.13151114),
      parse("1976-05", -0.35880587, -0.42114106, -0.29647064),
      parse("1976-06", -0.23977908, -0.3094969, -0.17006126),
      parse("1976-07", -0.17275204, -0.25314432, -0.09235974),
      parse("1976-08", -0.24235126, -0.3215088, -0.16319375),
      parse("1976-09", -0.15119453, -0.22489694, -0.07749211),
      parse("1976-10", -0.3322862, -0.3963161, -0.26825628),
      parse("1976-11", -0.13943368, -0.19311805, -0.08574931),
      parse("1976-12", -0.02945889, -0.08022431, 0.02130653),
      parse("1977-01", 0.15336634, 0.10795948, 0.1987732),
      parse("1977-02", 0.10381479, 0.053849906, 0.15377967),
      parse("1977-03", 0.07475545, 0.024980498, 0.124530405),
      parse("1977-04", 0.13446581, 0.08024284, 0.18868878),
      parse("1977-05", 0.22771347, 0.16327552, 0.2921514),
      parse("1977-06", 0.20780508, 0.14048275, 0.2751274),
      parse("1977-07", 0.14251913, 0.061580263, 0.22345799),
      parse("1977-08", 0.1298329, 0.04911758, 0.2105482),
      parse("1977-09", -0.0019903248, -0.07811894, 0.07413829),
      parse("1977-10", -0.01617223, -0.083238326, 0.050893866),
      parse("1977-11", 0.12106576, 0.06669167, 0.17543985),
      parse("1977-12", -0.04011389, -0.09081486, 0.010587082),
      parse("1978-01", 0.036995538, -0.0071859756, 0.081177056),
      parse("1978-02", 0.07385509, 0.02860965, 0.11910052),
      parse("1978-03", 0.059285987, 0.014052451, 0.10451952),
      parse("1978-04", 0.053218562, -0.00055365695, 0.106990784),
      parse("1978-05", 0.0115159005, -0.052302517, 0.07533432),
      parse("1978-06", -0.055066437, -0.123960376, 0.01382751),
      parse("1978-07", 0.016265042, -0.05862395, 0.09115403),
      parse("1978-08", -0.2168574, -0.2955802, -0.1381346),
      parse("1978-09", 0.04174579, -0.036548592, 0.12004017),
      parse("1978-10", -0.059895515, -0.12789159, 0.008100553),
      parse("1978-11", 0.091779426, 0.04129526, 0.14226359),
      parse("1978-12", 0.010227291, -0.032376792, 0.05283137),
      parse("1979-01", 0.0516812, 0.014161454, 0.08920094),
      parse("1979-02", -0.17951806, -0.2162433, -0.14279284),
      parse("1979-03", 0.06694152, 0.030953744, 0.102929294),
      parse("1979-04", 0.07699999, 0.03878828, 0.1152117),
      parse("1979-05", -0.022435743, -0.06633464, 0.021463156),
      parse("1979-06", 0.050191652, 0.006530249, 0.09385306),
      parse("1979-07", 0.018005762, -0.03262814, 0.068639666),
      parse("1979-08", 0.114524774, 0.053652395, 0.17539716),
      parse("1979-09", 0.17285156, 0.11173119, 0.23397192),
      parse("1979-10", 0.14730756, 0.09926776, 0.19534737),
      parse("1979-11", 0.20778532, 0.16501372, 0.25055695),
      parse("1979-12", 0.38596204, 0.3516528, 0.4202713),
      parse("1980-01", 0.2527261, 0.21300751, 0.29244468),
      parse("1980-02", 0.3607043, 0.32330173, 0.3981069),
      parse("1980-03", 0.21667273, 0.17630167, 0.25704378),
      parse("1980-04", 0.23393515, 0.19394667, 0.2739236),
      parse("1980-05", 0.2456877, 0.19866508, 0.2927103),
      parse("1980-06", 0.13180715, 0.082248576, 0.18136574),
      parse("1980-07", 0.20636384, 0.15439135, 0.25833634),
      parse("1980-08", 0.16441017, 0.10678383, 0.22203653),
      parse("1980-09", 0.123579785, 0.060605038, 0.18655454),
      parse("1980-10", 0.097044446, 0.043979082, 0.15010981),
      parse("1980-11", 0.20625903, 0.15803419, 0.25448385),
      parse("1980-12", 0.113674425, 0.07168679, 0.15566206),
      parse("1981-01", 0.44390386, 0.39915735, 0.4886504),
      parse("1981-02", 0.33654687, 0.30277714, 0.3703166),
      parse("1981-03", 0.33077422, 0.29371208, 0.3678364),
      parse("1981-04", 0.19894964, 0.15788971, 0.24000956),
      parse("1981-05", 0.14773263, 0.101183265, 0.194282),
      parse("1981-06", 0.24469398, 0.19016218, 0.29922578),
      parse("1981-07", 0.26434663, 0.2016414, 0.3270519),
      parse("1981-08", 0.31205258, 0.25005865, 0.3740465),
      parse("1981-09", 0.1367701, 0.07926537, 0.19427481),
      parse("1981-10", 0.05529363, 0.0070646233, 0.103522636),
      parse("1981-11", 0.16566867, 0.12102321, 0.21031414),
      parse("1981-12", 0.3634116, 0.3172231, 0.4096001),
      parse("1982-01", -0.024068523, -0.07235162, 0.024214571),
      parse("1982-02", 0.06821714, 0.0223868, 0.11404749),
      parse("1982-03", -0.13698223, -0.1845714, -0.08939307),
      parse("1982-04", 0.020402594, -0.029829908, 0.070635095),
      parse("1982-05", 0.09071928, 0.03224222, 0.14919634),
      parse("1982-06", -0.0013139228, -0.064454645, 0.061826803),
      parse("1982-07", 0.075844936, 0.004775244, 0.14691463),
      parse("1982-08", -0.045588225, -0.11802615, 0.026849702),
      parse("1982-09", 0.016534334, -0.050160784, 0.08322945),
      parse("1982-10", 0.041061886, -0.018483559, 0.10060733),
      parse("1982-11", 0.029048067, -0.020967891, 0.07906403),
      parse("1982-12", 0.27728456, 0.23279157, 0.32177758),
      parse("1983-01", 0.3584244, 0.31470326, 0.40214556),
      parse("1983-02", 0.33635557, 0.2908501, 0.381861),
      parse("1983-03", 0.24756545, 0.20511036, 0.29002056),
      parse("1983-04", 0.17535253, 0.12663808, 0.22406697),
      parse("1983-05", 0.22204514, 0.16600104, 0.27808926),
      parse("1983-06", 0.16691586, 0.10999884, 0.22383289),
      parse("1983-07", 0.114927486, 0.056167133, 0.17368785),
      parse("1983-08", 0.28377086, 0.21628153, 0.3512602),
      parse("1983-09", 0.3260587, 0.2662912, 0.38582617),
      parse("1983-10", 0.113148384, 0.057640273, 0.16865648),
      parse("1983-11", 0.24547733, 0.19653328, 0.29442137),
      parse("1983-12", 0.096021675, 0.049950793, 0.14209256),
      parse("1984-01", 0.15764055, 0.11318809, 0.202093),
      parse("1984-02", 0.02001983, -0.025347726, 0.06538738),
      parse("1984-03", 0.14398187, 0.09306026, 0.1949035),
      parse("1984-04", 0.010137831, -0.039188646, 0.05946431),
      parse("1984-05", 0.21433139, 0.16248918, 0.2661736),
      parse("1984-06", -0.038575307, -0.100490406, 0.023339793),
      parse("1984-07", 0.024865314, -0.037082333, 0.086812966),
      parse("1984-08", 0.08923666, 0.023951871, 0.15452145),
      parse("1984-09", 0.12719487, 0.06800548, 0.18638426),
      parse("1984-10", 0.040902097, -0.012790171, 0.09459436),
      parse("1984-11", -0.05765004, -0.104584284, -0.010715791),
      parse("1984-12", -0.15602855, -0.19877177, -0.11328532),
      parse("1985-01", 0.1155558, 0.070438474, 0.16067311),
      parse("1985-02", -0.104183674, -0.14948314, -0.05888421),
      parse("1985-03", 0.04271185, -0.0051474376, 0.090571135),
      parse("1985-04", 0.0028204513, -0.042313404, 0.047954306),
      parse("1985-05", 0.067987494, 0.019966573, 0.116008416),
      parse("1985-06", 0.12117828, 0.071005605, 0.17135094),
      parse("1985-07", -0.022891713, -0.07350898, 0.027725551),
      parse("1985-08", 0.1431363, 0.08397767, 0.20229492),
      parse("1985-09", 0.11767927, 0.061845258, 0.17351328),
      parse("1985-10", 0.039328277, -0.011885249, 0.0905418),
      parse("1985-11", 0.018120727, -0.028659811, 0.06490126),
      parse("1985-12", 0.05531433, 0.014779543, 0.09584911),
      parse("1986-01", 0.11670128, 0.07163385, 0.1617687),
      parse("1986-02", 0.3093589, 0.26656476, 0.352153),
      parse("1986-03", 0.13820964, 0.09782837, 0.17859091),
      parse("1986-04", 0.13485684, 0.09247309, 0.1772406),
      parse("1986-05", 0.13101041, 0.08589921, 0.17612162),
      parse("1986-06", 0.039869342, -0.008415869, 0.088154554),
      parse("1986-07", 0.042954296, -0.01108, 0.096988596),
      parse("1986-08", 0.08790163, 0.0326548, 0.14314847),
      parse("1986-09", 0.024142634, -0.0311676, 0.07945287),
      parse("1986-10", 0.051004555, 0.0039842986, 0.098024815),
      parse("1986-11", 0.04011416, 0.00013273644, 0.08009558),
      parse("1986-12", 0.03212002, -0.009131731, 0.073371775),
      parse("1987-01", 0.22004549, 0.17728503, 0.26280597),
      parse("1987-02", 0.31599578, 0.27342454, 0.35856703),
      parse("1987-03", 0.104251325, 0.06377447, 0.14472818),
      parse("1987-04", 0.1889307, 0.1432275, 0.23463391),
      parse("1987-05", 0.16939573, 0.122770004, 0.21602145),
      parse("1987-06", 0.26585034, 0.21572928, 0.31597137),
      parse("1987-07", 0.34769198, 0.3002212, 0.39516273),
      parse("1987-08", 0.21050136, 0.15359204, 0.2674107),
      parse("1987-09", 0.26593548, 0.21810848, 0.31376252),
      parse("1987-10", 0.26956236, 0.22279249, 0.31633225),
      parse("1987-11", 0.18613108, 0.14134122, 0.23092094),
      parse("1987-12", 0.37202516, 0.3352671, 0.40878323),
      parse("1988-01", 0.45912874, 0.4158971, 0.50236034),
      parse("1988-02", 0.33302152, 0.288721, 0.37732208),
      parse("1988-03", 0.38519633, 0.34249073, 0.42790192),
      parse("1988-04", 0.32767174, 0.28327718, 0.37206626),
      parse("1988-05", 0.31933275, 0.2659124, 0.3727531),
      parse("1988-06", 0.33036658, 0.27507606, 0.3856571),
      parse("1988-07", 0.2556859, 0.2018744, 0.30949736),
      parse("1988-08", 0.25767827, 0.19392912, 0.32142743),
      parse("1988-09", 0.25825685, 0.19838016, 0.31813353),
      parse("1988-10", 0.254709, 0.20196135, 0.30745664),
      parse("1988-11", 0.028571693, -0.021743037, 0.07888642),
      parse("1988-12", 0.17620139, 0.12991531, 0.22248745),
      parse("1989-01", -0.026005011, -0.07035818, 0.018348154),
      parse("1989-02", 0.23738506, 0.19713072, 0.27763942),
      parse("1989-03", 0.18512788, 0.1426663, 0.22758949),
      parse("1989-04", 0.14958581, 0.10371123, 0.19546038),
      parse("1989-05", 0.038256668, -0.013571449, 0.090084784),
      parse("1989-06", 0.09383323, 0.03904897, 0.14861748),
      parse("1989-07", 0.26380813, 0.20505361, 0.32256266),
      parse("1989-08", 0.25223345, 0.185825, 0.3186419),
      parse("1989-09", 0.29773393, 0.23395298, 0.36151487),
      parse("1989-10", 0.26001906, 0.20135662, 0.31868154),
      parse("1989-11", 0.11531052, 0.06379637, 0.16682467),
      parse("1989-12", 0.2837144, 0.23685499, 0.33057386),
      parse("1990-01", 0.28277373, 0.23656435, 0.32898313),
      parse("1990-02", 0.27884814, 0.23266093, 0.3250354),
      parse("1990-03", 0.6347864, 0.58848524, 0.6810877),
      parse("1990-04", 0.4940245, 0.44746447, 0.5405845),
      parse("1990-05", 0.40596974, 0.35395133, 0.45798814),
      parse("1990-06", 0.3151888, 0.26191312, 0.36846447),
      parse("1990-07", 0.3796524, 0.32039037, 0.43891445),
      parse("1990-08", 0.28095096, 0.21359323, 0.34830868),
      parse("1990-09", 0.22031836, 0.1604603, 0.28017643),
      parse("1990-10", 0.38645974, 0.33040056, 0.4425189),
      parse("1990-11", 0.35899177, 0.30630797, 0.4116756),
      parse("1990-12", 0.28878504, 0.24264595, 0.33492413),
      parse("1991-01", 0.30340114, 0.26020792, 0.34659436),
      parse("1991-02", 0.4070458, 0.35951877, 0.45457286),
      parse("1991-03", 0.24740836, 0.20353888, 0.29127786),
      parse("1991-04", 0.43600866, 0.3861235, 0.48589385),
      parse("1991-05", 0.31398752, 0.2661917, 0.36178336),
      parse("1991-06", 0.4334769, 0.38315243, 0.4838014),
      parse("1991-07", 0.46741664, 0.413612, 0.5212213),
      parse("1991-08", 0.3620418, 0.30141258, 0.422671),
      parse("1991-09", 0.35283542, 0.2949484, 0.41072243),
      parse("1991-10", 0.25118244, 0.19777474, 0.30459014),
      parse("1991-11", 0.25364134, 0.20526768, 0.30201498),
      parse("1991-12", 0.23831235, 0.19523758, 0.28138712),
      parse("1992-01", 0.36292845, 0.32279372, 0.40306315),
      parse("1992-02", 0.34844932, 0.30590802, 0.39099064),
      parse("1992-03", 0.30922496, 0.26654962, 0.35190034),
      parse("1992-04", 0.16185239, 0.11636393, 0.20734085),
      parse("1992-05", 0.16797747, 0.115565054, 0.22038987),
      parse("1992-06", 0.15643431, 0.09561311, 0.21725553),
      parse("1992-07", 0.005340144, -0.05662222, 0.0673025),
      parse("1992-08", 0.03774724, -0.039666817, 0.1151613),
      parse("1992-09", -0.08989003, -0.16200817, -0.017771905),
      parse("1992-10", -0.022682328, -0.07580555, 0.030440897),
      parse("1992-11", -0.03733942, -0.08598786, 0.011309022),
      parse("1992-12", 0.09871902, 0.054537915, 0.14290014),
      parse("1993-01", 0.26102662, 0.21946819, 0.30258507),
      parse("1993-02", 0.28333446, 0.23878285, 0.32788607),
      parse("1993-03", 0.23608007, 0.18892586, 0.28323427),
      parse("1993-04", 0.19514883, 0.14572291, 0.24457474),
      parse("1993-05", 0.19503304, 0.13349326, 0.25657284),
      parse("1993-06", 0.16445984, 0.099572934, 0.22934674),
      parse("1993-07", 0.15863112, 0.09205189, 0.22521032),
      parse("1993-08", 0.08418279, 0.013109553, 0.15525602),
      parse("1993-09", 0.045521, -0.027268391, 0.11831039),
      parse("1993-10", 0.15051654, 0.09072666, 0.21030644),
      parse("1993-11", 0.06566895, 0.012819225, 0.11851867),
      parse("1993-12", 0.14829831, 0.1003046, 0.19629201),
      parse("1994-01", 0.20031649, 0.15430397, 0.24632901),
      parse("1994-02", 0.014015401, -0.037698988, 0.06572979),
      parse("1994-03", 0.23547028, 0.1929952, 0.27794534),
      parse("1994-04", 0.25558206, 0.20292777, 0.30823633),
      parse("1994-05", 0.20769499, 0.14959231, 0.26579767),
      parse("1994-06", 0.31435218, 0.25530413, 0.37340024),
      parse("1994-07", 0.18217762, 0.11941934, 0.2449359),
      parse("1994-08", 0.1732785, 0.10758695, 0.23897003),
      parse("1994-09", 0.2722316, 0.21403906, 0.3304242),
      parse("1994-10", 0.34696406, 0.29957974, 0.39434835),
      parse("1994-11", 0.3531186, 0.30807722, 0.39815998),
      parse("1994-12", 0.24739546, 0.20374133, 0.29104957),
      parse("1995-01", 0.41360325, 0.36785004, 0.4593565),
      parse("1995-02", 0.6401344, 0.5955497, 0.6847191),
      parse("1995-03", 0.3564086, 0.3137096, 0.39910764),
      parse("1995-04", 0.42731214, 0.38019723, 0.474427),
      parse("1995-05", 0.26464352, 0.2158012, 0.31348586),
      parse("1995-06", 0.37844238, 0.32494324, 0.4319415),
      parse("1995-07", 0.44184983, 0.3830831, 0.50061655),
      parse("1995-08", 0.391613, 0.32795027, 0.45527574),
      parse("1995-09", 0.27957153, 0.22031401, 0.33882907),
      parse("1995-10", 0.37940124, 0.3315722, 0.42723027),
      parse("1995-11", 0.37539726, 0.32783252, 0.42296204),
      parse("1995-12", 0.1740167, 0.13141942, 0.21661398),
      parse("1996-01", 0.21639675, 0.17465207, 0.25814143),
      parse("1996-02", 0.46155688, 0.4207942, 0.5023196),
      parse("1996-03", 0.2883724, 0.24666715, 0.33007762),
      parse("1996-04", 0.25163049, 0.2030822, 0.3001788),
      parse("1996-05", 0.23714963, 0.18536083, 0.2889384),
      parse("1996-06", 0.21651414, 0.16119543, 0.27183285),
      parse("1996-07", 0.3253906, 0.2691372, 0.381644),
      parse("1996-08", 0.35149437, 0.2850047, 0.41798404),
      parse("1996-09", 0.20457508, 0.1391627, 0.26998743),
      parse("1996-10", 0.12766548, 0.07437373, 0.18095721),
      parse("1996-11", 0.3162583, 0.26804507, 0.36447155),
      parse("1996-12", 0.32326886, 0.28039086, 0.36614683),
      parse("1997-01", 0.24903603, 0.2050436, 0.29302847),
      parse("1997-02", 0.30442297, 0.26189378, 0.34695214),
      parse("1997-03", 0.39708516, 0.35149974, 0.44267058),
      parse("1997-04", 0.3775284, 0.33273223, 0.4223246),
      parse("1997-05", 0.31710884, 0.26625335, 0.36796433),
      parse("1997-06", 0.49175587, 0.43785518, 0.54565656),
      parse("1997-07", 0.3461335, 0.28540578, 0.4068612),
      parse("1997-08", 0.40574697, 0.3444696, 0.46702433),
      parse("1997-09", 0.4674913, 0.4117658, 0.5232167),
      parse("1997-10", 0.5718022, 0.5265241, 0.61708033),
      parse("1997-11", 0.61571497, 0.573657, 0.657773),
      parse("1997-12", 0.52387595, 0.48183948, 0.5659124),
      parse("1998-01", 0.5629589, 0.5195781, 0.6063397),
      parse("1998-02", 0.7973742, 0.75377077, 0.84097755),
      parse("1998-03", 0.5894701, 0.54612184, 0.6328183),
      parse("1998-04", 0.65089357, 0.6040493, 0.6977379),
      parse("1998-05", 0.6390668, 0.5907695, 0.6873641),
      parse("1998-06", 0.6679476, 0.6111207, 0.7247745),
      parse("1998-07", 0.65909415, 0.59979075, 0.71839756),
      parse("1998-08", 0.6453173, 0.57817835, 0.7124562),
      parse("1998-09", 0.43478084, 0.375726, 0.49383566),
      parse("1998-10", 0.41109028, 0.35487813, 0.4673024),
      parse("1998-11", 0.38224158, 0.33153662, 0.43294653),
      parse("1998-12", 0.48756245, 0.4447607, 0.53036416),
      parse("1999-01", 0.3872738, 0.34417182, 0.43037575),
      parse("1999-02", 0.60396034, 0.55778724, 0.6501335),
      parse("1999-03", 0.26383877, 0.22156142, 0.30611613),
      parse("1999-04", 0.26947576, 0.22510633, 0.3138452),
      parse("1999-05", 0.2477816, 0.19969408, 0.29586914),
      parse("1999-06", 0.32476822, 0.2663474, 0.383189),
      parse("1999-07", 0.3104075, 0.2524305, 0.36838448),
      parse("1999-08", 0.3025803, 0.23762035, 0.36754024),
      parse("1999-09", 0.27602673, 0.21561834, 0.33643514),
      parse("1999-10", 0.30699936, 0.25596854, 0.3580302),
      parse("1999-11", 0.28691775, 0.2385114, 0.3353241),
      parse("1999-12", 0.3137894, 0.2717762, 0.35580257),
      parse("2000-01", 0.18133692, 0.13842326, 0.22425057),
      parse("2000-02", 0.49915195, 0.45606333, 0.54224056),
      parse("2000-03", 0.44504288, 0.40051213, 0.48957363),
      parse("2000-04", 0.5384154, 0.49182132, 0.5850094),
      parse("2000-05", 0.30583385, 0.25669247, 0.35497522),
      parse("2000-06", 0.3502256, 0.29859963, 0.40185153),
      parse("2000-07", 0.32576227, 0.2669195, 0.38460502),
      parse("2000-08", 0.3747202, 0.31097934, 0.43846104),
      parse("2000-09", 0.3107709, 0.2432111, 0.37833068),
      parse("2000-10", 0.19478564, 0.1407225, 0.24884878),
      parse("2000-11", 0.24529691, 0.19789627, 0.29269755),
      parse("2000-12", 0.20167494, 0.16030242, 0.24304746),
      parse("2001-01", 0.36417356, 0.3213304, 0.40701672),
      parse("2001-02", 0.3918914, 0.3534353, 0.43034744),
      parse("2001-03", 0.5204522, 0.47700635, 0.563898),
      parse("2001-04", 0.45251647, 0.4085218, 0.49651113),
      parse("2001-05", 0.48934218, 0.43208975, 0.5465946),
      parse("2001-06", 0.44591305, 0.3939379, 0.4978882),
      parse("2001-07", 0.50912946, 0.45009238, 0.5681665),
      parse("2001-08", 0.5074936, 0.4429187, 0.5720685),
      parse("2001-09", 0.49257064, 0.4302663, 0.554875),
      parse("2001-10", 0.46419922, 0.41107866, 0.51731974),
      parse("2001-11", 0.69211686, 0.640878, 0.7433557),
      parse("2001-12", 0.54156566, 0.49947885, 0.5836525),
      parse("2002-01", 0.6473876, 0.60533094, 0.68944436),
      parse("2002-02", 0.6426373, 0.6011374, 0.68413717),
      parse("2002-03", 0.79940027, 0.7590046, 0.8397959),
      parse("2002-04", 0.56820655, 0.52739567, 0.6090175),
      parse("2002-05", 0.502807, 0.44722888, 0.5583851),
      parse("2002-06", 0.4493149, 0.39931378, 0.49931604),
      parse("2002-07", 0.56947935, 0.51337254, 0.62558615),
      parse("2002-08", 0.4984804, 0.43160668, 0.56535417),
      parse("2002-09", 0.50192404, 0.44515252, 0.5586955),
      parse("2002-10", 0.50043756, 0.45524058, 0.54563457),
      parse("2002-11", 0.4944799, 0.4534452, 0.5355146),
      parse("2002-12", 0.34704304, 0.30875584, 0.3853302),
      parse("2003-01", 0.65388364, 0.613197, 0.6945702),
      parse("2003-02", 0.48562348, 0.4458717, 0.52537525),
      parse("2003-03", 0.48364088, 0.44157836, 0.52570343),
      parse("2003-04", 0.47277164, 0.42678806, 0.51875526),
      parse("2003-05", 0.52228177, 0.47656673, 0.5679968),
      parse("2003-06", 0.42513415, 0.37852696, 0.47174132),
      parse("2003-07", 0.48383972, 0.43063387, 0.53704554),
      parse("2003-08", 0.6121032, 0.550043, 0.6741635),
      parse("2003-09", 0.5746869, 0.5124665, 0.6369072),
      parse("2003-10", 0.65677655, 0.6032623, 0.7102908),
      parse("2003-11", 0.4990721, 0.45430422, 0.54384),
      parse("2003-12", 0.66022813, 0.6204066, 0.7000496),
      parse("2004-01", 0.5089659, 0.46465796, 0.55327386),
      parse("2004-02", 0.634869, 0.5934909, 0.67624706),
      parse("2004-03", 0.52825564, 0.48533162, 0.5711797),
      parse("2004-04", 0.52279246, 0.47966897, 0.56591594),
      parse("2004-05", 0.35092646, 0.29784238, 0.4040105),
      parse("2004-06", 0.31097007, 0.25695875, 0.3649814),
      parse("2004-07", 0.23719142, 0.18061385, 0.293769),
      parse("2004-08", 0.45830214, 0.3906122, 0.5259921),
      parse("2004-09", 0.46726716, 0.40468165, 0.5298527),
      parse("2004-10", 0.52292866, 0.47372162, 0.57213575),
      parse("2004-11", 0.6510514, 0.6063923, 0.6957104),
      parse("2004-12", 0.41492838, 0.37327114, 0.4565856),
      parse("2005-01", 0.6500713, 0.609467, 0.6906757),
      parse("2005-02", 0.5338277, 0.49320507, 0.5744504),
      parse("2005-03", 0.6459099, 0.6046563, 0.6871636),
      parse("2005-04", 0.63165, 0.5915574, 0.6717425),
      parse("2005-05", 0.553209, 0.5060686, 0.6003494),
      parse("2005-06", 0.5688213, 0.5192257, 0.61841685),
      parse("2005-07", 0.55488116, 0.5001601, 0.6096023),
      parse("2005-08", 0.58098936, 0.5232058, 0.6387729),
      parse("2005-09", 0.6340549, 0.5779788, 0.690131),
      parse("2005-10", 0.68443674, 0.63147235, 0.73740107),
      parse("2005-11", 0.6432208, 0.60146093, 0.6849806),
      parse("2005-12", 0.6012786, 0.5628747, 0.63968253),
      parse("2006-01", 0.4999044, 0.46040437, 0.53940445),
      parse("2006-02", 0.6455559, 0.6057573, 0.6853545),
      parse("2006-03", 0.543684, 0.5041917, 0.5831763),
      parse("2006-04", 0.48621997, 0.44641006, 0.5260299),
      parse("2006-05", 0.47178116, 0.42676216, 0.51680017),
      parse("2006-06", 0.57188183, 0.5230483, 0.6207154),
      parse("2006-07", 0.4763779, 0.4250063, 0.52774954),
      parse("2006-08", 0.6333706, 0.57561433, 0.6911268),
      parse("2006-09", 0.5499711, 0.49338928, 0.60655284),
      parse("2006-10", 0.6137471, 0.5645416, 0.6629527),
      parse("2006-11", 0.66086054, 0.62075555, 0.7009655),
      parse("2006-12", 0.7172778, 0.67794055, 0.75661516),
      parse("2007-01", 0.8930489, 0.8556196, 0.9304782),
      parse("2007-02", 0.63967174, 0.5991852, 0.6801582),
      parse("2007-03", 0.59194857, 0.5498087, 0.63408846),
      parse("2007-04", 0.7261497, 0.6873898, 0.7649096),
      parse("2007-05", 0.58012205, 0.53377515, 0.62646896),
      parse("2007-06", 0.5006901, 0.4553407, 0.5460395),
      parse("2007-07", 0.5630122, 0.5168375, 0.6091869),
      parse("2007-08", 0.5837867, 0.5294786, 0.63809484),
      parse("2007-09", 0.5083473, 0.45169324, 0.56500137),
      parse("2007-10", 0.5590798, 0.511534, 0.60662574),
      parse("2007-11", 0.5355275, 0.4951343, 0.57592076),
      parse("2007-12", 0.41903135, 0.37986842, 0.45819432),
      parse("2008-01", 0.21021158, 0.17132828, 0.24909489),
      parse("2008-02", 0.2693002, 0.23020747, 0.30839294),
      parse("2008-03", 0.6136748, 0.57399446, 0.65335524),
      parse("2008-04", 0.44805115, 0.40573728, 0.490365),
      parse("2008-05", 0.43248925, 0.38698703, 0.47799146),
      parse("2008-06", 0.38399753, 0.33127367, 0.43672138),
      parse("2008-07", 0.5340773, 0.47861314, 0.58954144),
      parse("2008-08", 0.45276764, 0.38723072, 0.5183046),
      parse("2008-09", 0.5480162, 0.48638335, 0.609649),
      parse("2008-10", 0.5909558, 0.5442111, 0.63770056),
      parse("2008-11", 0.63220257, 0.58828634, 0.67611885),
      parse("2008-12", 0.4720541, 0.4311543, 0.5129539),
      parse("2009-01", 0.59029907, 0.5528154, 0.6277828),
      parse("2009-02", 0.47127843, 0.43303752, 0.50951934),
      parse("2009-03", 0.446841, 0.40850705, 0.48517495),
      parse("2009-04", 0.528052, 0.48462898, 0.571475),
      parse("2009-05", 0.5503384, 0.5067513, 0.5939255),
      parse("2009-06", 0.6234883, 0.57204455, 0.67493206),
      parse("2009-07", 0.68074554, 0.6269582, 0.7345329),
      parse("2009-08", 0.6221325, 0.5652589, 0.679006),
      parse("2009-09", 0.68177086, 0.6316205, 0.7319212),
      parse("2009-10", 0.6202136, 0.57169586, 0.6687314),
      parse("2009-11", 0.7060583, 0.6633704, 0.74874634),
      parse("2009-12", 0.64016175, 0.59658796, 0.68373555),
      parse("2010-01", 0.7106075, 0.6701205, 0.75109464),
      parse("2010-02", 0.7901331, 0.7487547, 0.8315116),
      parse("2010-03", 0.8602274, 0.82209677, 0.8983581),
      parse("2010-04", 0.83354694, 0.79155594, 0.87553793),
      parse("2010-05", 0.71491516, 0.668846, 0.76098436),
      parse("2010-06", 0.6450033, 0.59584945, 0.69415724),
      parse("2010-07", 0.61741793, 0.5650828, 0.66975313),
      parse("2010-08", 0.6146318, 0.5553926, 0.67387086),
      parse("2010-09", 0.5642509, 0.51248723, 0.6160146),
      parse("2010-10", 0.64213544, 0.5979481, 0.68632275),
      parse("2010-11", 0.73456544, 0.69051653, 0.7786144),
      parse("2010-12", 0.4370223, 0.39815724, 0.4758874),
      parse("2011-01", 0.45676905, 0.41436893, 0.49916914),
      parse("2011-02", 0.42912033, 0.3897128, 0.46852788),
      parse("2011-03", 0.5563974, 0.51538527, 0.5974095),
      parse("2011-04", 0.60221547, 0.56289977, 0.64153117),
      parse("2011-05", 0.44336236, 0.40157145, 0.4851533),
      parse("2011-06", 0.5482127, 0.502444, 0.5939814),
      parse("2011-07", 0.64491206, 0.58925456, 0.7005696),
      parse("2011-08", 0.6510249, 0.5970996, 0.70495015),
      parse("2011-09", 0.53794116, 0.48637974, 0.5895026),
      parse("2011-10", 0.5763366, 0.5300275, 0.6226457),
      parse("2011-11", 0.5112523, 0.46869895, 0.5538056),
      parse("2011-12", 0.49482837, 0.4525499, 0.5371068),
      parse("2012-01", 0.3695542, 0.3276059, 0.4115025),
      parse("2012-02", 0.4647088, 0.41840962, 0.51100796),
      parse("2012-03", 0.4209961, 0.3828921, 0.4591001),
      parse("2012-04", 0.6105002, 0.56782424, 0.6531762),
      parse("2012-05", 0.6589486, 0.6107233, 0.70717394),
      parse("2012-06", 0.629351, 0.5786319, 0.6800701),
      parse("2012-07", 0.56779647, 0.51425034, 0.6213426),
      parse("2012-08", 0.6449439, 0.5771274, 0.7127604),
      parse("2012-09", 0.64759123, 0.5868464, 0.708336),
      parse("2012-10", 0.7321626, 0.6801398, 0.7841854),
      parse("2012-11", 0.7210487, 0.6773536, 0.76474386),
      parse("2012-12", 0.4636831, 0.42318177, 0.5041845),
      parse("2013-01", 0.6078036, 0.5659251, 0.64968204),
      parse("2013-02", 0.52651656, 0.48379186, 0.5692413),
      parse("2013-03", 0.5829417, 0.541925, 0.62395847),
      parse("2013-04", 0.57481265, 0.5257574, 0.6238679),
      parse("2013-05", 0.5568857, 0.50146765, 0.6123038),
      parse("2013-06", 0.6593197, 0.6030043, 0.7156352),
      parse("2013-07", 0.58917844, 0.5326067, 0.64575016),
      parse("2013-08", 0.6352067, 0.57162017, 0.6987932),
      parse("2013-09", 0.7221902, 0.6662515, 0.77812886),
      parse("2013-10", 0.6503963, 0.597056, 0.70373666),
      parse("2013-11", 0.7700114, 0.7229114, 0.81711143),
      parse("2013-12", 0.60764134, 0.56141466, 0.65386796),
      parse("2014-01", 0.6695646, 0.622098, 0.7170312),
      parse("2014-02", 0.49080652, 0.44725463, 0.5343584),
      parse("2014-03", 0.694528, 0.648441, 0.740615),
      parse("2014-04", 0.73631227, 0.6886264, 0.78399813),
      parse("2014-05", 0.73708785, 0.6884094, 0.78576624),
      parse("2014-06", 0.61747456, 0.5643953, 0.6705538),
      parse("2014-07", 0.5809606, 0.52662504, 0.63529617),
      parse("2014-08", 0.73377496, 0.67367274, 0.7938772),
      parse("2014-09", 0.756769, 0.7046591, 0.8088789),
      parse("2014-10", 0.7423641, 0.69290525, 0.7918229),
      parse("2014-11", 0.63000286, 0.5861616, 0.6738441),
      parse("2014-12", 0.6848145, 0.6452265, 0.7244026),
      parse("2015-01", 0.7400082, 0.69822913, 0.7817873),
      parse("2015-02", 0.78175026, 0.740612, 0.8228885),
      parse("2015-03", 0.79528373, 0.75049824, 0.8400692),
      parse("2015-04", 0.70091575, 0.65298545, 0.74884605),
      parse("2015-05", 0.7309856, 0.6839389, 0.77803224),
      parse("2015-06", 0.7596305, 0.7079085, 0.81135243),
      parse("2015-07", 0.7145488, 0.66327834, 0.7658193),
      parse("2015-08", 0.7991504, 0.7368658, 0.861435),
      parse("2015-09", 0.82609314, 0.77123094, 0.88095534),
      parse("2015-10", 1.0153373, 0.96493423, 1.0657405),
      parse("2015-11", 0.9959859, 0.95247537, 1.0394964),
      parse("2015-12", 1.0416831, 0.99845076, 1.0849154),
      parse("2016-01", 1.0877537, 1.0464499, 1.1290573),
      parse("2016-02", 1.2235882, 1.1839018, 1.2632747),
      parse("2016-03", 1.1820779, 1.1462066, 1.217949),
      parse("2016-04", 1.0296463, 0.9872375, 1.072055),
      parse("2016-05", 0.8804874, 0.83482504, 0.9261497),
      parse("2016-06", 0.77708083, 0.7294142, 0.82474744),
      parse("2016-07", 0.77772677, 0.72627807, 0.82917553),
      parse("2016-08", 0.94857633, 0.8921002, 1.0050524),
      parse("2016-09", 0.8455436, 0.79136264, 0.8997246),
      parse("2016-10", 0.81902903, 0.7713468, 0.8667112),
      parse("2016-11", 0.8444866, 0.80383766, 0.8851355),
      parse("2016-12", 0.77912897, 0.7382809, 0.8199771),
      parse("2017-01", 0.9519715, 0.91184795, 0.99209505),
      parse("2017-02", 1.0670907, 1.0290638, 1.1051178),
      parse("2017-03", 1.0645919, 1.0271796, 1.1020042),
      parse("2017-04", 0.84599763, 0.80479693, 0.88719827),
      parse("2017-05", 0.7798254, 0.73829556, 0.8213552),
      parse("2017-06", 0.6575456, 0.609505, 0.70558625),
      parse("2017-07", 0.80458516, 0.7507668, 0.85840356),
      parse("2017-08", 0.81099087, 0.7511646, 0.8708171),
      parse("2017-09", 0.72930163, 0.6776079, 0.78099537),
      parse("2017-10", 0.80949205, 0.764127, 0.85485715),
      parse("2017-11", 0.8058706, 0.763549, 0.8481923),
      parse("2017-12", 0.8148281, 0.7754213, 0.85423493),
      parse("2018-01", 0.7106047, 0.6729177, 0.7482917),
      parse("2018-02", 0.7957914, 0.75522476, 0.836358),
      parse("2018-03", 0.7900751, 0.7507628, 0.82938737),
      parse("2018-04", 0.8223178, 0.7767505, 0.86788505),
      parse("2018-05", 0.71316856, 0.6692025, 0.7571347),
      parse("2018-06", 0.7375257, 0.6873122, 0.7877392),
      parse("2018-07", 0.7334142, 0.67723614, 0.78959227),
      parse("2018-08", 0.73473924, 0.672286, 0.79719245),
      parse("2018-09", 0.67586756, 0.6212568, 0.7304783),
      parse("2018-10", 0.8693208, 0.8213614, 0.9172802),
      parse("2018-11", 0.7447671, 0.7012972, 0.78823704),
      parse("2018-12", 0.8242562, 0.7847442, 0.86376816),
      parse("2019-01", 0.7995958, 0.75716734, 0.84202427),
      parse("2019-02", 0.8442263, 0.79712707, 0.89132553),
      parse("2019-03", 1.0757369, 1.0314062, 1.1200676),
      parse("2019-04", 0.93880504, 0.89124775, 0.9863623),
      parse("2019-05", 0.7776251, 0.73376954, 0.8214807),
      parse("2019-06", 0.80911994, 0.7547636, 0.8634763),
      parse("2019-07", 0.8571952, 0.79993093, 0.9144594),
      parse("2019-08", 0.85825413, 0.79831004, 0.91819817),
      parse("2019-09", 0.8026575, 0.74497795, 0.860337),
      parse("2019-10", 0.9556284, 0.9032823, 1.0079745),
      parse("2019-11", 0.9372168, 0.8904445, 0.9839892),
      parse("2019-12", 1.0368102, 0.9909035, 1.0827167),
      parse("2020-01", 1.0694563, 1.0249017, 1.1140109),
      parse("2020-02", 1.1133685, 1.0673538, 1.1593832),
      parse("2020-03", 1.0943997, 1.0482767, 1.1405227),
      parse("2020-04", 1.0626723, 1.0151452, 1.1101993),
      parse("2020-05", 0.90792704, 0.8561947, 0.9596594),
      parse("2020-06", 0.8251334, 0.7627924, 0.8874744),
      parse("2020-07", 0.81607646, 0.7542902, 0.87786263),
      parse("2020-08", 0.80101144, 0.7415303, 0.8604925),
      parse("2020-09", 0.8670323, 0.80767053, 0.92639405),
      parse("2020-10", 0.8107299, 0.76469195, 0.8567679),
      parse("2020-11", 1.0130944, 0.9651476, 1.0610412),
      parse("2020-12", 0.6926237, 0.6460196, 0.7392277),
      parse("2021-01", 0.70082587, 0.6534755, 0.7481762),
      parse("2021-02", 0.56454515, 0.51881725, 0.61027306),
      parse("2021-03", 0.7262527, 0.6786184, 0.77388704),
      parse("2021-04", 0.7601092, 0.71282953, 0.80738884),
      parse("2021-05", 0.7064251, 0.6576904, 0.75515974),
      parse("2021-06", 0.71322304, 0.66050047, 0.7659456),
      parse("2021-07", 0.7916223, 0.72672427, 0.85652035),
      parse("2021-08", 0.79945594, 0.7385131, 0.8603988),
      parse("2021-09", 0.8674762, 0.81049454, 0.92445797),
      parse("2021-10", 0.907391, 0.8540407, 0.9607413),
      parse("2021-11", 0.8536163, 0.8015729, 0.9056597),
      parse("2021-12", 0.7513281, 0.7025465, 0.80010974),
      parse("2022-01", 0.7788991, 0.7316793, 0.8261188),
      parse("2022-02", 0.76354825, 0.71481043, 0.812286),
      parse("2022-03", 0.88959247, 0.8434075, 0.9357775),
      parse("2022-04", 0.7703609, 0.7202506, 0.82047117),
      parse("2022-05", 0.7613267, 0.71092033, 0.811733),
      parse("2022-06", 0.8581897, 0.80506545, 0.911314),
      parse("2022-07", 0.77835214, 0.7176057, 0.8390986),
      parse("2022-08", 0.85325915, 0.7917466, 0.9147717),
      parse("2022-09", 0.79008293, 0.73376393, 0.84640193),
      parse("2022-10", 0.9288468, 0.87793136, 0.97976226),
      parse("2022-11", 0.674607, 0.6267639, 0.7224501),
      parse("2022-12", 0.76834446, 0.72198945, 0.8146995),
      parse("2023-01", 0.7786378, 0.7296263, 0.8276493),
      parse("2023-02", 0.87222534, 0.82195526, 0.9224954),
      parse("2023-03", 1.1243817, 1.0749991, 1.1737643),
      parse("2023-04", 0.9272796, 0.87626445, 0.97829473),
      parse("2023-05", 0.87121195, 0.817516, 0.9249078),
      parse("2023-06", 1.0509841, 0.9951409, 1.1068273)
    )
}
