/**
 * This file is part of the CRISTAL-iSE kernel.
 * Copyright (c) 2001-2015 The CRISTAL Consortium. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; with out even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * http://www.fsf.org/licensing/licenses/lgpl.html
 */
package org.cristalise.kernel.collection;

import org.cristalise.kernel.common.InvalidCollectionModification;
import org.cristalise.kernel.common.ObjectNotFoundException;
import org.cristalise.kernel.entity.proxy.ItemProxy;
import org.cristalise.kernel.lookup.ItemPath;
import org.cristalise.kernel.utils.CastorHashMap;


/**
 * CollectionMembers are pointers to individual Items in a collection. Under 
 * certain circumstances they may be empty, or they can be created and destroyed
 * as required. They can hold properties containing annotations about the link,
 * including information on the typing of the Item they should point to. 
 * Properties that dictate type information are specified in the ClassProps, and
 * assignment of an Item will fail if those member properties do not match the
 * Property values of the Item attempting to be assigned, throwing a detailing the mismatches. 
 *
 */

public interface CollectionMember {
	
	/**
	 * Returns the current assigned ItemPath
	 * @return the ItemPath, null if empty
	 */
	public ItemPath getItemPath();
	
	/**
	 * Returns the UUID of the currently assigned Item
	 * @return UUID
	 */
	public String getChildUUID();

	/**
	 * Assign the given item to the slot, if it fits
	 * 
	 * @param itemPath The item to assign
	 * @throws InvalidCollectionModification When the Item Properties don't match the typing specification in ClassProps and Properties
	 */
    public void assignItem(ItemPath itemPath) throws InvalidCollectionModification;
    
    /**
     * De-assign the slot, leaving it empty.
     */
    public void clearItem();
    
    /**
     * Resolve the Item currently assigned
     * @return the ItemProxy of the assigned Item
     * @throws ObjectNotFoundException When empty
     */
    public ItemProxy resolveItem() throws ObjectNotFoundException;

	/**
	 * Get the integer slot ID, as generated by the parent Collection
	 * @return ID
	 */
	public int getID();

	/**
	 * Gets the member properties.
	 * @return CastorHashMap
	 */
	public CastorHashMap getProperties();

	/**
	 * Gets the class properties, which specify type information in child Items.
	 * On instantiation from a CollectionDescription, the ClassProps are 
	 * generated from the PropertyDescriptions which have the isClassIdentifier 
	 * boolean set to true.
	 * 
	 * @return Comma-separated list of property names that relate to Item type
	 */
	public String getClassProps();
}
