package org.cryptimeleon.craco.secretsharing.accessstructure.visitors;

import org.cryptimeleon.craco.secretsharing.accessstructure.exceptions.WrongAccessStructureException;
import org.cryptimeleon.craco.secretsharing.accessstructure.utils.TreeNode;
import org.cryptimeleon.math.structures.rings.zn.Zp;
import org.cryptimeleon.math.structures.rings.zn.Zp.ZpElement;

import java.util.ArrayList;

/**
 * This visitor calculates the monotone span program matrix for the tree it visits.
 */
public class MonotoneSpanProgramGetMatrixVisitor implements Visitor<Integer> {

    /**
     * Field over which the matrix is calculated.
     */
    private final Zp field;

    /**
     * Contain the values for the columns belong to higher hierarchies.
     */
    private final ArrayList<ZpElement> prefix;

    /**
     * Matrix containing all rows calculated so far.
     */
    private final ArrayList<ArrayList<ZpElement>> matrix;

    /**
     * The threshold of the node the visitor is performed on.
     */
    private int threshold;

    /**
     * Internal counter, that count how many visitors for children were.
     * requested so far
     */
    private ZpElement counterN;

    /**
     * The number of columns this subtree with the current node as root node
     * need in the monotone span program.
     */
    private int ownOffset = 0;

    /**
     * The current node.
     */
    private TreeNode node;

    /**
     * @param field  field over which the monotone span program is calculated
     * @param prefix the values for the columns belong to higher hierarchies
     * @param matrix matrix containing all rows calculated so far and the new rows
     *               will be stored in this instance
     */
    public MonotoneSpanProgramGetMatrixVisitor(Zp field, ArrayList<ZpElement> prefix,
                                               ArrayList<ArrayList<ZpElement>> matrix) {
        this.field = field;
        this.prefix = new ArrayList<>();
        for (ZpElement ele : prefix) {
            this.prefix.add(field.getOneElement().mul(ele));
        }
        this.matrix = matrix;
        counterN = field.getZeroElement();
    }

    @Override
    public Integer getResultOfCurrentNode() {

        if (threshold == 0) {
            return 0;
        }

        return (ownOffset + threshold - 1);
    }

    @Override
    public MonotoneSpanProgramGetMatrixVisitor getVisitorForNextChild() throws WrongAccessStructureException {
        ZpElement value = field.getOneElement();

        counterN = counterN.add(field.getOneElement());

        // calculate the new values for the counterN child
        @SuppressWarnings("unchecked")
        ArrayList<ZpElement> tempPrefix = (ArrayList<ZpElement>) prefix.clone();
        if (threshold != 0) {
            for (int counterT = 1; counterT < threshold; counterT++) {
                value = value.mul(counterN);
                tempPrefix.add(value);
            }
        } else {
            throw new WrongAccessStructureException(
                    "Tree contains a node with children and Threshold 0 \n 0 is not a valid threshold");
        }
        // extend the prefix vector for all further children
        for (int counter = 0; counter < ownOffset; counter++) {
            tempPrefix.add(field.getZeroElement());
        }

        return new MonotoneSpanProgramGetMatrixVisitor(field, tempPrefix, matrix);
    }

    @Override
    public void putResultOfChild(Integer input) {
        ownOffset += input;
    }

    @Override
    public void visit(TreeNode currentNode) {
        node = currentNode;
        threshold = node.getThreshold();
        int numberOfNodes = node.getNumberOfChildren();
        // check if the current node is a leaf, then add a new row
        if (numberOfNodes == 0) {
            matrix.add(prefix);
        }

    }

}
