package org.cryptimeleon.craco.protocols.arguments.sigma.schnorr.setmembership;


import org.cryptimeleon.math.serialization.Representable;
import org.cryptimeleon.math.serialization.Representation;
import org.cryptimeleon.math.serialization.annotations.ReprUtil;
import org.cryptimeleon.math.serialization.annotations.Represented;
import org.cryptimeleon.math.structures.groups.GroupElement;
import org.cryptimeleon.math.structures.groups.elliptic.BilinearGroup;
import org.cryptimeleon.math.structures.rings.zn.Zn;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * Public parameters used for set membership proofs. Consists mostly of signatures on the values in the set.
 * Shall be generated by a trusted party.
 */
public class SetMembershipPublicParameters implements Representable {
    BilinearGroup bilinearGroup;
    @Represented(restorer = "G1")
    GroupElement g1;
    @Represented(restorer = "G2")
    GroupElement g2;
    @Represented(restorer = "G2")
    GroupElement pk;
    @Represented(restorer = "int -> G1")
    HashMap<BigInteger, GroupElement> signatures;
    GroupElement egg;

    public SetMembershipPublicParameters(BilinearGroup bilinearGroup, GroupElement g1, GroupElement g2, GroupElement pk,
                                         HashMap<BigInteger, GroupElement> signatures) {
        this.bilinearGroup = bilinearGroup;
        this.g1 = g1;
        this.g2 = g2;
        this.pk = pk;
        this.signatures = signatures;
        this.egg = bilinearGroup.getBilinearMap().apply(g1, g2);
    }

    public SetMembershipPublicParameters(BilinearGroup group, Representation repr) {
        this.bilinearGroup = group;
        new ReprUtil(this).register(group).deserialize(repr);
        this.egg = bilinearGroup.getBilinearMap().apply(g1, g2);
    }

    public static SetMembershipPublicParameters generate(BilinearGroup group, Collection<BigInteger> set) {
        GroupElement g1 = group.getG1().getUniformlyRandomNonNeutral();
        GroupElement g2 = group.getG2().getUniformlyRandomNonNeutral();
        Zn.ZnElement sk = group.getG1().getUniformlyRandomExponent();
        GroupElement pk = g2.pow(sk);
        HashMap<BigInteger, GroupElement> signatures = new HashMap<>();
        for (BigInteger i : set) {
            signatures.put(i, g1.pow(sk.add(group.getZn().valueOf(i)).inv()));
            if (i.compareTo(group.getG1().size()) > 0)
                throw new IllegalArgumentException("Set contains too large a number.");
        }

        return new SetMembershipPublicParameters(group, g1, g2, pk, signatures);
    }

    public static SetMembershipPublicParameters generate(BilinearGroup group, BigInteger... set) {
        return generate(group, Arrays.asList(set));
    }

    public static SetMembershipPublicParameters generate(BilinearGroup group, int... set) {
        return generate(group, Arrays.stream(set).mapToObj(BigInteger::valueOf).toArray(BigInteger[]::new));
    }

    /**
     * Generates parameters for a set membership proof for [leftBoundInclusive, rightBoundExclusive).
     */
    public static SetMembershipPublicParameters generateInterval(BilinearGroup group, int leftBoundInclusive, int rightBoundExclusive) {
        return SetMembershipPublicParameters.generate(group, IntStream.range(leftBoundInclusive, rightBoundExclusive)
                .mapToObj(BigInteger::valueOf).collect(Collectors.toSet()));
    }

    @Override
    public Representation getRepresentation() {
        return ReprUtil.serialize(this);
    }

    public Zn getZn() {
        return g1.getStructure().getZn();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SetMembershipPublicParameters that = (SetMembershipPublicParameters) o;
        return Objects.equals(bilinearGroup, that.bilinearGroup) && Objects.equals(g1, that.g1) && Objects.equals(g2, that.g2) && Objects.equals(pk, that.pk) && Objects.equals(signatures, that.signatures) && Objects.equals(egg, that.egg);
    }

    @Override
    public int hashCode() {
        return Objects.hash(bilinearGroup, g1, g2, pk, signatures, egg);
    }
}
