package org.cryptimeleon.craco.sig.sps.akot15.fsp2;

import org.cryptimeleon.craco.commitment.CommitmentPair;
import org.cryptimeleon.craco.sig.Signature;
import org.cryptimeleon.craco.sig.sps.akot15.tc.TCAKOT15OpenValue;
import org.cryptimeleon.craco.sig.sps.akot15.tcgamma.TCGAKOT15Commitment;
import org.cryptimeleon.craco.sig.sps.akot15.tcgamma.TCGAKOT15XSIGCommitment;
import org.cryptimeleon.craco.sig.sps.akot15.xsig.SPSXSIGSignature;
import org.cryptimeleon.math.serialization.ObjectRepresentation;
import org.cryptimeleon.math.serialization.Representation;
import org.cryptimeleon.math.structures.groups.Group;

import java.util.Objects;

/**
 * A signature as generated by the fully SPS scheme {@link SPSFSP2Signature}.
 *
 */
public class SPSFSP2Signature implements Signature {

    /**
     * The signature the building block {@link org.cryptimeleon.craco.sig.sps.akot15.xsig.SPSXSIGSignatureScheme}
     * generated.
     */
    SPSXSIGSignature sigmaXSIG;

    /**
     * The commitment the building block {@link org.cryptimeleon.craco.sig.sps.akot15.tc.TCAKOT15CommitmentScheme}
     * generated.
     */
    CommitmentPair commitmentPairTC;


    public SPSFSP2Signature(SPSXSIGSignature sigmaXSIG, CommitmentPair commitmentPairTC) {
        this.sigmaXSIG = sigmaXSIG;
        this.commitmentPairTC = commitmentPairTC;
    }

    public SPSFSP2Signature(Group g1, Group g2, Representation repr) {
        ObjectRepresentation objRepr = (ObjectRepresentation) repr;

        this.sigmaXSIG = new SPSXSIGSignature(objRepr.get("sigma"), g1, g2);

        TCGAKOT15Commitment com;

        // if the representation contains an instance of TCGAKOT15XSIGCommitment, use that
        if(objRepr.get("comXSIG") != null) {
            com = new TCGAKOT15XSIGCommitment(g2, objRepr.get("comXSIG"));
        }
        else {
            com = new TCGAKOT15Commitment(g2, objRepr.get("com"));
        }
        TCAKOT15OpenValue open = new TCAKOT15OpenValue(g1, g2, objRepr.get("open"));

        this.commitmentPairTC = new CommitmentPair(com, open);
    }


    @Override
    public Representation getRepresentation() {
        ObjectRepresentation objRepr = new ObjectRepresentation();

        objRepr.put("sigma", sigmaXSIG.getRepresentation());

        //if special XSIG variant is used, put that instead
        if(commitmentPairTC.getCommitment() instanceof TCGAKOT15XSIGCommitment) {
            objRepr.put("comXSIG", commitmentPairTC.getCommitment().getRepresentation());
        }
        else {
            objRepr.put("com", commitmentPairTC.getCommitment().getRepresentation());
        }
        objRepr.put("open", commitmentPairTC.getOpenValue().getRepresentation());

        return objRepr;
    }

    public SPSXSIGSignature getSigmaXSIG() {
        return sigmaXSIG;
    }

    public CommitmentPair getCommitmentPairTC() {
        return commitmentPairTC;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SPSFSP2Signature that = (SPSFSP2Signature) o;
        return Objects.equals(sigmaXSIG, that.sigmaXSIG)
                && Objects.equals(commitmentPairTC.getOpenValue(), that.commitmentPairTC.getOpenValue())
                && Objects.equals(commitmentPairTC.getCommitment(), that.commitmentPairTC.getCommitment());
    }

    @Override
    public int hashCode() {
        return Objects.hash(sigmaXSIG, commitmentPairTC.getOpenValue(), commitmentPairTC.getOpenValue());
    }

}
