/*
 * © OOO "SI IKS LAB", 2022-2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.cxbox.core.dto.rowmeta;

import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import java.util.Map;
import java.util.function.UnaryOperator;
import lombok.NonNull;
import org.cxbox.core.dto.rowmeta.PreActionConfirmBuilder.Cf;
import org.cxbox.core.dto.rowmeta.PreActionConfirmWithWidgetBuilder.Cfw;
import org.cxbox.meta.ui.field.WidgetName;
import org.cxbox.meta.ui.field.WidgetTypeFamily;

public class PreAction {

	public static final String WITHOUT_MESSAGE = "";

	private final PreActionType preActionType;

	private final String message;

	private final Map<String, String> customParameters;

	public PreAction(PreActionType preActionType, String message, Map<String, String> customParameters) {
		this.preActionType = preActionType;
		this.message = message;
		this.customParameters = customParameters;
	}

	private static PreAction preAction(PreActionType preAction, String message, Map<String, String> customParameters) {
		return PreAction.builder()
				.preActionType(preAction)
				.message(message)
				.customParameters(customParameters)
				.build();
	}

	/**
	 * confirm with default {@code title} and auto-generated {@code text}
	 * <br>
	 * <br>{@code title} - default 'Are you sure?' {@code title} will be shown (actually its translation defined in Frontend i18n)
	 * <br>{@code text} - will be auto-generated by backend from {@link PreActionType#CONFIRMATION} template.
	 */
	public static PreAction confirm() {
		return PreAction.confirm(cf -> cf);
	}


	/**
	 * <br>
	 * @deprecated as of release 4.0.0-M16,
	 * replaced by configurable {@code confirm(cf -> cf.text("Changes will be saved"))}.
	 * <br>
	 */
	@Deprecated(since = "4.0.0-M16", forRemoval = true)
	public static PreAction confirm(String text) {
		return PreAction.confirm(cf -> cf.text(text));
	}

	/**
	 * confirm with your {@code title}, {@code text}, and button names ({@code yesText} and {@code noText})
	 * <pre>
	 * {@code
	 * PreAction.confirm(cf -> cf
	 *  .title("Approve?") //or withoutTitle()
	 *  .text("Changes will ba saved") //or withoutText()
	 *  .yesText("Yes")
	 *  .noText("Cancel"))
	 * }
	 * </pre>
	 * @return preActio
	 */
	public static PreAction confirm(@NonNull UnaryOperator<Cf> cf) {
		return Cf.confirmBuilder(cf);
	}

	/**
	 * confirm with your {@code widget}, {@code title}, and button names ({@code yesText} and {@code noText})
	 *
	 * @param widget widget name of any {@code *FormPopup.widget.json} placed on same {@code view} and having same {@code bc},
	 * as widget with button, that triggered confirmation
	 * <pre>
	 * {@code
	 * PreAction.confirmWithWidget("myWidget", cf -> cf
	 *  .title("Approve?") //or withoutTitle()
	 *  .yesText("Yes")
	 *  .noText("Cancel"))
	 * }
	 * </pre>
	 * @return preAction
	 */
	public static PreAction confirmWithWidget(
			@NonNull @WidgetName(typeFamily = WidgetTypeFamily.FORM_POPUP) String widget,
			@NonNull UnaryOperator<Cfw> cf) {
		return Cfw.confirmWithWidgetBuilder(widget, cf);
	}

	public static PreAction info(String message) {
		return PreAction.preAction(PreActionType.INFORMATION, message, null);
	}

	public static PreAction info() {
		return PreAction.info(null);
	}

	public static PreAction error(String message) {
		return PreAction.preAction(PreActionType.ERROR, message, null);
	}

	public static PreAction error() {
		return PreAction.error(null);
	}

	public static PreAction custom(String message, Map<String, String> customParameters) {
		return PreAction.preAction(PreActionType.CUSTOM, message, customParameters);
	}

	public static PreActionBuilder builder() {
		return new PreActionBuilder(
		);
	}

	public String getType() {
		return preActionType.getType();
	}

	public String getMessage() {
		return message;
	}

	@JsonAnyGetter
	public Map<String, String> getCustomParameters() {
		return customParameters;
	}

	public String getMessage(String action) {
		if (WITHOUT_MESSAGE.equals(message)) {
			return message;
		}
		return isBlank(message) && nonNull(preActionType) ? preActionType.getMessage(trimToEmpty(action)) : message;
	}

	public static class PreActionBuilder {

		private PreActionType preActionType;

		private String message;

		private Map<String, String> customParameters;

		PreActionBuilder() {
		}

		public PreActionBuilder preActionType(PreActionType preActionType) {
			this.preActionType = preActionType;
			return this;
		}

		public PreActionBuilder message(String message) {
			this.message = message;
			return this;
		}

		public PreActionBuilder customParameters(Map<String, String> customParameters) {
			this.customParameters = customParameters;
			return this;
		}

		public PreAction build() {
			return new PreAction(this.preActionType, this.message, this.customParameters);
		}

		public String toString() {
			return "PreAction.PreActionBuilder(preActionType=" + this.preActionType + ", message=" + this.message
					+ ", customParameters=" + this.customParameters + ")";
		}

	}

}
