/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.transport.http;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.Writer;
import java.net.HttpCookie;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.Set;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.annotations.Nullable;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.file.FileSnapshot;
import org.eclipse.jgit.internal.storage.file.LockFile;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.util.FileUtils;
import org.eclipse.jgit.util.IO;
import org.eclipse.jgit.util.RawParseUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NetscapeCookieFile {
    private static final String HTTP_ONLY_PREAMBLE = "#HttpOnly_";
    private static final String COLUMN_SEPARATOR = "\t";
    private static final String LINE_SEPARATOR = "\n";
    private static final int LOCK_ACQUIRE_MAX_RETRY_COUNT = 4;
    private static final int LOCK_ACQUIRE_RETRY_SLEEP = 500;
    private final Path path;
    private FileSnapshot snapshot;
    private byte[] hash;
    final Date creationDate;
    private Set<HttpCookie> cookies = null;
    private static final Logger LOG = LoggerFactory.getLogger(NetscapeCookieFile.class);

    public NetscapeCookieFile(Path path) {
        this(path, new Date());
    }

    NetscapeCookieFile(Path path, Date creationDate) {
        this.path = path;
        this.snapshot = FileSnapshot.DIRTY;
        this.creationDate = creationDate;
    }

    public Path getPath() {
        return this.path;
    }

    public Set<HttpCookie> getCookies(boolean refresh) {
        block3: {
            if (this.cookies == null || refresh) {
                try {
                    byte[] in = this.getFileContentIfModified();
                    Set<HttpCookie> newCookies = NetscapeCookieFile.parseCookieFile(in, this.creationDate);
                    this.cookies = this.cookies != null ? NetscapeCookieFile.mergeCookies(newCookies, this.cookies) : newCookies;
                    return this.cookies;
                }
                catch (IOException | IllegalArgumentException e) {
                    LOG.warn(MessageFormat.format(JGitText.get().couldNotReadCookieFile, this.path), e);
                    if (this.cookies != null) break block3;
                    this.cookies = new LinkedHashSet<HttpCookie>();
                }
            }
        }
        return this.cookies;
    }

    private static Set<HttpCookie> parseCookieFile(@NonNull byte[] input, @NonNull Date creationDate) throws IOException, IllegalArgumentException {
        String decoded = RawParseUtils.decode(StandardCharsets.US_ASCII, input);
        LinkedHashSet<HttpCookie> cookies = new LinkedHashSet<HttpCookie>();
        Throwable throwable = null;
        Object var5_6 = null;
        try (BufferedReader reader = new BufferedReader(new StringReader(decoded));){
            String line;
            while ((line = reader.readLine()) != null) {
                HttpCookie cookie = NetscapeCookieFile.parseLine(line, creationDate);
                if (cookie == null) continue;
                cookies.add(cookie);
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
        return cookies;
    }

    private static HttpCookie parseLine(@NonNull String line, @NonNull Date creationDate) {
        if (line.isEmpty() || line.startsWith("#") && !line.startsWith(HTTP_ONLY_PREAMBLE)) {
            return null;
        }
        String[] cookieLineParts = line.split(COLUMN_SEPARATOR, 7);
        if (cookieLineParts == null) {
            throw new IllegalArgumentException(MessageFormat.format(JGitText.get().couldNotFindTabInLine, line));
        }
        if (cookieLineParts.length < 7) {
            throw new IllegalArgumentException(MessageFormat.format(JGitText.get().couldNotFindSixTabsInLine, cookieLineParts.length, line));
        }
        String name = cookieLineParts[5];
        String value2 = cookieLineParts[6];
        HttpCookie cookie = new HttpCookie(name, value2);
        String domain = cookieLineParts[0];
        if (domain.startsWith(HTTP_ONLY_PREAMBLE)) {
            cookie.setHttpOnly(true);
            domain = domain.substring(HTTP_ONLY_PREAMBLE.length());
        }
        if (domain.startsWith(".")) {
            domain = domain.substring(1);
        }
        cookie.setDomain(domain);
        cookie.setPath(cookieLineParts[2]);
        cookie.setSecure(Boolean.parseBoolean(cookieLineParts[3]));
        long expires = Long.parseLong(cookieLineParts[4]);
        long maxAge = (expires - creationDate.getTime()) / 1000L;
        if (maxAge <= 0L) {
            return null;
        }
        cookie.setMaxAge(maxAge);
        return cookie;
    }

    private byte[] getFileContentIfModified() throws IOException {
        int maxStaleRetries = 5;
        int retries = 0;
        File file2 = this.getPath().toFile();
        if (!file2.exists()) {
            LOG.warn(MessageFormat.format(JGitText.get().missingCookieFile, file2.getAbsolutePath()));
            return new byte[0];
        }
        while (true) {
            FileSnapshot oldSnapshot = this.snapshot;
            FileSnapshot newSnapshot = FileSnapshot.save(file2);
            try {
                byte[] in = IO.readFully(file2);
                byte[] newHash = NetscapeCookieFile.hash(in);
                if (Arrays.equals(this.hash, newHash)) {
                    if (oldSnapshot.equals(newSnapshot)) {
                        oldSnapshot.setClean(newSnapshot);
                    } else {
                        this.snapshot = newSnapshot;
                    }
                } else {
                    this.snapshot = newSnapshot;
                    this.hash = newHash;
                }
                return in;
            }
            catch (FileNotFoundException e) {
                throw e;
            }
            catch (IOException e) {
                if (FileUtils.isStaleFileHandle(e) && retries < 5) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug(MessageFormat.format(JGitText.get().configHandleIsStale, retries), e);
                    }
                    ++retries;
                    continue;
                }
                throw new IOException(MessageFormat.format(JGitText.get().cannotReadFile, this.getPath()), e);
            }
            break;
        }
    }

    private static byte[] hash(byte[] in) {
        return Constants.newMessageDigest().digest(in);
    }

    public void write(URL url2) throws IOException, InterruptedException {
        Object cookiesFromFile;
        try {
            byte[] cookieFileContent = this.getFileContentIfModified();
            if (cookieFileContent != null) {
                LOG.debug("Reading the underlying cookie file '{}' as it has been modified since the last access", (Object)this.path);
                cookiesFromFile = NetscapeCookieFile.parseCookieFile(cookieFileContent, this.creationDate);
                this.cookies = NetscapeCookieFile.mergeCookies(cookiesFromFile, this.cookies);
            }
        }
        catch (FileNotFoundException cookieFileContent) {
            // empty catch block
        }
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        cookiesFromFile = null;
        Object var4_6 = null;
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)output, StandardCharsets.US_ASCII);){
            NetscapeCookieFile.write(writer, this.cookies, url2, this.creationDate);
        }
        catch (Throwable throwable) {
            if (cookiesFromFile == null) {
                cookiesFromFile = throwable;
            } else if (cookiesFromFile != throwable) {
                ((Throwable)cookiesFromFile).addSuppressed(throwable);
            }
            throw cookiesFromFile;
        }
        LockFile lockFile = new LockFile(this.path.toFile());
        int retryCount = 0;
        while (retryCount < 4) {
            if (lockFile.lock()) {
                try {
                    lockFile.setNeedSnapshot(true);
                    lockFile.write(output.toByteArray());
                    if (!lockFile.commit()) {
                        throw new IOException(MessageFormat.format(JGitText.get().cannotCommitWriteTo, this.path));
                    }
                }
                finally {
                    lockFile.unlock();
                }
                return;
            }
            Thread.sleep(500L);
            ++retryCount;
        }
        throw new IOException(MessageFormat.format(JGitText.get().cannotLock, lockFile));
    }

    static void write(@NonNull Writer writer, @NonNull Collection<HttpCookie> cookies, @NonNull URL url2, @NonNull Date creationDate) throws IOException {
        for (HttpCookie cookie : cookies) {
            NetscapeCookieFile.writeCookie(writer, cookie, url2, creationDate);
        }
    }

    private static void writeCookie(@NonNull Writer writer, @NonNull HttpCookie cookie, @NonNull URL url2, @NonNull Date creationDate) throws IOException {
        if (cookie.getMaxAge() <= 0L) {
            return;
        }
        String domain = "";
        if (cookie.isHttpOnly()) {
            domain = HTTP_ONLY_PREAMBLE;
        }
        domain = cookie.getDomain() != null ? String.valueOf(domain) + cookie.getDomain() : String.valueOf(domain) + url2.getHost();
        writer.write(domain);
        writer.write(COLUMN_SEPARATOR);
        writer.write("TRUE");
        writer.write(COLUMN_SEPARATOR);
        String path = cookie.getPath();
        if (path == null) {
            path = url2.getPath();
        }
        writer.write(path);
        writer.write(COLUMN_SEPARATOR);
        writer.write(Boolean.toString(cookie.getSecure()).toUpperCase());
        writer.write(COLUMN_SEPARATOR);
        String expirationDate = String.valueOf(creationDate.getTime() + cookie.getMaxAge() * 1000L);
        writer.write(expirationDate);
        writer.write(COLUMN_SEPARATOR);
        writer.write(cookie.getName());
        writer.write(COLUMN_SEPARATOR);
        writer.write(cookie.getValue());
        writer.write(LINE_SEPARATOR);
    }

    static Set<HttpCookie> mergeCookies(Set<HttpCookie> cookies1, @Nullable Set<HttpCookie> cookies2) {
        LinkedHashSet<HttpCookie> mergedCookies = new LinkedHashSet<HttpCookie>(cookies1);
        if (cookies2 != null) {
            mergedCookies.addAll(cookies2);
        }
        return mergedCookies;
    }
}

