/**
 * Copyright (c) 2023-present, Dash Core Group
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
package org.dashj.platform.sdk.platform

import java.lang.UnsupportedOperationException
import kotlin.collections.listOf
import org.bitcoinj.core.ECKey
import org.bitcoinj.core.Sha256Hash
import org.dashj.platform.dapiclient.model.DocumentQuery
import org.dashj.platform.dpp.Factory
import org.dashj.platform.dpp.ProtocolVersion
import org.dashj.platform.dpp.contract.DataContract
import org.dashj.platform.dpp.contract.DataContractCreateTransition
import org.dashj.platform.dpp.deepCopy
import org.dashj.platform.dpp.document.Document
import org.dashj.platform.dpp.document.DocumentCreateTransition
import org.dashj.platform.dpp.document.DocumentReplaceTransition
import org.dashj.platform.dpp.document.DocumentTransition
import org.dashj.platform.dpp.document.DocumentsBatchTransition
import org.dashj.platform.dpp.identifier.Identifier
import org.dashj.platform.dpp.identity.Identity
import org.dashj.platform.dpp.identity.IdentityCreateTransition
import org.dashj.platform.dpp.identity.IdentityPublicKey
import org.dashj.platform.dpp.statetransition.StateTransitionIdentitySigned
import org.dashj.platform.dpp.toHex
import org.dashj.platform.dpp.util.Converters
import org.dashj.platform.sdk.KeyType
import org.slf4j.Logger
import org.slf4j.LoggerFactory

open class MockPlatformStateRepository(platform: Platform) : PlatformStateRepository(platform) {
    companion object {
        private val log: Logger = LoggerFactory.getLogger(MockPlatformStateRepository::class.java)
    }

    init {
        loadMockData()
    }

    private fun compare(a: Any, b: Any?, operator: String): Boolean {
        when (operator) {
            "==" -> {
                return when {
                    a is ByteArray && b is ByteArray -> a.contentEquals(b as? ByteArray)
                    a is Identifier && b is ByteArray -> a == Identifier.from(b)
                    a is ByteArray && b is Identifier -> Identifier.from(a) == b
                    else -> a == b
                }
            }

            "startsWith" -> {
                return when {
                    a is String && b is String -> (b as String).startsWith(a)
                    else -> throw UnsupportedOperationException("startsWith has non String operands")
                }
            }

            "in" -> return when {
                a is List<*> && b is Identifier -> (a as List<*>).indexOf(b) != -1
                a is List<*> && b is ByteArray -> {
                    var found = false
                    for (item in a) {
                        if (item is ByteArray) {
                            if (item.contentEquals(b)) {
                                found = true
                                break
                            }
                        } else if (item is Identifier) {
                            if (item == Identifier.from(b)) {
                                found = true
                                break
                            }
                        }
                    }
                    found
                }

                else -> throw UnsupportedOperationException("in has unknown operands: " + a.javaClass.name + " and " + b?.javaClass?.name)
            }

            ">" -> return when {
                a is Int && b is Int -> b.toInt() > a.toInt()
                a is Long && b is Long -> b.toLong() > a.toLong()
                a is Int && b is Long -> b.toInt() > a.toLong()
                a is Long && b is Int -> b.toLong() > a.toInt()
                else -> throw throw UnsupportedOperationException("> has unknown operands: " + a.javaClass.name + " and " + b?.javaClass?.name)
            }
            ">=" -> return when {
                a is Int && b is Int -> b.toInt() >= a.toInt()
                a is Long && b is Long -> b.toLong() >= a.toLong()
                a is Int && b is Long -> b.toInt() >= a.toLong()
                a is Long && b is Int -> b.toLong() >= a.toInt()
                else -> throw throw UnsupportedOperationException(">= has unknown operands: " + a.javaClass.name + " and " + b?.javaClass?.name)
            }

            else -> {
                throw UnsupportedOperationException(operator)
            }
        }
    }

    private fun matchWhere(where: List<Any>, document: Document): Boolean {
        val field = where[0] as String
        val operator = where[1] as String
        val value = where[2]
        val rawDocument = document.toObject()
        return if (field.startsWith("\$")) {
            val docValue = rawDocument[field]
            compare(value, docValue, operator)
        } else {
            val docValue = document.get(field)
            compare(value, docValue, operator)
        }
    }

    override fun fetchDocuments(contractId: Identifier, documentType: String, where: Any): List<Document> {
        val query = where as DocumentQuery
        val byType = documentsMap.filter {
            it.value.dataContractId == contractId && it.value.type == documentType
        }

        val results = byType.filter { entry ->
            val document = entry.value
            if (query.where == null) {
                true
            } else {
                query.where!!.map { whereClause ->
                    whereClause as List<*>
                    matchWhere(whereClause as List<Any>, document)
                }.all { queryResult ->
                    queryResult
                }
            }
        }
        return results.values.toList()
    }

    override fun storeIdentity(identity: Identity) {
        if (!identityMap.containsKey(identity.id)) {
            identityMap[identity.id] = identity
        }
        storeIdentityPublicKeyHashes(
            identity.id,
            identity.publicKeys.map {
                when (it.type) {
                    KeyType.ECDSA_HASH160, KeyType.BIP13_SCRIPT_HASH -> it.data
                    KeyType.ECDSA_SECP256K1 -> ECKey.fromPublicOnly(it.data).pubKeyHash
                    else -> Sha256Hash.twiceOf(it.data).bytes
                }
            }
        )
        log.info("store identity: {}", identity.toBuffer().toHex())
    }

    // for Mocking
    private fun loadMockData() {
        val contractList = listOf(
            "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",
            "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",
            "01a5632469645820d97524b363666f1667d003765cef445312b543088bfe3e64959dd51d5df0bfb46724736368656d61783468747470733a2f2f736368656d612e646173682e6f72672f6470702d302d342d302f6d6574612f646174612d636f6e7472616374676f776e65724964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a058136776657273696f6e0169646f63756d656e7473a16b74785f6d65746164617461a56474797065666f626a65637467696e646963657382a2646e616d65676f776e657249646a70726f7065727469657381a168246f776e6572496463617363a2646e616d65736f776e65724964416e644372656174656441746a70726f7065727469657382a168246f776e6572496463617363a16a246372656174656441746361736368726571756972656484686b6579496e64657872656e6372797074696f6e4b6579496e64657871656e637279707465644d657461646174616a246372656174656441746a70726f70657274696573a3686b6579496e646578a3647479706567696e7465676572676d696e696d756d006b6465736372697074696f6e784e54686520696e646578206f6620746865206f776e657273206964656e74697479207075626c6963206b6579207573656420746f206465726976652074686520656e6372797074696f6e206b65792e71656e637279707465644d65746164617461a56474797065656172726179686d61784974656d73191000686d696e4974656d73182069627974654172726179f56b6465736372697074696f6e7824656e63727970746564206d65746164617461207573696e67204145532d4342432d32353672656e6372797074696f6e4b6579496e646578a3647479706567696e7465676572676d696e696d756d006b6465736372697074696f6e7868546865207365636f6e6461727920696e646578207573656420746f206465726976652074686520656e6372797074696f6e206b65792074686174206973207573656420746f20656e637279707420616e64206465637279707420656e63727970746564446174612e746164646974696f6e616c50726f70657274696573f4"
        )

        contractList.forEach {
            val (protocolVersion, rawDataContract) = Factory.decodeProtocolEntity(Converters.fromHex(it))
            rawDataContract["protocolVersion"] = protocolVersion
            val contract = DataContract(rawDataContract)
            storeDataContract(contract)
        }

        val identityList = listOf<String>()
        val documentList = listOf<String>()

        identityList.forEach {
            val (protocolVersion, rawIdentity) = Factory.decodeProtocolEntity(Converters.fromHex(it))
            rawIdentity["protocolVersion"] = protocolVersion
            val identity = Identity(rawIdentity)
            storeIdentity(identity)
        }

        documentList.forEach {
            val (protocolVersion, rawDocument) = Factory.decodeProtocolEntity(Converters.fromHex(it))
            rawDocument["protocolVersion"] = protocolVersion
            val document = Document(rawDocument, fetchDataContract(Identifier.from(rawDocument["\$dataContractId"]))!!)
            storeDocument(document)
        }

        storeUsername(
            "also negative confirm imitate balance foam edit under mule indoor cream huge",
            "dashpay27",
            "01a4626964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a058136762616c616e636500687265766973696f6e006a7075626c69634b65797382a76269640064646174615821037c0d8903a6f931d4011cc618dd720402c35d780aed2172df7bc9a7e78c2d390464747970650067707572706f73650068726561644f6e6c79f4697369676e617475726558411f4934365775e7738fff3a4c3b0964971bed30455d34df9863d3b6c2dce6eea80a2a2cf6fcc0287cb46d421cc155d12acdf4a1d6d2fb53556ce7bbbe652ba4239b6d73656375726974794c6576656c00a762696401646461746158210315c19126ec2c161496977c6e768fb44a76c68ba33e62da4fa027a997df6eaaa064747970650067707572706f73650068726561644f6e6c79f4697369676e6174757265584120063ed67920ef91b280467bfd3e0f33df90931d69d15c6df7599bf448aa187a1d5728e59897307b28b0383b70afcfe92a6057e3baeaf5aab15b6df30f8f752ce16d73656375726974794c6576656c02",
            "01ac6324696458202ac576395625a78f365cd160fb8c9b49fd5bc401ef6b762d08a1bc2342334f1565247479706566646f6d61696e656c6162656c69646173687061793237677265636f726473a17464617368556e697175654964656e746974794964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a0581368246f776e65724964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a0581369247265766973696f6e006c7072656f7264657253616c745820beb4bd4bc476017ea23216b958fb7be8473ead5852d63143d6f6e53d46e45c896e737562646f6d61696e52756c6573a16f616c6c6f77537562646f6d61696e73f46f2464617461436f6e747261637449645820e668c659af66aee1e72c186dde7b5b7e0a1d712a09c40d5721f622bf53c531556f6e6f726d616c697a65644c6162656c69646173687061793237702470726f746f636f6c56657273696f6e01781a6e6f726d616c697a6564506172656e74446f6d61696e4e616d656464617368",
            "01a9632469645820ff6d899351d65c4b01f055f24cca518751b084acf03621a28773b3916f85f04a6524747970656770726f66696c6568246f776e65724964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a0581369247265766973696f6e006a246372656174656441741b00000189f5a1b0846a247570646174656441741b00000189f5a1b0846b646973706c61794e616d656f4461736850617920557365722032376f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01",
            listOf()
        )

        storeUsername(
            "leg sister movie kind fatigue middle plunge hammer estate make dawn joke",
            "jamesholden",
            "01a46269645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b3819016762616c616e636500687265766973696f6e006a7075626c69634b65797382a662696400646461746158210346b27d7ac56af3dad59730b8bfca8335678332db0fc9d672877660d70771ddb664747970650067707572706f73650068726561644f6e6c79f46d73656375726974794c6576656c00a6626964016464617461582102b8de78395e1c518fcd048923eb191697ce783c0a1676e347f7b254d6ab653e4164747970650067707572706f73650068726561644f6e6c79f46d73656375726974794c6576656c02",
            "01ac632469645820770638d52a01b730ece5c098c6fcd346995a03dbf20a062284f23b72aa8a0b5d65247479706566646f6d61696e656c6162656c6b6a616d6573686f6c64656e677265636f726473a17464617368556e697175654964656e7469747949645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b38190168246f776e657249645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b38190169247265766973696f6e006c7072656f7264657253616c745820b5c8688f32570839b5e14b61413b4358b8ef113765ce68a94c9f72d4e8812f8b6e737562646f6d61696e52756c6573a16f616c6c6f77537562646f6d61696e73f46f2464617461436f6e747261637449645820e668c659af66aee1e72c186dde7b5b7e0a1d712a09c40d5721f622bf53c531556f6e6f726d616c697a65644c6162656c6b6a616d6573686f6c64656e702470726f746f636f6c56657273696f6e01781a6e6f726d616c697a6564506172656e74446f6d61696e4e616d656464617368",
            "01aa63246964582091803cd8f7324a5bbc5fe5e12c752be960d0c517b6c08bfbf78364390ae08d2a6524747970656770726f66696c6568246f776e657249645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b38190169247265766973696f6e006a246372656174656441741b00000189f4c664646a247570646174656441741b00000189f4c664646b646973706c61794e616d656c4a616d657320486f6c64656e6d7075626c69634d6573736167656b507572202620436c65616e6f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01",
            listOf("01ad63246964582086d59757bd4cec90f41fbedb35db2eddc97e111e9ee18238e4d2bff3769fb5216524747970656e636f6e746163745265717565737468246f776e657249645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b38190168746f557365724964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a0581369247265766973696f6e006a246372656174656441741b00000189f5066b9e6e73656e6465724b6579496e646578016f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01706163636f756e745265666572656e63651a04bc3c6b71726563697069656e744b6579496e6465780072656e637279707465645075626c69634b6579586033b87daf4068128e87b3f67182e5fb2ae05798bf3de61dd2bef33d160a0235bae7c6c03880b1f2137959954572ecf99b0d841d25b034b8825375c2c0e4b6a4f52fafacc56aaedf99caa4c1ea11ec4db63b1926a0fd7a1495c70347827d3cdb2c75656e637279707465644163636f756e744c6162656c5830d44b39df6cd3978c5d8b7f7f264e7716e3b287d2770d21269b017323dfa340e269a63c0329ba47da7899693b5e3b0c86")
        )
        storeUsername(
            "solve cover ugly sister make divorce grab lawsuit update mimic pass circle",
            "eric",
            "01a462696458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d906762616c616e636500687265766973696f6e006a7075626c69634b65797382a762696400646461746158210336c820f3b7b6e96f72c32e82febb164650f2a43149a4e10fc821cb998872431e64747970650067707572706f73650068726561644f6e6c79f4697369676e6174757265584120bc670f5c56db34d7ca400dd2cb5a4e10722dbb34d3bb704d3039d0dd93615da76a147bc97c7bd9fe58bab85e39895aecaff833621d5f63ad9f04c4b7a208a2936d73656375726974794c6576656c00a7626964016464617461582102c2a8a7c105772f3da49a17af7009f9c14e5b154dede0002da8c69a6968bd6b4e64747970650067707572706f73650068726561644f6e6c79f4697369676e617475726558412013bb21e4be02a637a2cc32c4fbe4ba7a831f200fe182c24f8ae2a6dde999a5ad393c98295f51f5157ace9373e4bf26e3644e370dc7fd3fa162dab3a0dfab88b96d73656375726974794c6576656c02",
            "01ac63246964582035b0d289743f9f957f68e17d8138c81ef01752e41d76326d3a518e486d09dc5065247479706566646f6d61696e656c6162656c6465726963677265636f726473a17464617368556e697175654964656e74697479496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d9068246f776e6572496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d9069247265766973696f6e006c7072656f7264657253616c7458202950ab7e158594df182d20c4d25546b8f653614ab74dc108322f463066b49e336e737562646f6d61696e52756c6573a16f616c6c6f77537562646f6d61696e73f46f2464617461436f6e747261637449645820e668c659af66aee1e72c186dde7b5b7e0a1d712a09c40d5721f622bf53c531556f6e6f726d616c697a65644c6162656c6465726963702470726f746f636f6c56657273696f6e01781a6e6f726d616c697a6564506172656e74446f6d61696e4e616d656464617368",
            "01a96324696458209f12e9243cb5cdbb676dff0fe57347bafc8a4c54d2acfa769152adabf91159586524747970656770726f66696c6568246f776e6572496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d9069247265766973696f6e006a246372656174656441741b00000189f999a77f6a247570646174656441741b00000189f999a77f6b646973706c61794e616d6564457269636f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01",
            listOf(
                "01ad632469645820586e096d8d4805bd37df3dc253d9a47b8d99a718871164771aa868e719a9fba36524747970656e636f6e746163745265717565737468246f776e6572496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d9068746f5573657249645820941895d08cf0f281b4952796cf0b6d86b5450f8720c4b4ba7d79a6560b38190169247265766973696f6e006a246372656174656441741b00000189f9992c496e73656e6465724b6579496e646578016f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01706163636f756e745265666572656e63651a09569a2971726563697069656e744b6579496e6465780072656e637279707465645075626c69634b65795860fd3fe7289fe395961d0783a7a1236d6b57053875250dfe2d225908c9c377d0b81f09b764d4bad0479024eacaae1180410300499244fc8ebc85e25ba43447fd0d4976bc8dae41ec1f6e6d6f2229c090c0d2bdccc57c8881a0d87787bd6f9c8e9a75656e637279707465644163636f756e744c6162656c583036238ac8983ecf006c5800259c25c4ff07b9062703fdabf05a0eb845199287536526b72c60ac452d386e2876ea7fb734",
                "01ad632469645820e3fb9387b514ecf48085ab7ff1c83f868d53b566f5b876fe55eabb9550e7a77d6524747970656e636f6e746163745265717565737468246f776e6572496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d9068746f557365724964582057f67ce35b6a9a4eea7fab9c93856d157547cb1599738a43332a3750d8a0581369247265766973696f6e006a246372656174656441741b00000189fca7db936e73656e6465724b6579496e646578016f2464617461436f6e747261637449645820a2a1b4ac6fef22ea2a1a68e8123644b357875f6b412c18109281c146e7b271bc702470726f746f636f6c56657273696f6e01706163636f756e745265666572656e63651a00c3a5d571726563697069656e744b6579496e6465780072656e637279707465645075626c69634b65795860c39e99475b093a92a372885df984495fba8eeb79dd0fe40900406609711e5c68c63381115990ab1095cb13569dd871b42534dddfd859ea4e449c7dc82d113eb6c0e9b71f57fc2f53b9e9d87de84c907392faefe827a8278d2425e768ab3502ba75656e637279707465644163636f756e744c6162656c5830bf04a16542f6c1b468002d4a74a3f7195fc6a4aef3bc552598daa5d0238e0f40b2abed62b89978777351bd04589e3aa7"
            ),
            listOf(
                "01aa6324696458202d1051c98ac606d6b007c535aeee77a957af2e46a4ca48faa3449e4b3bb98bec6524747970656b74785f6d6574616461746168246f776e6572496458204fe9f3236505829de94de9e72840966d6d866a06832c894bf533bb9533ae2d90686b6579496e6465780169247265766973696f6e006a246372656174656441741b00000189f9a47f4f6f2464617461436f6e747261637449645820d97524b363666f1667d003765cef445312b543088bfe3e64959dd51d5df0bfb4702470726f746f636f6c56657273696f6e0171656e637279707465644d657461646174615880ef22fb2b1aa04d7a04050889517260540ecc7dac3ec4105fa5fc5f70b0f79f622ccb2c1b0743cb2b9483415d75cc8473752535166220dc1c918539cd96aaea64a57718ff5ebd941d2dce2fcd256a9b2311543a64dec3d02318165a47f6fbbf6163d12c42fc8da7f7127a20d58d1d6d6ae84cba857c443c78e5fe458a3ba0903172656e6372797074696f6e4b6579496e64657800"
            )
        )
        dump()
    }

    private fun storeUsername(
        mnemonic: String,
        username: String,
        identity: String,
        domain: String,
        profile: String,
        contactRequests: List<String> = listOf(),
        txMetadata: List<String> = listOf()
    ) {
        storeIdentity(Converters.fromHex(identity))
        storeDocument(Converters.fromHex(domain))
        storeDocument(Converters.fromHex(profile))
        contactRequests.forEach {
            storeDocument(Converters.fromHex(it))
        }
        txMetadata.forEach {
            storeDocument(Converters.fromHex(it))
        }
    }

    override fun broadcastStateTransition(signedStateTransition: StateTransitionIdentitySigned) {
        when (signedStateTransition) {
            is IdentityCreateTransition -> {
                val identity = Identity(
                    signedStateTransition.identityId,
                    signedStateTransition.publicKeys,
                    0,
                    0,
                    ProtocolVersion.latestVersion
                )
                addValidIdentity(identity.id)
                storeIdentity(identity)
            }

            is DocumentsBatchTransition -> {
                signedStateTransition.transitions.forEach {
                    when (it.action) {
                        DocumentTransition.Action.CREATE -> {
                            it as DocumentCreateTransition
                            val newDocument = hashMapOf<String, Any?>(
                                "protocolVersion" to ProtocolVersion.latestVersion,
                                "\$id" to it.id,
                                "\$type" to it.type,
                                "\$dataContractId" to it.dataContractId,
                                "\$ownerId" to signedStateTransition.ownerId,
                                "\$revision" to 0
                            )
                            val deepCopy = it.data.deepCopy()
                            newDocument.putAll(deepCopy)

                            it.createdAt?.let { time -> newDocument["\$createdAt"] = time }
                            it.updatedAt?.let { time -> newDocument["\$updatedAt"] = time }
                            storeDocument(Document(newDocument, it.dataContract))
                        }

                        DocumentTransition.Action.REPLACE -> {
                            it as DocumentReplaceTransition
                            storeDocument(Document(it.toObject(), it.dataContract))
                        }

                        DocumentTransition.Action.DELETE -> {
                            // do nothing for now
                        }
                    }
                }
            }

            is DataContractCreateTransition -> {
                storeDataContract(signedStateTransition.dataContract)
            }
        }
    }
}
