/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.autoversioning;

import org.dspace.content.Item;
import org.dspace.core.Context;
import org.dspace.storage.rdbms.TableRow;
import java.util.ArrayList;
import java.util.List;

/**
 *
 *
 * @author Fabio Bolognesi (fabio at atmire dot com)
 * @author Mark Diggory (markd at atmire dot com)
 * @author Ben Bosman (ben at atmire dot com)
 */
public class AutoVersionHistoryImpl implements AutoVersionHistory
{

    private int versionHistoryId;
    private List<AutoVersion>versions;

    private Context myContext;
    private TableRow myRow;


    protected AutoVersionHistoryImpl(AutoVersionHistoryDAO vhDAO){

    }


    protected AutoVersionHistoryImpl(Context c, TableRow row)
    {
        myContext = c;
        myRow = row;

        c.cache(this, row.getIntColumn(AutoVersionHistoryDAO.VERSION_HISTORY_ID));
    }

    public int getVersionHistoryId() {
        return myRow.getIntColumn(AutoVersionHistoryDAO.VERSION_HISTORY_ID);
    }

    // LIST order: descending
    public AutoVersion getPrevious(AutoVersion version) {
        int index = versions.indexOf(version);

        if( (index+1)==versions.size()) return null;

        return versions.get(index+1);
    }

    // LIST order: descending
    public AutoVersion getNext(AutoVersion version)
    {

        int index = versions.indexOf(version);

        if(index==0)
        {
            return null;
        }

        return versions.get(index-1);
    }

    public AutoVersion getVersion(Item item) {
        for(AutoVersion v : versions)
        {
            if(v.getItem()!=null && v.getItem().getID()==item.getID())
            {
                return v;
            }
        }
        return null;
    }

    public boolean hasNext(Item item)
    {
        AutoVersion version = getVersion(item);
        return hasNext(version);
    }

    public boolean hasNext(AutoVersion version)
    {
        return getNext(version)!=null;
    }

    public List<AutoVersion> getVersions()
    {
        return versions;
    }

    public void setVersions(List<AutoVersion> versions)
    {
        this.versions = versions;
    }

    public void add(AutoVersion version)
    {
        if(versions==null) versions=new ArrayList<AutoVersion>();
        versions.add(0, version);
    }

    public AutoVersion getLatestVersion()
    {
        if(versions==null || versions.size()==0)
        {
            return null;
        }

        for(AutoVersion version : versions){
            if(version.getVersionNumber() != -1)
                return version;
        }

        return null;
    }

    public AutoVersion getFirstVersion()
    {
        if(versions==null || versions.size()==0)
        {
            return null;
        }

        return versions.get(versions.size()-1);
    }


    public boolean isFirstVersion(AutoVersion version)
    {
        AutoVersion first = versions.get(versions.size()-1);
        return first.equals(version);
    }

    public boolean isLastVersion(AutoVersion version)
    {
        AutoVersion last = versions.get(0);
        return last.equals(version);
    }

    public void remove(AutoVersion version)
    {
        versions.remove(version);
    }

    public boolean isEmpty()
    {
        return versions.size()==0;
    }

    public int size()
    {
        return versions.size();
    }

    protected TableRow getMyRow()
    {
        return myRow;
    }



    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }

        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        AutoVersionHistoryImpl that = (AutoVersionHistoryImpl) o;
        return versionHistoryId == that.versionHistoryId;

    }

    @Override
    public int hashCode()
    {
        int hash=7;
        hash=79*hash+ (this.getVersionHistoryId() ^ (this.getVersionHistoryId() >>> 32));
        return hash;
    }
}
